# -*- coding: utf-8 -*-
"""
/***************************************************************************
 LaharFlowMapTools
                                 A QGIS plugin
 This plugin is for opening and processing results from LaharFlow
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-06-06
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Mark Woodhouse
        email                : mark.woodhouse@bristol.ac.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os
import os.path
import re
import shutil

import processing
from netCDF4 import Dataset as ncDataset
from qgis.core import (Qgis, QgsColorRampShader, QgsLayerTreeGroup,
                       QgsLayerTreeLayer, QgsProcessing, QgsProcessingUtils,
                       QgsProject, QgsRasterBandStats, QgsRasterLayer,
                       QgsRasterShader, QgsSingleBandColorDataRenderer,
                       QgsSingleBandPseudoColorRenderer, QgsStyle)
from qgis.gui import QgsMessageBar
from qgis.PyQt.QtCore import (QCoreApplication, QFileInfo, QSettings,
                              QTranslator)
from qgis.PyQt.QtGui import QColor, QIcon
from qgis.PyQt.QtWidgets import QAction, QDialog, QFileDialog, QSizePolicy

# Import the code for the dialog
from .laharflow_maptools_dialog import LaharFlowMapToolsDialog
# Initialize Qt resources from file resources.py
from .resources import *


def str_to_num(value):
    try:
        value=float(value)
    except ValueError:
        try:
            value=int(value)
        except ValueError:
            pass
    return value


def is_num(value):
    try:
        value = float(value)
    except ValueError:
        try:
            value = int(value)
        except ValueError:
            return False
    return True


def get_metadata(layer):

    data = layer.dataProvider()
    
    m = data.htmlMetadata()

    x = m.split('</tr>')

    md = {}

    for xx in x:
        t = re.findall('>(.+?)<\/', xx)
        if len(t)==0:
            pass
        elif len(t)==2:
            key = t[0].replace('<td class="highlight">','').replace('<td>','').replace('</td>','')
            value = t[1].replace('<td class="highlight">','').replace('<td>','').replace('</td>','')
            md[key] = str_to_num(value)
        else:
            key = t[0].replace('<td class="highlight">','').replace('<td>','').replace('</td>','')
            md[key] = {}
            for tt in t[1:]:
                tt = tt.replace('<td class="highlight">','').replace('<td>','').replace('</td>','').replace('<tr>','').replace('</tr>','')
                subs = re.split('=|:', tt)
                if len(subs)==2:
                    subkey = subs[0].replace('<td>','').replace('<ul>','').replace('<li>','').replace('</ul>','')
                    subvalue = subs[1].replace('<td>','').replace('<ul>','').replace('<li>','').replace('</ul>','')
                    md[key][subkey] = str_to_num(subvalue)

    return md


class LaharFlowMapTools(QDialog):
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        QDialog.__init__(self)
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'LaharFlowMapTools_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Lahar Flow Map Tools')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('LaharFlowMapTools', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToRasterMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/laharflow_maptools/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'LaharFlow Map Tools'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginRasterMenu(
                self.tr(u'&Lahar Flow Map Tools'),
                action)
            self.iface.removeToolBarIcon(action)



    def show_message(self, message, level):
        # self.iface.messageBar.clearWidgets() # Remove previous messages before showing a new one
        self.iface.messageBar().pushMessage(message, level, 10)


    def lastUsedDir(self):
        settings = QSettings('laharflow_maptools')
        return settings.value("lastUsedDir", str(""))


    def setLastUsedDir(self, lastDir):
        path = QFileInfo(lastDir).absolutePath()
        QSettings('laharflow_maptools').setValue("lastUsedDir", str(path))


    def select_input_file(self):
        lastdir = self.lastUsedDir()
        filename, _filter = QFileDialog.getOpenFileName(self.dlg, "Select LaharFlow netcdf file ",lastdir, '*.nc')
        self.dlg.fileSelectorText.setText(filename)
        print(f'FILENAME = {filename}')
        rootdata = ncDataset(filename, "r", format="NETCDF4")
        rootfilename = rootdata.getncattr('title')
        erosion = rootdata.getncattr('erosion choice')
        print(f'erosion = {erosion}')
        if not 'LaharFlow' in rootfilename:
            filetype = 'invalid'
            self.show_message(f'File = {filename} is not recognised as a LaharFlow output file', Qgis.Critical)
        if erosion=='off':
            self.dlg.erosionImportCheckBox.setEnabled(False)
            self.dlg.erosionImportCheckBox.setChecked(False)
            self.dlg.erosionMinInput.setDisabled(True)
            self.dlg.erosionMaxInput.setDisabled(True)
            self.dlg.erosionDisplayCheckBox.setEnabled(False)
            self.dlg.erosionDisplayCheckBox.setChecked(False)
        else:
            self.dlg.erosionImportCheckBox.setEnabled(True)
            self.dlg.erosionMinInput.setDisabled(False)
            self.dlg.erosionMaxInput.setDisabled(False)
            self.dlg.erosionDisplayCheckBox.setEnabled(True)

    
    def get_filetype(self, filename):
        rootdata = ncDataset(filename, "r", format="NETCDF4")
        rootfilename = rootdata.getncattr('title')
        rootdata.close()
        if not 'LaharFlow' in rootfilename:
            filetype = 'invalid'
            self.show_message(f'File = {filename} is not recognised as a LaharFlow output file', Qgis.Critical)
        else:
            rootfilename = rootfilename.replace('LaharFlow: ','')
            if rootfilename=='Maximums':
                filetype = 'Maximums'
                print('Selected Maximums')
                self.dlg.depthLabel.setText('Maximum depth (m)')
            else:
                filetype = 'Snapshot'
                print('Selected Snapshot')
                self.dlg.depthLabel.setText('Flow depth (m)')
        return filetype

    def get_variables(self, filename):
        rootdata = ncDataset(filename, "r", format="NETCDF4")
        variables = rootdata.variables.keys()
        rootdata.close()
        return variables

    def inputValidate(self):
        Hn_min_input = self.dlg.depthMinInput.value()
        Hn_max_input = self.dlg.depthMaxInput.value()
        if Hn_max_input<Hn_min_input:
            self.show_message('Maximum flow depth must be greater than minimum flow depth', Qgis.Critical)
            return False
        return True
    
    def emptyMinMaxInputs(self, MinInput, MaxInput):
        MinInput.setClearValueMode(2)
        MaxInput.setClearValueMode(2)
        MinInput.setClearValue(None, '')
        MaxInput.setClearValue(None, '')
        MinInput.clear()
        MaxInput.clear()

    def initializeMinMaxInputs(self, MinInput, MaxInput, minValue=0, minClear=0.01, maxValue=9999, maxClear='maximum'):
        MinInput.MinimumValue = minValue
        MaxInput.MinimumValue = minValue
        MinInput.MaximumValue = maxValue
        MaxInput.MaximumValue = maxValue
        MinInput.setClearValueMode(2)
        if minClear=='minimum':
            MinInput.setClearValue(-1, 'minimum')
        else:
            MinInput.setClearValue(minClear, str(minClear))
        MaxInput.setClearValueMode(2)
        if maxClear=='maximum':
            MaxInput.setClearValue(-1, 'maximum')
        else:
            MinInput.setClearValue(maxClear, str(maxClear))
        MinInput.clear()
        MaxInput.clear()

    def shaderAndRender(self,
                        layer: QgsRasterLayer,
                        minValue: float,
                        maxValue: float,
                        color_lst: list[tuple[int,int,int,int]]) -> QgsRasterLayer:
        
        value_range = maxValue - minValue

        # Define a ColorRamp
        fcn = QgsColorRampShader(minimumValue=minValue, maximumValue=maxValue)
        fcn.setColorRampType(QgsColorRampShader.Interpolated)

        N_colors = len(color_lst)
        #Create qgs color list
        qgs_color_lst = []
        for j, col in enumerate(color_lst):
            qgs_color_lst.append(
                QgsColorRampShader.ColorRampItem(minValue + j*value_range/(N_colors-1.0),
                                                 QColor(*col))    
            )
        qgs_color_lst.append(
            QgsColorRampShader.ColorRampItem(maxValue*1.1,
                                             QColor(*color_lst[-1]))    
        )

        fcn.setColorRampItemList(qgs_color_lst)
        fcn.setClassificationMode(QgsColorRampShader.Continuous)
        fcn.setClip(True)

        shader = QgsRasterShader()
        shader.setRasterShaderFunction(fcn)

        renderer = QgsSingleBandPseudoColorRenderer(layer.dataProvider(), 1, shader)

        layer.setRenderer(renderer)
        layer.triggerRepaint()
        layer.emitStyleChanged()

        return layer

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = LaharFlowMapToolsDialog()
            self.dlg.fileSelectorButton.clicked.connect(self.select_input_file)
            
            self.initializeMinMaxInputs(self.dlg.depthMinInput, self.dlg.depthMaxInput)
            self.initializeMinMaxInputs(self.dlg.speedMinInput, self.dlg.speedMaxInput, minClear=0)
            self.initializeMinMaxInputs(self.dlg.psiMinInput, self.dlg.psiMaxInput, minClear=0, minValue=0, maxValue=1)
            self.initializeMinMaxInputs(self.dlg.erosionMinInput, self.dlg.erosionMaxInput, minClear=0.001, minValue=0)
            self.initializeMinMaxInputs(self.dlg.depositMinInput, self.dlg.depositMaxInput, minClear=0.001, minValue=0)

            # self.dlg.depthMinInput.MinimumValue = 0
            # self.dlg.depthMaxInput.MinimumValue = 0
            # self.dlg.depthMinInput.MaximumValue = 9999
            # self.dlg.depthMaxInput.MaximumValue = 9999
            # self.dlg.depthMinInput.setClearValueMode(2)
            # self.dlg.depthMinInput.setClearValue(0.01, '0.01')
            # self.dlg.depthMaxInput.setClearValueMode(2)
            # self.dlg.depthMaxInput.setClearValue(-1, 'maximum')
            # self.dlg.depthMinInput.clear()
            # self.dlg.depthMaxInput.clear()

            # self.dlg.depthMaxInput.setText(10)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:

            source_input_file = self.dlg.fileSelectorText.text()
            self.setLastUsedDir(source_input_file)

            # Copy input file to temporary file -- this allows processing to be done while LaharFlow runs progress
            input_file = QgsProcessingUtils.generateTempFilename(source_input_file)
            shutil.copyfile(source_input_file, input_file)

            filetype = self.get_filetype(input_file)
            print(f'filetype = {filetype}')
            if filetype=='invalid':
                self.show_message(f'File {source_input_file} is not recognized as a LaharFlow netcdf file', Qgis.Critical)
                return
            
            variables = self.get_variables(input_file)
            
            data_name = os.path.splitext(os.path.basename(source_input_file))[0]
            group_name = os.path.basename(os.path.dirname(source_input_file))
          
            uri = f'NETCDF:"{input_file}"'

            # Load flow depth layer -- it will be used to extract LaharFlow settings, plotted, and used to plot other fields
            Hn_layer_name = f'flow_depth (m)'
            variable = 'flow_depth' if filetype=='Snapshot' else 'maximum_depth'
            if variable not in variables:
                self.show_message(f"Variable {variable} not found in {source_input_file}", Qgis.Critical)
                return
            
            Hn = QgsRasterLayer(f'{uri}:{variable}', Hn_layer_name)
            if not Hn.isValid():
                self.show_message(f"Failed to load {variable} from {source_input_file}", Qgis.Critical)
                return
            
            Hn_metadata = get_metadata(Hn)

            Hcrit = Hn_metadata['More information']['NC_GLOBAL#heightThreshold']

            Hn_min_input = self.dlg.depthMinInput.value()
            Hn_max_input = self.dlg.depthMaxInput.value()

            if Hn_max_input<=0:
                Hn_max_input = Hn_metadata['Band 1']['STATISTICS_MAXIMUM']

            if Hn_min_input<Hcrit:
                self.show_message(f'Minimum flow depth should be great than height threshold for LaharFlow simulation ({Hcrit})', Qgis.Info)
            
            if Hn_max_input<Hn_min_input:
                self.show_message('Maximum flow depth must be greater than minimum flow depth', Qgis.Critical)
                return
        
            # Use Plotly Teal colormap
            teal_lst = [
                (209,238,234,255),
                (168,219,217,255),
                (133,196,201,255),
                (104,171,184,255),
                ( 79,144,166,255),
                ( 59,115,143,255),
                ( 42, 86,116,255)
            ]

            Hn = self.shaderAndRender(Hn, Hn_min_input, Hn_max_input, teal_lst)

            # Get layer tree
            # layerTree = self.iface.layerTreeCanvasBridge().rootGroup()
            root = QgsProject.instance().layerTreeRoot()

            # Create containing group if it does not exist
            run_group = root.findGroup(group_name) or root.insertGroup(0, group_name)
                        
            # Create time-stamped subgroup
            time_groups_names = [t.name() for t in run_group.findGroups()]
            new_time_group = data_name
            if new_time_group in time_groups_names:
                for child in run_group.children():
                    if child.name() == new_time_group:
                        run_group.removeChildNode(child)
            time_groups_names.append(new_time_group)
            time_group_loc = sorted(time_groups_names, reverse=True).index(new_time_group)
            time_group = run_group.findGroup(data_name) or run_group.insertGroup(time_group_loc, data_name)

            # Add flow depth layer
            QgsProject.instance().addMapLayer(Hn, False).id()
            time_group.addLayer(Hn)
            Hn_node = QgsProject.instance().layerTreeRoot().findLayer(Hn.id())
            Hn_node.setItemVisibilityChecked(self.dlg.depthDisplayCheckBox.isChecked())

            # Hn_style = Hn.styleManager().currentStyle()
            Hn.triggerRepaint()
            Hn.emitStyleChanged()

            Spd_layer_name = f'flow_speed (m/s)'
            variable = 'flow_speed' if filetype=='Snapshot' else 'maximum_speed'
            import_speed = self.dlg.speedImportCheckBox.isChecked()
            if import_speed and variable not in variables:
                self.show_message(f"Variable {variable} not found in {source_input_file}", Qgis.Warning)
                import_speed = False
            
            # Is flow speed imported?
            if import_speed:
                Spd_full = QgsRasterLayer(f'{uri}:{variable}', Spd_layer_name)
                if not Spd_full.isValid():
                    self.show_message(f"Failed to load {Spd_layer_name}", Qgis.Warning)
                    return
                
                # Spd_path = os.path.join(self.tmp_path, f"{group_name}_{data_name}_Spd")
                parameters = {'INPUT_A': Hn,
                            'BAND_A': 1,
                            'INPUT_B': Spd_full,
                            'BAND_B': 1,
                            'FORMULA': f"-1 + (B+1) * logical_and(A>{Hcrit}, logical_not(isnan(B)))", # Find speed where 
                            'NODATA': '-1',
                            'TYPE': 'Float64',
                            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT}
                Spd_source = processing.run('gdal:rastercalculator', parameters)['OUTPUT']

                Spd = QgsRasterLayer(Spd_source, Spd_layer_name)

                Spd_metadata = get_metadata(Spd)

                Spd_min_input = self.dlg.speedMinInput.value()
                Spd_max_input = self.dlg.speedMaxInput.value()

                if Spd_max_input<=0:
                    Spd_max_input = Spd_metadata['Band 1']['STATISTICS_MAXIMUM']

                if Spd_max_input<Spd_min_input:
                    self.show_message('Maximum flow speed must be greater than minimum flow speed', Qgis.Warning)
                    return
                
                sunsetdata_lst = [
                    (252,222,156,255),
                    (250,164,118,255),
                    (240,116,110,255),
                    (227, 79,111,255),
                    (220, 57,119,255),
                    (185, 37,122,255),
                    (124, 29,111,255)
                ]
                
                Spd = self.shaderAndRender(Spd, Spd_min_input, Spd_max_input, sunsetdata_lst)

                QgsProject.instance().addMapLayer(Spd, False)
                time_group.addLayer(Spd)
                Spd_node = QgsProject.instance().layerTreeRoot().findLayer(Spd.id())
                Spd_node.setItemVisibilityChecked(self.dlg.speedDisplayCheckBox.isChecked())

            psi_layer_name = f'solids_fraction'
            variable = 'solids_fraction' if filetype=='Snapshot' else 'maximum_solids_fraction'
            import_psi = self.dlg.psiImportCheckBox.isChecked()
            if import_speed and variable not in variables:
                self.show_message(f"Variable {variable} not found in {source_input_file}", Qgis.Warning)
                import_psi = False
            # Is flow concentration imported?
            if import_psi:
                # Load solids fraction (psi) layer
                psi_full = QgsRasterLayer(f'{uri}:{variable}', psi_layer_name)
                if not psi_full.isValid():
                    self.show_message(f"Failed to load {psi_layer_name}", Qgis.Warning)
                    return
                
                parameters = {'INPUT_A': Hn,
                            'BAND_A': 1,
                            'INPUT_B': psi_full,
                            'BAND_B': 1,
                            'FORMULA': f"-1 + (B+1) * logical_and(A>{Hcrit}, logical_not(isnan(B)))", # Find psi where flow depth greater than critical height
                            'NODATA': '-1',
                            'TYPE': 'Float64',
                            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT}
                psi_source = processing.run('gdal:rastercalculator', parameters)['OUTPUT']

                psi = QgsRasterLayer(psi_source, psi_layer_name)

                psi_metadata = get_metadata(psi)

                psi_min_input = self.dlg.psiMinInput.value()
                psi_max_input = self.dlg.psiMaxInput.value()

                if psi_max_input<=0:
                    psi_max_input = psi_metadata['Band 1']['STATISTICS_MAXIMUM']

                if psi_max_input<psi_min_input:
                    self.show_message('Maximum solids fraction must be greater than minimum solids fraction', Qgis.Critical)
                    return
                
                blue_brown_lst = [
                    (0  ,153,191,255),
                    (37, 147,166,255),
                    (76, 141,144,255),
                    (114,134,123,255),
                    (155,127,104,255),
                    (196,119, 87,255),
                    (147, 81, 22,255)
                ]
                
                psi = self.shaderAndRender(psi, psi_min_input, psi_max_input, blue_brown_lst)

                QgsProject.instance().addMapLayer(psi, False)
                time_group.addLayer(psi)
                psi_node = QgsProject.instance().layerTreeRoot().findLayer(psi.id())
                psi_node.setItemVisibilityChecked(self.dlg.psiDisplayCheckBox.isChecked())

            # Is erosion required
            import_erosion = self.dlg.erosionImportCheckBox.isEnabled() and self.dlg.erosionImportCheckBox.isChecked()
            if filetype=='Snapshot':
                variable = 'elevation_change'
            else:
                variable = 'maximum_erosion'
            if variable not in variables:
                self.show_message(f"Variable {variable} not found in {source_input_file}", Qgis.Warning)
                import_erosion = False
            
            if import_erosion:
                                        
                Ero_min_input = self.dlg.erosionMinInput.value()
                Ero_max_input = self.dlg.erosionMaxInput.value()
                
                if filetype=='Snapshot':
                    ElevChange_layer_name = f'elevation_change (m)'
                    ElevChange = QgsRasterLayer(f'{uri}:{variable}', ElevChange_layer_name)
                
                    if not ElevChange.isValid():
                        self.show_message(f"Failed to load {ElevChange_layer_name}", Qgis.Warning)
                
                    ElevChange_metadata = get_metadata(ElevChange)
                    EroCrit = ElevChange_metadata['More information']['NC_GLOBAL#erosion critical height']
                    Ero_max = -ElevChange_metadata['Band 1']['STATISTICS_MINIMUM']
                    
                    if Ero_max_input<=0:
                        Ero_max_input = Ero_max
                    if Ero_min_input<=-EroCrit:
                        Ero_min_input=EroCrit
                        self.show_message('Minimum erosion depth must be greater than erosion critical height', Qgis.Warning)
                        return
                    if Ero_max_input<Ero_min_input:
                        self.show_message('Maximum erosion depth must be greater than minimum erosion depth', Qgis.Critical)
                        return
                    parameters = {'INPUT_A': ElevChange,
                                  'BAND_A': 1,
                                  'FORMULA': f"-1 + (1-A) * logical_not(isnan(A))", # Find erosion between bounds
                                  'NODATA': '-1',
                                  'TYPE': 'Float64',
                                  'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT}
                    erosion_source = processing.run('gdal:rastercalculator', parameters)['OUTPUT']
            
                else:
                    erosion_layer_name = f'maximum_erosion'
                    erosion_full = QgsRasterLayer(f'{uri}:{variable}', 'Eroded depth')
                    
                    if not erosion_full.isValid():
                        self.show_message(f"Failed to load {erosion_layer_name}", Qgis.Warning)
                    else:
                        self.show_message(f"Loaded {erosion_layer_name}", Qgis.Info)
                
                    erosion_metadata = get_metadata(erosion_full)
                    EroCrit = erosion_metadata['More information']['NC_GLOBAL#erosion critical height']
                    Ero_max = erosion_metadata['Band 1']['STATISTICS_MAXIMUM']
                                        
                    if Ero_max_input<=0:
                        Ero_max_input = Ero_max
                    
                    if Ero_min_input<=EroCrit:
                        Ero_min_input=EroCrit
                        # self.show_message('Minimum erosion depth must be greater than erosion critical height', Qgis.Warning)
                    
                    if Ero_max_input<Ero_min_input:
                        self.show_message('Maximum erosion depth must be greater than minimum erosion depth', Qgis.Critical)
                        return
                    parameters = {'INPUT_A': erosion_full,
                                  'BAND_A': 1,
                                  'FORMULA': f"-1 + (A+1) * logical_not(isnan(A))", # Find erosion between bounds
                                  'NODATA': '-1',
                                  'TYPE': 'Float64',
                                  'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT}
                    erosion_source = processing.run('gdal:rastercalculator', parameters)['OUTPUT']
                    
                Erosion_layer_name = f'Eroded depth'
                erosion = QgsRasterLayer(erosion_source, Erosion_layer_name)

                # Use ColorBrewer Reds colormap
                reds_lst = [
                    (255,245,240,255),
                    (254,224,210,255),
                    (252,187,161,255),
                    (251,106, 74,255),
                    (239, 59, 44,255),
                    (203, 24, 29,255),
                    (165, 15, 21,255),
                    (103,  0, 13,255)
                ]
                
                erosion = self.shaderAndRender(erosion, Ero_min_input, Ero_max_input, reds_lst)
                
                QgsProject.instance().addMapLayer(erosion, False)
                time_group.addLayer(erosion)
                erosion_node = QgsProject.instance().layerTreeRoot().findLayer(erosion.id())
                erosion_node.setItemVisibilityChecked(self.dlg.erosionDisplayCheckBox.isChecked())

            import_deposit = self.dlg.depositImportCheckBox.isEnabled() and self.dlg.depositImportCheckBox.isChecked()
            if filetype=='Snapshot':
                variable = 'elevation_change'
            else:
                variable = 'maximum_deposit'
            if variable not in variables:
                self.show_message(f"Variable {variable} not found in {source_input_file}", Qgis.Warning)
                import_deposit = False
            # Is deposit imported?
            if import_deposit:
                Dep_min_input = self.dlg.depositMinInput.value()
                Dep_max_input = self.dlg.depositMaxInput.value()

                if filetype=='Snapshot':
                    if 'ElevChange' not in locals():
                        ElevChange_layer_name = f'elevation_change (m)'
                        ElevChange = QgsRasterLayer(f'{uri}:{variable}', ElevChange_layer_name)
                
                        if not ElevChange.isValid():
                            self.show_message(f"Failed to load {ElevChange_layer_name}", Qgis.Warning)
                
                        ElevChange_metadata = get_metadata(ElevChange)
                    
                    if Dep_max_input<=0:
                        Dep_max_input = ElevChange_metadata['Band 1']['STATISTICS_MAXIMUM']

                    parameters = {'INPUT_A': ElevChange,
                                'BAND_A': 1,
                                'FORMULA': f"-1 + (A+1) * logical_and(A>0, logical_not(isnan(A)))", # Find deposition between bounds
                                'NODATA': '-1',
                                'TYPE': 'Float64',
                                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT}
                    deposition_source = processing.run('gdal:rastercalculator', parameters)['OUTPUT']
                else:
                    deposit_layer_name = f'maximum_deposit'
                    deposit_full = QgsRasterLayer(f'{uri}:{variable}', 'Deposit depth')
                    print(deposit_full)
                    
                    if not deposit_full.isValid():
                        self.show_message(f"Failed to load {deposit_layer_name}", Qgis.Warning)
                
                    deposit_metadata = get_metadata(deposit_full)
                    
                    if Dep_max_input<=0:
                        Dep_max_input = deposit_metadata['Band 1']['STATISTICS_MAXIMUM']
                    
                    parameters = {
                            'INPUT_A': deposit_full,
                            'BAND_A': 1,
                            'FORMULA': f"-1 + (A+1) * logical_not(isnan(A))", # Find deposit depth where flow depth greater than critical height
                            'NODATA': '-1',
                            'TYPE': 'Float64',
                            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT}
                    deposition_source = processing.run('gdal:rastercalculator', parameters)['OUTPUT']
                    
                Deposition_layer_name = f'Deposit depth'
                deposit = QgsRasterLayer(deposition_source, Deposition_layer_name)
                    
                # Use ColorBrewer Blues colormap
                blues_lst = [
                    (222,235,247,255),
                    (222,235,247,255),
                    (198,219,239,255),
                    (158,202,225,255),
                    (107,174,214,255),
                    (66, 146,198,255),
                    (33, 113,181,255),
                    (8,  81, 156,255),
                    (8,  48, 107,255)
                ]

                if Dep_max_input<=0:
                    self.show_message(f'Maximum deposit depth must be greater than zero.  Setting maximum deposit depth = 1', Qgis.Warning)
                    Dep_max_input = 1

                if Dep_max_input<Dep_min_input:
                    self.show_message(f'Maximum deposit depth must be greater than minimum deposit depth.  Setting minimum deposit depth = {Hcrit}', Qgis.Warning)
                    Dep_min_input = Hcrit

                deposit = self.shaderAndRender(deposit, Dep_min_input, Dep_max_input, blues_lst)
                QgsProject.instance().addMapLayer(deposit, False)
                time_group.addLayer(deposit)
                deposit_node = QgsProject.instance().layerTreeRoot().findLayer(deposit.id())
                deposit_node.setItemVisibilityChecked(self.dlg.depositDisplayCheckBox.isChecked())

            # try:
            #     os.remove(input_file)
            # except:
            #     self.show_message(f'Could not remove temporary file {input_file}', Qgis.Warning)    
            
            self.iface.messageBar().pushMessage(
                "Success", f"LaharFlow results {source_input_file} loaded",
                level=Qgis.Success, duration=3)