# -*- coding: utf-8 -*-
"""
QGIS plugin to copy labeling rule properties between vector layers.
"""# -*- coding: utf-8 -*-
"""
QGIS plugin to copy labeling rule properties between vector layers.
"""
import os
from qgis.PyQt.QtCore import Qt, QCoreApplication, pyqtSignal
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import (
    QAction, QDockWidget, QWidget, QVBoxLayout, QTabWidget, QGroupBox,
    QFormLayout, QLabel, QComboBox, QListWidget, QCheckBox, QPushButton,
    QHBoxLayout, QSpacerItem, QSizePolicy, QTextEdit, QListWidgetItem,
    QMessageBox
)
from qgis.core import (
    QgsProject, QgsVectorLayer, QgsRuleBasedLabeling, QgsPalLayerSettings,
    QgsTextFormat, Qgis, QgsMarkerSymbol, QgsTextBackgroundSettings, QgsSymbolBufferSettings, QgsReadWriteContext, QgsLabelObstacleSettings
)

PLUGIN_NAME = "Labeling Rule Properties Copier"
PLUGIN_ICON_PATH = os.path.join(os.path.dirname(__file__), "icon.png")
RULE_DATA_ROLE = Qt.UserRole + 1

def find_rule_by_key(root_rule, key):
    """Recursively find a rule by its key."""
    if not root_rule: return None
    if root_rule.ruleKey() == key: return root_rule
    for child in root_rule.children():
        found = find_rule_by_key(child, key)
        if found: return found
    return None

def get_nested_rules_for_display(rule, index, prefix="", rules_list=None, exclude_key=None):
    """Recursively collect rules for display, excluding a specific key if provided."""
    if rules_list is None: rules_list = []
    if not rule: return rules_list

    display_text = ""
    process_children = True
    description = rule.description() or f'Rule {index + 1} (unnamed)'
    if rule.ruleKey() != exclude_key and rule.parent(): # Skip root and excluded
        display_text = f"{prefix}{description}"
        rules_list.append((display_text, rule))

    if process_children:
        if rule.parent() and rule.parent().ruleKey() != exclude_key:
            new_prefix = f"{prefix}{display_text} > " if display_text else prefix
        else:
            new_prefix = f"{prefix}{display_text.split('>')[-1].strip()} > " if display_text else prefix
        for i, child_rule in enumerate(rule.children()):
            get_nested_rules_for_display(child_rule, i, new_prefix, rules_list, exclude_key)
    return rules_list

class LabelingRulePropertiesCopierPlugin:
    """Main QGIS Plugin class."""
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = PLUGIN_NAME
        self.toolbar = self.iface.pluginToolBar()
        self.toolbar.setObjectName(PLUGIN_NAME)
        self.dock_widget = None

    def tr(self, message):
        return QCoreApplication.translate("LabelRuleCopier", message)

    def add_action(self, icon_path, text, callback, parent=None, status_tip=None, add_to_toolbar=True):
        """Helper to add QAction to menu and toolbar."""
        for existing_action in self.toolbar.actions():
            if existing_action.text() == text:
                self.toolbar.removeAction(existing_action)
        action = QAction(QIcon(icon_path), text, parent)
        action.triggered.connect(callback)
        if status_tip: action.setStatusTip(status_tip)
        if add_to_toolbar: self.toolbar.addAction(action)
        self.iface.addPluginToMenu(self.menu, action)
        self.actions.append(action)
        return action

    def initGui(self):
        """Initialize GUI elements."""
        self.add_action(PLUGIN_ICON_PATH, self.tr(PLUGIN_NAME), self.run, self.iface.mainWindow())

    def onClosePlugin(self):
        """Handle plugin dock widget closing."""
        if self.dock_widget:
            self.dock_widget.closingPlugin.disconnect(self.onClosePlugin)
            self.dock_widget = None

    def unload(self):
        """Clean up when plugin is unloaded."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr(PLUGIN_NAME), action)
            self.iface.removeToolBarIcon(action)
        del self.toolbar
        if self.dock_widget:
            self.dock_widget.close()
            self.dock_widget = None

    def run(self):
        """Show the plugin dock widget."""
        if not self.dock_widget:
            self.dock_widget = LabelingRulePropertiesCopierDockWidget(self.iface, self.iface.mainWindow())
            self.dock_widget.closingPlugin.connect(self.onClosePlugin)
            # self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dock_widget)

            self.dock_widget.setFloating(True)
            self.dock_widget.show()
            self.dock_widget.raise_()
            main_window = self.iface.mainWindow()
            dw_size = self.dock_widget.size()
            mw_size = main_window.size()
            mw_pos = main_window.pos()
            center_x = mw_pos.x() + (mw_size.width() - dw_size.width()) / 2
            center_y = mw_pos.y() + (mw_size.height() - dw_size.height()) / 2
            self.dock_widget.move(int(center_x), int(center_y))

class LabelingRulePropertiesCopierDockWidget(QDockWidget):
    """Dockable widget for the Label Rule Copier."""
    closingPlugin = pyqtSignal()

    def __init__(self, iface, parent=None):
        super().__init__(parent=parent)
        self.iface = iface
        self.setObjectName(PLUGIN_NAME + "DockWidget")
        self.setWindowTitle(PLUGIN_NAME)
        self.setWindowIcon(QIcon(PLUGIN_ICON_PATH))
        self.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
        self.setMinimumSize(215, 450)
        self.resize(215, 450)

        self.main_widget = QWidget()
        self.main_widget.setMinimumSize(215, 450)
        self.main_widget.resize(215, 450)
        self.main_layout = QVBoxLayout(self.main_widget)
        self.setWidget(self.main_widget)

        self._create_tabs()
        self._create_bottom_buttons()
        self._populate_widgets()
        self._connect_signals()

        self.update_source_rules()
        self.update_target_rules()
        self.update_placement_checkbox_state()
        self.tab_properties_layout.addStretch()
        self.tab_rules_layout.addStretch()

    def _create_tabs(self):
        """Create the tab widget and its tabs."""
        self.tab_widget = QTabWidget()
        self.main_layout.addWidget(self.tab_widget)

        # Rules Tab
        self.tab_rules = QWidget()
        self.tab_rules_layout = QVBoxLayout(self.tab_rules)
        self.tab_widget.addTab(self.tab_rules, self.tr("Rules"))
        self._create_rules_tab_content()

        # Properties Tab
        self.tab_properties = QWidget()
        self.tab_properties_layout = QVBoxLayout(self.tab_properties)
        self.tab_widget.addTab(self.tab_properties, self.tr("Properties"))
        self._create_properties_tab_content()

        # Help Tab
        self.tab_help = QWidget()
        self.tab_help_layout = QVBoxLayout(self.tab_help)
        self.tab_widget.addTab(self.tab_help, self.tr("Help"))
        self._create_help_tab_content()

    def _create_rules_tab_content(self):
        """Create content for the 'Rules' tab."""
        # Update/Convert Group
        convert_grp = QGroupBox(self.tr("Convertion"))
        layout_convert = QFormLayout(convert_grp)
        self.btn_convert_single = QPushButton(self.tr("Convert Single to Rule-Based"))
        self.lbl_convert_help = QLabel(self.tr('Convert active layer\'s single labels to rule-based to use them here.'))
        self.lbl_convert_help.setWordWrap(True)
        self.lbl_convert_help.setStyleSheet("color: gray; font-style: italic;")
        layout_convert.addRow(self.lbl_convert_help)
        layout_convert.addRow(self.btn_convert_single)
        self.tab_rules_layout.addWidget(convert_grp)

        # Source Group
        self.grp_source = QGroupBox(self.tr("Source"))
        layout_source = QFormLayout(self.grp_source)
        self.cmb_source_layer = QComboBox()
        self.cmb_source_rule = QComboBox()
        self.lbl_source_status = QLabel(self.tr("Select layer"))
        self.lbl_source_status.setStyleSheet("color: gray; font-style: italic;")
        layout_source.addRow(self.tr("Layer:"), self.cmb_source_layer)
        layout_source.addRow(self.tr("Rule:"), self.cmb_source_rule)
        layout_source.addRow(self.lbl_source_status)
        self.cmb_source_rule.hide()
        self.tab_rules_layout.addWidget(self.grp_source)

        # Target Group
        self.grp_target = QGroupBox(self.tr("Target"))
        layout_target = QFormLayout(self.grp_target)
        self.cmb_target_layer = QComboBox()
        self.lst_target_rules = QListWidget()
        self.lst_target_rules.setSelectionMode(QListWidget.ExtendedSelection)
        self.lbl_target_status = QLabel(self.tr("Select layer"))
        self.lbl_target_status.setStyleSheet("color: gray; font-style: italic;")
        layout_target.addRow(self.tr("Layer:"), self.cmb_target_layer)
        layout_target.addRow(self.tr("Rules:"), self.lst_target_rules)
        layout_target.addRow(self.lbl_target_status)
        self.lst_target_rules.hide()
        self.tab_rules_layout.addWidget(self.grp_target)

    def _create_properties_tab_content(self):
        """Create content for the 'Properties' tab."""
        self.grp_props_to_copy = QGroupBox(self.tr("Properties to Copy"))
        layout_props = QVBoxLayout(self.grp_props_to_copy)
        self.chk_filter = QCheckBox(self.tr("Filter Expression"))
        self.chk_scale = QCheckBox(self.tr("Scale Range"))
        self.chk_bg_layers = QCheckBox(self.tr("Background Symbol Layers"))
        self.chk_bg_buffer = QCheckBox(self.tr("Background Symbol Buffer"))
        self.chk_text_expr = QCheckBox(self.tr("Text Expression"))
        self.chk_text_format = QCheckBox(self.tr("Text Format"))
        self.chk_priority = QCheckBox(self.tr("Priority Value"))
        self.chk_obstacle = QCheckBox(self.tr("Obstacle Settings"))
        self.chk_bg_symbol = QCheckBox(self.tr("Placement Settings"))
        for chk in [self.chk_filter, self.chk_scale, self.chk_bg_layers, self.chk_bg_buffer,
                    self.chk_text_expr, self.chk_text_format, self.chk_priority,
                    self.chk_obstacle, self.chk_bg_symbol]:
            layout_props.addWidget(chk)
        self.tab_properties_layout.addWidget(self.grp_props_to_copy)

        self.grp_symbol_layers = QGroupBox(self.tr("background Symbol Options"))
        layout_symbol = QFormLayout(self.grp_symbol_layers)
        self.cmb_symbol_source = QComboBox()
        self.cmb_symbol_source.addItems([self.tr("All"), self.tr("Top"), self.tr("Bottom")])
        self.cmb_symbol_method = QComboBox()
        self.cmb_symbol_method.addItems([self.tr("Add"), self.tr("Replace")])
        self.cmb_symbol_position = QComboBox()
        self.cmb_symbol_position.addItems([self.tr("Top"), self.tr("Bottom")])
        layout_symbol.addRow(self.tr("Source Layers:"), self.cmb_symbol_source)
        layout_symbol.addRow(self.tr("Copy Method:"), self.cmb_symbol_method)
        layout_symbol.addRow(self.tr("Target Position:"), self.cmb_symbol_position)
        self.tab_properties_layout.addWidget(self.grp_symbol_layers)
        self.grp_symbol_layers.setEnabled(False)

    def _create_help_tab_content(self):
        """Create content for the 'Help' tab."""
        self.txt_help = QTextEdit()
        self.txt_help.setReadOnly(True)
        icon_html = f'<h2 align="center"><br><img src="{PLUGIN_ICON_PATH}" width="60"><br>{PLUGIN_NAME}</h2>'
        help_content = f"""
        {icon_html}
        <h3 align="center">Overview</h3>
        <p align="justify">Copies selected properties from a source labeling rule to target rule(s).</p>
        <h3 align="center">Limitations</h3>
        <p align="justify">
        Both layers must use Rule-Based Labeling. Placement Settings require same geometry type.
        </p>
        """
        self.txt_help.setHtml(help_content)
        self.txt_help.setStyleSheet("border: none;")
        self.tab_help_layout.addWidget(self.txt_help)

    def _create_bottom_buttons(self):
        """Create the Run and Close buttons."""
        self.btn_layout = QHBoxLayout()
        self.btn_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        self.btn_run = QPushButton(self.tr("Run"))
        self.btn_close = QPushButton(self.tr("Close"))
        self.btn_layout.addWidget(self.btn_run)
        self.btn_layout.addWidget(self.btn_close)
        self.main_layout.addLayout(self.btn_layout)

    def _populate_widgets(self):
        """Populate layer comboboxes."""
        self.cmb_source_layer.blockSignals(True)
        self.cmb_target_layer.blockSignals(True)
        self.cmb_source_layer.clear()
        self.cmb_target_layer.clear()

        layers = [l for l in QgsProject.instance().mapLayers().values()
                  if isinstance(l, QgsVectorLayer) and l.geometryType() != 4]
        if not layers:
            no_layers_msg = self.tr("No vector layers")
            self.cmb_source_layer.addItem(no_layers_msg)
            self.cmb_target_layer.addItem(no_layers_msg)
            self.cmb_source_layer.setEnabled(False)
            self.cmb_target_layer.setEnabled(False)
        else:
            self.cmb_source_layer.setEnabled(True)
            self.cmb_target_layer.setEnabled(True)
            sorted_layers = sorted(layers, key=lambda x: x.name())
            active_layer = self.iface.activeLayer()
            if active_layer and active_layer in sorted_layers:
                sorted_layers.remove(active_layer)
                sorted_layers.insert(0, active_layer)
            for layer in sorted_layers:
                self.cmb_source_layer.addItem(layer.name(), layer.id())
                self.cmb_target_layer.addItem(layer.name(), layer.id())
        self.cmb_source_layer.blockSignals(False)
        self.cmb_target_layer.blockSignals(False)

    def _connect_signals(self):
        """Connect widget signals to slots."""
        # Connect layers content.
        # pj = QgsProject.instance()
        # for lyr in pj.mapLayers().values():
        #     lyr.nameChanged.connect(self._populate_widgets)
        #     lyr.styleChanged.connect(self._populate_widgets)
        # pj.layerWasAdded.connect(self._populate_widgets)
        # pj.layerRemoved.connect(self._populate_widgets)
        # pj.layerWillBeRemoved.connect(self._populate_widgets)
        # Connect rules content.
        self.cmb_source_layer.currentIndexChanged.connect(self.update_source_rules)
        self.cmb_target_layer.currentIndexChanged.connect(self.update_target_rules)
        self.cmb_source_layer.currentIndexChanged.connect(self.update_placement_checkbox_state)
        self.cmb_target_layer.currentIndexChanged.connect(self.update_placement_checkbox_state)
        self.cmb_source_rule.currentIndexChanged.connect(self.update_target_rules) # Re-filter target
        self.chk_bg_layers.stateChanged.connect(lambda state: self.grp_symbol_layers.setEnabled(state == Qt.Checked))
        self.btn_run.clicked.connect(self.run_copy_process)
        self.btn_close.clicked.connect(self.close)
        self.btn_convert_single.clicked.connect(self.convert_single_to_rule_based)

    def _get_layer_from_combo(self, combo):
        """Get QgsVectorLayer from a QComboBox."""
        layer_id = combo.currentData()
        return QgsProject.instance().mapLayer(layer_id) if layer_id else None

    def update_source_rules(self):
        """Populate source rule combobox."""
        self.cmb_source_rule.blockSignals(True)
        current_layer_id = self.cmb_source_layer.currentData()
        self.cmb_source_rule.clear()
        layer = self._get_layer_from_combo(self.cmb_source_layer)
        status_label = self.lbl_source_status
        rule_combo = self.cmb_source_rule

        if layer and layer.labeling() and layer.labelsEnabled() and isinstance(layer.labeling(), QgsRuleBasedLabeling):
            root_rule = layer.labeling().rootRule()
            rules = get_nested_rules_for_display(root_rule, 0)
            if rules:
                rule_combo.setEnabled(True)
                status_label.hide()
                rule_combo.show()
                for text, rule_obj in rules:
                    rule_combo.addItem(text, rule_obj.ruleKey())
            else:
                rule_combo.setEnabled(False)
                status_label.setText(self.tr("No rules"))
                status_label.show()
                rule_combo.hide()
        else:
            rule_combo.setEnabled(False)
            status_label.setText(self.tr("No rules") if layer else self.tr("Select layer"))
            status_label.show()
            rule_combo.hide()
        if current_layer_id:
            for index in range(self.cmb_source_rule.count()):
                if self.cmb_source_rule.itemData(index) ==  current_layer_id:
                    self.cmb_source_rule.setCurrentIndex(index)
                    break
        self.cmb_source_rule.blockSignals(False)
        self.update_target_rules() # Source change might affect target list

    def update_target_rules(self):
        """Populate target rule list widget."""
        self.lst_target_rules.blockSignals(True)
        current_layer_id = self.cmb_target_layer.currentData()
        selected_target_rules = [item.data(RULE_DATA_ROLE) for item in self.lst_target_rules.selectedItems()]
        self.lst_target_rules.clear()
        target_layer = self._get_layer_from_combo(self.cmb_target_layer)
        source_layer = self._get_layer_from_combo(self.cmb_source_layer)
        source_rule_key = self.cmb_source_rule.currentData()
        status_label = self.lbl_target_status
        rule_list = self.lst_target_rules

        exclude_key = source_rule_key if source_layer and target_layer and source_layer.id() == target_layer.id() else None

        if target_layer and target_layer.labelsEnabled() and target_layer.labeling() and isinstance(target_layer.labeling(), QgsRuleBasedLabeling):
            root_rule = target_layer.labeling().rootRule()
            rules = get_nested_rules_for_display(root_rule, 0, exclude_key=exclude_key)
            if rules:
                rule_list.setEnabled(True)
                status_label.hide()
                rule_list.show()
                for text, rule_obj in rules:
                    item = QListWidgetItem(text)
                    item.setData(RULE_DATA_ROLE, rule_obj.ruleKey())
                    rule_list.addItem(item)
            else:
                rule_list.setEnabled(False)
                status_label.setText(self.tr("No rules"))
                status_label.show()
                rule_list.hide()
        else:
            rule_list.setEnabled(False)
            status_label.setText(self.tr("No rules") if target_layer else self.tr("Select layer"))
            status_label.show()
            rule_list.hide()
        if current_layer_id:
            for index in range(self.cmb_source_rule.count()):
                if self.cmb_source_rule.itemData(index) ==  current_layer_id:
                    self.cmb_source_rule.setCurrentIndex(index)
                    break
        for item in self.lst_target_rules.selectedItems():
            if item.data(RULE_DATA_ROLE) in selected_target_rules:
                item.setSelected(True)
        self.lst_target_rules.blockSignals(False)

    def update_placement_checkbox_state(self):
        """Enable/disable placement checkbox based on geometry types."""
        source_layer = self._get_layer_from_combo(self.cmb_source_layer)
        target_layer = self._get_layer_from_combo(self.cmb_target_layer)
        can_copy = bool(source_layer and target_layer and source_layer.geometryType() == target_layer.geometryType())
        self.chk_bg_symbol.setEnabled(can_copy)
        if not can_copy:
            self.chk_bg_symbol.setChecked(False)
            self.chk_bg_symbol.setToolTip(self.tr("Source and Target layers must have same geometry type."))
        else:
            self.chk_bg_symbol.setToolTip("")

    def convert_single_to_rule_based(self):
        """Converts single labeling to rule-based for the active layer if applicable."""
        # Iterate over all layers to find those with simple labeling
        converted_count = 0
        project_layers = QgsProject.instance().mapLayers().values()
        for layer in project_layers:
            if (isinstance(layer, QgsVectorLayer) and
                    layer.labeling() and
                    layer.labeling().type() == 'simple' and # Check for simple labeling
                    layer.labeling().settings().drawLabels):

                simple_settings = layer.labeling().settings()
                # Create a new root rule and append the simple settings as its first child
                root_rule = QgsRuleBasedLabeling.Rule(QgsPalLayerSettings()) # Empty root
                target_rule = QgsRuleBasedLabeling.Rule(simple_settings)
                # target_rule.setDescription(self.tr("Imported from simple labeling")) # Optional description
                root_rule.appendChild(target_rule)

                new_labeling = QgsRuleBasedLabeling(root_rule)
                layer.setLabeling(new_labeling)
                layer.triggerRepaint()
                self.iface.layerTreeView().refreshLayerSymbology(layer.id())
                converted_count += 1

        if converted_count > 0:
            self.iface.messageBar().pushMessage(
                PLUGIN_NAME,
                self.tr(f"Converted {converted_count} layer(s) from single to rule-based labeling."),
                level=Qgis.Success, duration=5
            )
            self._populate_widgets() # Refresh layer lists
            self.update_source_rules()
            self.update_target_rules()
            self.refresh_styling_panel(self.iface.activeLayer())
        else:
            self.iface.messageBar().pushMessage(
                PLUGIN_NAME,
                self.tr("No layers found with active simple labeling to convert."),
                level=Qgis.Info, duration=5
            )

    def refresh_styling_panel(self, layer):
        """Refresh style panel."""
        # Get the currently active layer
        layer.emitStyleChanged()
        layer.triggerRepaint()
        self.iface.mapCanvas().refresh()
        self.iface.layerTreeView().refreshLayerSymbology(layer.id())

    def run_copy_process(self):
        """Main logic to copy properties."""
        source_layer = self._get_layer_from_combo(self.cmb_source_layer)
        target_layer = self._get_layer_from_combo(self.cmb_target_layer)
        source_rule_key = self.cmb_source_rule.currentData()
        target_rule_keys = [item.data(RULE_DATA_ROLE) for item in self.lst_target_rules.selectedItems()]
        
        if not all([source_layer, target_layer, source_rule_key, target_rule_keys]):
            QMessageBox.warning(self, "Input Error", "Please select source/target layers and rules.")
            return

        source_labeling = source_layer.labeling()
        target_labeling = target_layer.labeling()
        if not (isinstance(source_labeling, QgsRuleBasedLabeling) and isinstance(target_labeling, QgsRuleBasedLabeling)):
            QMessageBox.critical(self, "Error", "Layers must use Rule-Based Labeling.")
            return

        source_rule = find_rule_by_key(source_labeling.rootRule(), source_rule_key)
        if not source_rule:
            QMessageBox.critical(self, "Error", f"Source rule (Key: {source_rule_key}) not found.")
            return

        # --- Get Properties to Copy ---
        props_to_copy = {
            'filter': self.chk_filter.isChecked(),
            'scale': self.chk_scale.isChecked(),
            'bg_layers': self.chk_bg_layers.isChecked(),
            'bg_buffer': self.chk_bg_buffer.isChecked(),
            'text_expr': self.chk_text_expr.isChecked(),
            'text_format': self.chk_text_format.isChecked(),
            'priority': self.chk_priority.isChecked(),
            'obstacle': self.chk_obstacle.isChecked(),
            'placement': self.chk_bg_symbol.isChecked() and self.chk_bg_symbol.isEnabled()
        }
        if not any(props_to_copy.values()):
            self.iface.messageBar().pushMessage(PLUGIN_NAME, "No properties selected to copy.", level=Qgis.Info, duration=3)
            return


        # --- Symbol Layer Settings (if applicable) ---
        symbol_opts = None
        if props_to_copy['bg_layers'] and self.grp_symbol_layers.isEnabled():
            symbol_opts = {
                'source': self.cmb_symbol_source.currentText(),
                'method': self.cmb_symbol_method.currentText(),
                'position': self.cmb_symbol_position.currentText()
            }

        success_count, error_count = 0, 0
        target_root_rule = target_labeling.rootRule()

        for target_key in target_rule_keys:
            target_rule = find_rule_by_key(target_root_rule, target_key)
            if not target_rule:
                error_count += 1
                continue
            
            # Set rule base values.
            if props_to_copy['filter']:
                target_rule.setFilterExpression(source_rule.filterExpression())
            if props_to_copy['scale']:
                target_rule.setMinimumScale(source_rule.minimumScale())
                target_rule.setMaximumScale(source_rule.maximumScale())        
            if props_to_copy['placement']:
                target_rule.settings().placement = source_rule.settings().placement
                target_rule.settings().setPlacementSettings(source_rule.settings().placementSettings())
            if props_to_copy['text_format']:
                new_format = QgsTextFormat(source_rule.settings().format())
                if not new_format.isValid():
                    new_format.setValid()
                orig_bg_symbol = target_rule.settings().format().background().markerSymbol()
                if orig_bg_symbol:
                    target_bg_symbol = orig_bg_symbol.clone()
                    new_format.background().setMarkerSymbol(target_bg_symbol)
                target_rule.settings().setFormat(new_format)
            if props_to_copy['bg_buffer'] and source_rule.settings().format().background().enabled() and source_rule.settings().format().background().markerSymbol():
                    bg_buffer = source_rule.settings().format().background().markerSymbol().bufferSettings()
                    buffer_input = QgsSymbolBufferSettings(bg_buffer) if bg_buffer else None
                    target_rule.settings().format().background().markerSymbol().setBufferSettings(buffer_input)
            if props_to_copy['obstacle']:
                target_rule.settings().setObstacleSettings(QgsLabelObstacleSettings(source_rule.settings().obstacleSettings()))
            if props_to_copy['priority']:
                target_rule.settings().priority = source_rule.settings().priority
            if props_to_copy['text_expr']:
                target_rule.settings().fieldName = source_rule.settings().fieldName
                target_rule.settings().isExpression = source_rule.settings().isExpression
            if props_to_copy['bg_layers']:
                source_symbol = source_rule.settings().format().background().markerSymbol()
                target_symbol = target_rule.settings().format().background().markerSymbol()
                if target_symbol and source_symbol:
                    self._apply_symbol_layer_logic(source_symbol, target_symbol, symbol_opts)

            success_count += 1 # Update success

        if success_count > 0:
            # self._populate_widgets()
            # self.update_target_rules()
            # self.update_source_rules()
            target_layer.triggerRepaint()
            self.refresh_styling_panel(target_layer)
            self.iface.layerTreeView().refreshLayerSymbology(target_layer.id())
            self.iface.messageBar().pushMessage(PLUGIN_NAME,
                                                self.tr(f"Copied to {success_count} rule(s). Errors: {error_count}."),
                                                level=Qgis.Success if error_count == 0 else Qgis.Warning, duration=5)
        elif error_count > 0 :
                self.iface.messageBar().pushMessage(PLUGIN_NAME, self.tr(f"Failed for {error_count} rule(s)."), level=Qgis.Critical, duration=5)
        # else: No changes made or no rules selected already handled or implied

    def _apply_symbol_layer_logic(self, source_symbol, target_symbol, opts):
        """Helper to manage background symbol layer copying based on combobox options."""
        if not opts or not source_symbol: return # No options or no source symbol to copy from

        target_layers = target_symbol.symbolLayers()
        source_layers = source_symbol.symbolLayers()
        layers_to_copy = []
        if opts['source'] == self.tr("All"):
            layers_to_copy = [sl.clone() for sl in source_layers]
        elif opts['source'] == self.tr("Top") and source_layers:
            layers_to_copy = [source_layers[-1].clone()]
        elif opts['source'] == self.tr("Bottom") and source_layers:
            layers_to_copy = [source_layers[0].clone()]

        if not layers_to_copy: return # Nothing to copy

        # Add/Insert logic
        if opts['position'] == self.tr("Top"):
            for sl_clone in layers_to_copy:
                target_symbol.insertSymbolLayer(len(target_symbol.symbolLayers()), sl_clone) # Append
        else: # Bottom
            for sl_clone in reversed(layers_to_copy): # Insert at beginning, maintaining order
                target_symbol.insertSymbolLayer(0, sl_clone)

        # Replace logic
        if opts['method'] == self.tr("Replace"):
            if opts['source'] == self.tr("All"):
                layers_to_delete = target_layers
            elif opts['source'] == self.tr("Top"):
                layers_to_delete = [target_layers[-1]]
            else:
                layers_to_delete = [target_layers[0]]
            layers_index = [target_symbol.symbolLayers().index(layer) for layer in layers_to_delete]
            for index in sorted(layers_index, reverse=True): # Sort reverse to avoid false index numbers
                target_symbol.deleteSymbolLayer(index)

    def closeEvent(self, event):
        """Close Event"""
        self.closingPlugin.emit()
        event.accept()

# For QGIS console testing:
if __name__ == "__console__":
     lrc_plugin = LabelingRulePropertiesCopierPlugin(iface)
     lrc_plugin.initGui()
     