# Copyright 2025 Bunting Labs, Inc.
import json
import hashlib

import sip
from qgis.PyQt.QtCore import QEventLoop, QTimer, QUrl, QFile, QIODevice, Qt
from qgis.PyQt.QtNetwork import QNetworkRequest, QNetworkReply
from qgis.core import QgsTask, QgsNetworkAccessManager, QgsProject
from qgis.PyQt.QtCore import QByteArray
from qgis.PyQt.QtCore import QThread
from qgis.PyQt.QtCore import pyqtSignal

# Metadata for Kue cloud files
KC_LAYER_ID_ROLE = Qt.UserRole + 1
KC_LAYER_HASH_ROLE = Qt.UserRole + 2
KC_LAYER_ICON_ROLE = Qt.UserRole + 3  # LAYER_TYPE_ICONS
KC_LAYER_EXTENSION_ROLE = Qt.UserRole + 4


class KueFileTask(QgsTask):
    errorReceived = pyqtSignal(str)

    """
    :param extension: File extension (e.g. 'tif').
    :param item: A QListWidgetItem or similar representing the layer in a UI list.
    :param parent: The parent dialog or widget containing the active context list.
    :param filename: Path to the local file to be uploaded, if needed.
    """

    def __init__(self, extension, item, parent, filename, icon):
        super().__init__("Checking if layer is stored on Kue server...")
        self.layer_hash = None
        self.extension = extension
        self.item = item
        self.parent = parent
        self.filename = filename
        self.icon = icon

        self.response = None
        self.file_is_indexed = False

    def run(self):
        # Get layer hash directly from filename
        try:
            with open(self.filename, "rb") as f:
                sha256_hash = hashlib.sha256()
                for byte_block in iter(lambda: f.read(32768), b""):
                    if self.isCanceled():
                        return False
                    sha256_hash.update(byte_block)
                self.layer_hash = sha256_hash.hexdigest()
        except Exception as e:
            self.errorReceived.emit(str(e))
            return False

        # Make GET request to check index status
        check_url = QUrl(
            f"https://qgis-api.buntinglabs.com/tiffbox/v1/check-index/{self.layer_hash}.{self.extension}"
        )
        request = QNetworkRequest(check_url)
        request.setTransferTimeout(10000)
        nam = QgsNetworkAccessManager.instance()
        reply = nam.blockingGet(request)
        self.response = reply.content().data().decode()

        if not self.response:
            self.errorReceived.emit(
                "Could not reach Kue server in 10 seconds, please try again shortly."
            )
            return False
        try:
            data = json.loads(self.response)

            # Check if there's an error message from the server
            if "error_message" in data:
                self.errorReceived.emit(data["error_message"])
                return False

            # Check if file is already indexed
            if data.get("indexed", False):
                self.file_is_indexed = True
                self.setDescription("Layer is already indexed in Kue.")
                return True

            # Otherwise, upload the file
            post_data = data.get("presigned_put_url")
            if post_data:
                self.setDescription("Uploading file to Kue...")
                self.setProgress(0)

                # Prepare upload
                file_obj = QFile(self.filename)
                if not file_obj.open(QIODevice.ReadOnly):
                    self.errorReceived.emit(f"Cannot open file: {self.filename}")
                    return False

                upload_request = QNetworkRequest(QUrl(post_data))
                loop = QEventLoop()

                def handle_finished():
                    """Called when upload finishes."""
                    if reply_upload.error():
                        self.errorReceived.emit(reply_upload.errorString())
                        self.file_is_indexed = False
                    else:
                        self.file_is_indexed = True
                    loop.quit()

                def handle_progress(bytes_sent, total_bytes):
                    if total_bytes > 0:
                        self.setProgress(int((bytes_sent / total_bytes) * 100))

                reply_upload = nam.put(upload_request, file_obj)
                reply_upload.uploadProgress.connect(handle_progress)
                reply_upload.finished.connect(handle_finished)

                # Create a QTimer to periodically check if this task is canceled
                timer = QTimer()
                timer.timeout.connect(
                    lambda: (not sip.isdeleted(self))
                    and self.isCanceled()
                    and loop.quit()
                )
                timer.start(100)  # Check every 100ms
                # Keep the file object alive until the upload is done
                file_obj.setParent(reply_upload)

                loop.exec_()

            # Index the raster
            if self.extension != "tif":
                return True

            self.setDescription("Adding raster to Kue...")
            # Post filename to index raster endpoint
            index_url = QUrl(
                "https://qgis-api.buntinglabs.com/tiffbox/v1/index-raster/"
                + self.layer_hash
                + "."
                + self.extension
            )
            index_request = QNetworkRequest(index_url)
            index_reply = nam.post(index_request, QByteArray())

            index_loop = QEventLoop()
            index_reply.finished.connect(index_loop.quit)
            index_loop.exec_()

            if index_reply.error() != QNetworkReply.NoError:
                self.errorReceived.emit("Failed to start indexing")
                return False

            response = json.loads(bytes(index_reply.readAll()).decode())
            index_call_id = response.get("call_id")
            if not index_call_id:
                self.errorReceived.emit("No call ID received")
                return False

            # Poll progress endpoint
            while True:
                if self.isCanceled():
                    return False

                progress_url = QUrl(
                    f"https://qgis-api.buntinglabs.com/tiffbox/v1/progress/{index_call_id}"
                )
                progress_request = QNetworkRequest(progress_url)
                progress_reply = nam.get(progress_request)

                progress_loop = QEventLoop()
                progress_reply.finished.connect(progress_loop.quit)
                progress_loop.exec_()

                if progress_reply.error() == QNetworkReply.NoError:
                    response = json.loads(bytes(progress_reply.readAll()).decode())
                    progress = response.get("progress", 0)
                    self.setProgress(min(progress, 100))
                    if response.get("status") == "completed":
                        self.file_is_indexed = True
                        break
                    elif response.get("status") != "pending":
                        break
                QThread.msleep(1000)  # Wait 1 second before next poll

            return True

        except Exception as e:
            # Catch any unexpected exceptions
            import traceback

            traceback.print_exc()
            self.errorReceived.emit(str(e))
            return False

    def finished(self, success):
        """
        Called automatically by QGIS after run() completes.
        Enables the item if indexed successfully; otherwise removes it from the list.
        """
        if success and self.file_is_indexed:
            # File is indexed or upload succeeded => enable the UI item
            self.item.setFlags(self.item.flags() | Qt.ItemIsEnabled)
            self.item.setIcon(self.icon)
            self.item.setData(KC_LAYER_HASH_ROLE, self.layer_hash)
        else:
            self.parent.kue_cloud_list.takeItem(
                self.parent.kue_cloud_list.row(self.item)
            )
