# Copyright Bunting Labs, Inc. 2024

from PyQt5.QtWidgets import (
    QDockWidget,
    QStackedWidget,
    QWidget,
    QApplication,
    QPushButton,
    QHBoxLayout,
    QVBoxLayout,
    QListWidget,
    QFrame,
    QListWidgetItem,
    QLabel,
    QTextEdit,
    QCheckBox,
    QToolButton,
    QStyle,
    QAbstractItemDelegate,
    QSizePolicy,
    QListView,
)
from PyQt5.QtGui import QTextCursor, QFont, QColor, QDesktopServices, QIcon, QCursor
from PyQt5.QtCore import Qt, QSettings, QTimer, QMimeData, QUrl, QSize
from qgis.core import (
    QgsVectorLayer,
    QgsRasterLayer,
    QgsProject,
    QgsWkbTypes,
    QgsIconUtils,
    QgsMapLayer,
    QgsApplication,
)

import hashlib
from typing import Callable
from collections import defaultdict
import os
import re
from .kue_find import KueFind, VECTOR_EXTENSIONS, RASTER_EXTENSIONS
from .kue_messages import (
    KUE_FIND_FILTER_EXPLANATION,
    KUE_CLEAR_CHAT,
    KUE_DESCRIPTION,
    KUE_SUBSCRIPTION,
    KUE_LOGIN_BUTTON,
    KUE_START_BUTTON,
    KueResponseStatus,
    status_to_color,
)
from .kue_file import (
    KueFileTask,
    KC_LAYER_ID_ROLE,
    KC_LAYER_HASH_ROLE,
    KC_LAYER_ICON_ROLE,
    KC_LAYER_EXTENSION_ROLE,
)

import xml.etree.ElementTree as ET  # needed for parsing layer_tree_model_data

LAYER_TYPE_ICONS = defaultdict(
    lambda: QgsIconUtils.iconDefaultLayer(),
    {
        "point": QgsIconUtils.iconPoint(),
        "line": QgsIconUtils.iconLine(),
        "polygon": QgsIconUtils.iconPolygon(),
        "raster": QgsIconUtils.iconRaster(),
    },
)


class UnsupportedKueFileError(Exception):
    pass


class DynamicListWidget(QListWidget):
    """A QListWidget that dynamically adjusts its height based on content."""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.model().rowsInserted.connect(self.updateHeight)
        self.model().rowsRemoved.connect(self.updateHeight)
        self.setMouseTracking(True)  # Enable mouse tracking

    def sizeHint(self):
        """Override sizeHint to provide height based on content."""
        width = super().sizeHint().width()

        # Calculate height based on content
        height = 0
        if self.count() > 0:
            # Get the height of the first row
            row_height = self.sizeHintForRow(0)

            # Calculate total rows needed (considering wrapping)
            visible_width = self.viewport().width()
            if visible_width <= 0:  # If not yet visible, use the parent width
                visible_width = self.width()
                if visible_width <= 0 and self.parentWidget():
                    visible_width = self.parentWidget().width()
                    if visible_width <= 0:
                        visible_width = 300  # Fallback width

            total_width = 0
            rows_needed = 1

            # Simulate item layout to calculate rows
            for i in range(self.count()):
                item_width = self.sizeHintForColumn(0)  # Base width
                # Add item width plus spacing
                if total_width + item_width > visible_width:
                    rows_needed += 1
                    total_width = item_width
                else:
                    total_width += item_width

            # Calculate height based on rows
            height = rows_needed * row_height

            # Add some padding
            height += 2 * self.frameWidth() + 10

        # Ensure minimum height
        height = max(height, 30)

        return QSize(width, height)

    def mouseMoveEvent(self, event):
        item = self.itemAt(event.pos())
        if item and item.flags() & Qt.ItemIsEnabled:
            self.setCursor(Qt.PointingHandCursor)
            item.setIcon(QIcon(":/images/themes/default/mIconDelete.svg"))
        elif item:
            self.setCursor(Qt.WaitCursor)
            item.setIcon(QIcon(":/images/themes/default/mActionRefresh.svg"))
        else:
            self.setCursor(Qt.ArrowCursor)
            # Reset icons for all items when not hovering over any
            for i in range(self.count()):
                curr_item = self.item(i)
                icon_str = curr_item.data(KC_LAYER_ICON_ROLE)
                curr_item.setIcon(LAYER_TYPE_ICONS[icon_str])
        super().mouseMoveEvent(event)

    def updateHeight(self):
        """Update the widget height when content changes."""
        self.updateGeometry()

    def resizeEvent(self, event):
        """Handle resize events to update height when width changes."""
        super().resizeEvent(event)
        self.updateGeometry()


class KueSidebar(QDockWidget):
    def __init__(
        self,
        iface,
        messageSent: Callable,
        authenticateUser: Callable,
        kue_find: KueFind,
        ask_kue_message: str,
        lang: str,
        setChatMessageID: Callable,
        starter_messages: list[str],
        createUser: Callable,
    ):
        super().__init__("Kue AI", iface.mainWindow())

        # Properties
        self.iface = iface
        self.messageSent = messageSent
        self.authenticateUser = authenticateUser
        self.kue_find = kue_find
        self.lang = lang
        self.setChatMessageID = setChatMessageID
        self.starter_messages = starter_messages
        self.createUser = createUser
        # The parent widget is either kue or auth
        self.parent_widget = QStackedWidget()

        # Connect to map canvas extent changes
        self.iface.mapCanvas().extentsChanged.connect(self.maybeUpdateFindResults)
        # Also update when indexing is done, regardless of bbox checkbox
        self.kue_find.filesIndexed.connect(
            lambda cnt: self.maybeUpdateFindResults(only_for_bbox=False)
        )

        # Add auth widget
        self.auth_widget = QWidget()
        auth_layout = QVBoxLayout()
        auth_layout.setAlignment(Qt.AlignVCenter)

        title = QLabel("<h2>Kue</h2>")
        title.setContentsMargins(0, 0, 0, 10)

        description = QLabel(KUE_DESCRIPTION.get(lang, KUE_DESCRIPTION["en"]))
        description.setWordWrap(True)
        description.setContentsMargins(0, 0, 0, 10)
        description.setMinimumWidth(300)

        pricing = QLabel(KUE_SUBSCRIPTION.get(lang, KUE_SUBSCRIPTION["en"]))
        pricing.setWordWrap(True)
        pricing.setContentsMargins(0, 0, 0, 10)
        pricing.setMinimumWidth(300)

        get_started_button = QPushButton(
            KUE_START_BUTTON.get(lang, KUE_START_BUTTON["en"])
        )
        get_started_button.setFixedWidth(280)
        get_started_button.setStyleSheet(
            "QPushButton { background-color: #0d6efd; color: white; border: none; padding: 8px; border-radius: 4px; } QPushButton:hover { background-color: #0b5ed7; }"
        )
        get_started_button.clicked.connect(self.createUser)

        login_button = QPushButton(KUE_LOGIN_BUTTON.get(lang, KUE_LOGIN_BUTTON["en"]))
        login_button.setFixedWidth(280)
        login_button.setStyleSheet(
            "QPushButton { background-color: #6c757d; color: white; border: none; padding: 8px; border-radius: 4px; } QPushButton:hover { background-color: #5c636a; }"
        )
        login_button.clicked.connect(self.authenticateUser)

        button_layout = QVBoxLayout()
        button_layout.addWidget(get_started_button)
        button_layout.addWidget(login_button)

        auth_layout.addWidget(title)
        auth_layout.addWidget(description)
        auth_layout.addWidget(pricing)
        auth_layout.addLayout(button_layout)

        self.auth_widget.setLayout(auth_layout)

        # 1. Build the textbox and enter button widget
        self.message_bar_widget = QWidget()

        self.textbox = QTextEdit()
        self.textbox.setFixedHeight(50)
        self.textbox.setAcceptRichText(False)
        self.textbox.setPlaceholderText(ask_kue_message)
        self.textbox.textChanged.connect(
            lambda: self.onTextUpdate(self.textbox.toPlainText())
        )

        def handleKeyPress(e):
            if e.key() == Qt.Key_Return:
                self.onEnterClicked()
            else:
                QTextEdit.keyPressEvent(self.textbox, e)

        self.textbox.keyPressEvent = handleKeyPress

        self.enter_button = QPushButton("Enter")
        self.enter_button.setFixedSize(50, 20)
        self.enter_button.clicked.connect(self.onEnterClicked)

        # Chatbox and button at bottom
        self.h_layout = QHBoxLayout()
        self.h_layout.addWidget(self.textbox)
        self.h_layout.addWidget(self.enter_button)
        self.message_bar_widget.setLayout(self.h_layout)

        # 2. Build the parent for both kue and find
        self.above_mb_widget = QStackedWidget()

        # Build kue widget
        self.kue_widget = QWidget()

        self.chat_display = TextEditWithButtons()
        # self.chat_display.setReadOnly(True)
        self.chat_display.sidebar_parent = self
        self.chat_display.setFrameShape(QFrame.NoFrame)
        self.chat_display.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)

        self.kue_layout = QVBoxLayout()
        self.kue_layout.setContentsMargins(0, 0, 0, 0)
        self.kue_layout.addWidget(self.chat_display)
        self.kue_widget.setLayout(self.kue_layout)

        self.find_widget = QWidget()
        self.find_layout = QVBoxLayout()
        self.find_layout.setContentsMargins(0, 0, 0, 0)

        # Add checkbox above results
        translated_explanation = KUE_FIND_FILTER_EXPLANATION.get(
            lang, KUE_FIND_FILTER_EXPLANATION["en"]
        )
        self.map_canvas_filter = QCheckBox(translated_explanation)
        self.find_layout.addWidget(self.map_canvas_filter)

        self.find_results = FileListWidget()
        self.find_results.setWordWrap(True)
        self.find_results.setFrameShape(QFrame.NoFrame)
        self.find_results.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.find_results.setTextElideMode(Qt.ElideNone)
        self.find_results.setDragEnabled(True)
        self.find_results.setDragDropMode(QListWidget.DragOnly)

        # Handle opening a file
        delegate = KueFileResult()
        delegate.open_raster = self.openRasterFile
        delegate.open_vector = self.openVectorFile
        self.find_results.setItemDelegate(delegate)

        self.find_layout.addWidget(self.find_results)
        self.find_widget.setLayout(self.find_layout)

        self.above_mb_widget.addWidget(self.kue_widget)
        self.above_mb_widget.addWidget(self.find_widget)
        self.above_mb_widget.setCurrentIndex(0)

        # Create a layout for kue (kue chat + find)
        self.chat_layout = QVBoxLayout()

        self.chat_layout.addWidget(self.above_mb_widget, 1)

        # Files in Kue cloud
        self.kue_cloud_list = DynamicListWidget()
        self.kue_cloud_list.setFlow(
            QListWidget.LeftToRight
        )  # Make items flow horizontally
        self.kue_cloud_list.setViewMode(
            QListWidget.ListMode
        )  # Better for horizontal layout
        self.kue_cloud_list.setIconSize(QSize(16, 16))  # Set smaller icon size
        self.kue_cloud_list.setWrapping(True)  # Enable wrapping
        self.kue_cloud_list.setMovement(QListView.Static)
        self.kue_cloud_list.setVerticalScrollBarPolicy(
            Qt.ScrollBarAlwaysOff
        )  # No vertical scrollbar needed with dynamic sizing
        self.kue_cloud_list.setHorizontalScrollBarPolicy(
            Qt.ScrollBarAlwaysOff
        )  # Remove horizontal scrollbar
        self.kue_cloud_list.setFrameShape(QFrame.NoFrame)  # Remove border
        self.kue_cloud_list.setSpacing(4)  # Space between items
        self.kue_cloud_list.setSizePolicy(
            QSizePolicy.Expanding, QSizePolicy.Preferred
        )  # Allow widget to expand horizontally and adjust height as needed
        # No need for fixed height with dynamic sizing
        self.kue_cloud_list.setStyleSheet("""
            QListWidget {
                background-color: palette(window);
            }
            QListWidget::item {
                background-color: palette(base);
                border: 1px solid palette(dark);
                border-radius: 3px;
                padding: 2px;
                margin: 0;
            }
            QListWidget::item:disabled {
                background-color: palette(base);
                border: 1px dashed palette(dark);
                border-color: palette(dark);
                color: gray;
            }
            QListWidget::item:focus {
                background-color: palette(highlight);
                border-color: palette(highlight);
            }
        """)
        self.kue_cloud_list.itemClicked.connect(self.removeContextItem)

        self.chat_layout.addWidget(self.kue_cloud_list)

        self.chat_layout.addWidget(self.message_bar_widget, 0)

        # Add message bar widget to parent widget
        self.kue_widget = QWidget()
        self.kue_widget.setAcceptDrops(True)

        self.kue_widget.setLayout(self.chat_layout)
        self.parent_widget.addWidget(self.kue_widget)
        self.parent_widget.addWidget(self.auth_widget)

        title_widget = QWidget()
        title_layout = QHBoxLayout(title_widget)
        title_layout.setContentsMargins(8, 0, 8, 0)

        title_label = QLabel("Kue AI")

        self.reset_chat_btn = QPushButton(
            KUE_CLEAR_CHAT.get(lang, KUE_CLEAR_CHAT["en"])
        )
        self.reset_chat_btn.clicked.connect(self.reset)
        self.reset_chat_btn.setMinimumWidth(80)
        self.reset_chat_btn.setToolTip("Creates a new conversation")

        # Standard controls

        self.float_button = QToolButton(self)
        float_icon = self.style().standardIcon(QStyle.SP_TitleBarNormalButton)
        self.float_button.setIcon(float_icon)
        self.float_button.clicked.connect(
            lambda: self.setFloating(not self.isFloating())
        )
        self.float_button.setToolTip("Detach chat window")

        self.close_button = QToolButton(self)
        close_icon = self.style().standardIcon(QStyle.SP_TitleBarCloseButton)
        self.close_button.setIcon(close_icon)
        self.close_button.clicked.connect(self.close)
        self.close_button.setToolTip("Close chat window")

        title_layout.addWidget(title_label)
        title_layout.addStretch()
        title_layout.addWidget(self.reset_chat_btn)
        # standard
        title_layout.addWidget(self.float_button)
        title_layout.addWidget(self.close_button)

        self.setTitleBarWidget(title_widget)

        self.parent_widget.setCurrentIndex(
            0 if QSettings().value("buntinglabs-kue/auth_token") else 1
        )

        self.setWidget(self.parent_widget)

        # Set up a timer to poll the QSettings value
        self.poll_timer = QTimer(self)
        self.poll_timer.timeout.connect(self.checkAuthToken)
        self.poll_timer.start(5000)  # 5 seconds

        for msg in self.starter_messages:
            self.addMessage({"role": "assistant", "msg": msg})

        self.task_trash = []

    def checkAuthToken(self):
        # Check if the auth token is set and update the widget index accordingly
        if QSettings().value("buntinglabs-kue/auth_token"):
            self.parent_widget.setCurrentIndex(0)
        else:
            self.parent_widget.setCurrentIndex(1)

    def addAction(self, action):
        if action.get("kue_action_svg"):
            assert "message" in action

            self.resetTextCursor()
            cursor = self.chat_display.textCursor()
            while True:
                cursor.movePosition(cursor.Left, cursor.KeepAnchor)
                if cursor.selectedText() != "\u2029":
                    cursor.movePosition(cursor.Right)
                    break
                cursor.removeSelectedText()
            self.resetTextCursor()
            self.chat_display.append("")

            self.chat_display.setAlignment(Qt.AlignLeft)
            color = status_to_color(action["status"])
            self.chat_display.insertHtml(
                f"""<div style="margin: 8px;">
                <img src="{action["kue_action_svg"]}" width="16" height="16" style="vertical-align: middle"/>
                <span style="color: {color};">{action["message"]}</span>
                </div>"""
            )
            self.chat_display.append("")
            self.chat_display.verticalScrollBar().setValue(
                self.chat_display.verticalScrollBar().maximum()
            )

    def addMessage(self, msg):
        # Super simple markdown formatting
        msg["msg"] = msg["msg"].replace("\n", "<br>")
        msg["msg"] = re.sub(r"\*\*(.*?)\*\*", r"<b>\1</b>", msg["msg"])
        msg["msg"] = re.sub(r"\*(.*?)\*", r"<i>\1</i>", msg["msg"])

        # Format message based on role
        if msg["role"] == "user":
            html = f'<div style="text-align: right; margin: 8px;">{msg["msg"]}</div>'
        elif msg["role"] == "error":
            html = f'<div style="text-align: left; margin: 8px; color: red;">{msg["msg"]}</div>'
        elif msg["role"] == "geoprocessing":
            html = f"""
                <div style="margin: 8px;">
                    <img src=":/images/themes/default/processingAlgorithm.svg" width="16" height="16" style="vertical-align: middle"/>
                    <span>{msg["msg"]}</span>
                </div>
            """
        else:
            html = f'<div style="text-align: left; margin: 8px;">{msg["msg"]}</div>'

        # Append and scroll to bottom
        self.resetTextCursor()
        cursor = self.chat_display.textCursor()
        cursor.movePosition(cursor.Left, cursor.KeepAnchor)
        if cursor.selectedText() == "\u2029":
            cursor.removeSelectedText()

        self.chat_display.append("")
        self.chat_display.setAlignment(
            Qt.AlignRight if msg["role"] == "user" else Qt.AlignLeft
        )
        self.chat_display.insertHtml(html)
        self.chat_display.verticalScrollBar().setValue(
            self.chat_display.verticalScrollBar().maximum()
        )

    def addLayerToKueCloud(self, layer_id):
        # Check if layer_id is already in kue_cloud_list
        for i in range(self.kue_cloud_list.count()):
            item = self.kue_cloud_list.item(i)
            if item.data(KC_LAYER_ID_ROLE) == layer_id:
                return

        layer = QgsProject.instance().mapLayer(layer_id)
        if not layer:
            self.addMessage(
                {"role": "error", "msg": "Could not find layer ID in project"}
            )
            return

        try:
            filename = layer.source().split("|")[0]
            file_extension = filename.lower().split(".")[-1]
        except Exception:
            file_extension = None

        # Only allow locally stored raster layers or FlatGeobuf vector layers
        if layer.type() == QgsMapLayer.VectorLayer:
            if layer.dataProvider().name() != "ogr" or file_extension not in [
                "fgb",
                "gpkg",
                "geojson",
            ]:
                self.addMessage(
                    {
                        "role": "error",
                        "msg": "Only FlatGeobuf, GeoPackage, or GeoJSON vector layers can be uploaded to Kue cloud",
                    }
                )
                return

            if layer.geometryType() == QgsWkbTypes.PointGeometry:
                icon_str = "point"
            elif layer.geometryType() == QgsWkbTypes.LineGeometry:
                icon_str = "line"
            else:
                icon_str = "polygon"
        elif layer.type() == QgsMapLayer.RasterLayer:
            if layer.dataProvider().name() != "gdal" or file_extension != "tif":
                self.addMessage(
                    {
                        "role": "error",
                        "msg": "Only GDAL .tif raster layers can be uploaded to Kue cloud",
                    }
                )
                return

            icon_str = "raster"
        else:
            self.addMessage(
                {
                    "role": "error",
                    "msg": "Only vector and raster layers can be uploaded to Kue cloud",
                }
            )
            return

        item = QListWidgetItem(
            layer_id[
                : min(
                    layer_id.find("_") if layer_id.find("_") >= 0 else len(layer_id), 16
                )
            ]
        )
        item.setData(KC_LAYER_ID_ROLE, layer_id)
        item.setData(KC_LAYER_HASH_ROLE, "")
        item.setData(KC_LAYER_ICON_ROLE, icon_str)
        item.setData(KC_LAYER_EXTENSION_ROLE, file_extension)
        # Loading icon at first
        item.setIcon(QIcon(":/images/themes/default/mActionRefresh.svg"))
        item.setToolTip(layer.name() if layer else layer_id)
        item.setFlags(item.flags() & ~Qt.ItemIsEnabled)

        self.kue_cloud_list.addItem(item)

        # Update layout after adding item
        QTimer.singleShot(0, self.kue_cloud_list.updateHeight)

        task = KueFileTask(
            file_extension,
            item,
            self,
            filename,
            LAYER_TYPE_ICONS[icon_str],
        )
        task.errorReceived.connect(
            lambda msg: self.addMessage({"role": "error", "msg": msg})
        )
        QgsApplication.taskManager().addTask(task)
        self.task_trash.append(task)

    def removeContextItem(self, item):
        # Only remove if item is enabled (indexed)
        if item.flags() & Qt.ItemIsEnabled:
            idx = self.kue_cloud_list.row(item)
            if idx >= 0:
                # Remove from QListWidget
                self.kue_cloud_list.takeItem(idx)
                # Update layout after removing item
                QTimer.singleShot(0, self.kue_cloud_list.updateHeight)
        else:
            self.addMessage(
                {
                    "role": "error",
                    "msg": "You can cancel the file upload by cancelling the upload task in the task manager.",
                }
            )

    def addError(self, msg: str):
        self.insertChars(msg, QColor("red"))

    def insertChars(self, chars, start_color=None):
        self.chat_display.moveCursor(QTextCursor.End)
        if start_color:
            ccf = self.chat_display.currentCharFormat()
            ccf.setForeground(start_color)
            self.chat_display.setCurrentCharFormat(ccf)
        chars = chars.replace("\n\n", "\n")
        while chars:
            # Find first special marker (*, ** or markdown link)
            link_match = re.search(r"\[(.*?)\]\((.*?)\)", chars)
            link_pos = link_match.start() if link_match else -1
            marker_pos = min(
                (chars.find(x) for x in ["*", "**"] if x in chars), default=-1
            )

            # Handle text before any markers
            first_marker = (
                min(p for p in [marker_pos, link_pos] if p != -1)
                if marker_pos != -1 or link_pos != -1
                else -1
            )
            if first_marker == -1:
                self.chat_display.insertPlainText(chars)
                break
            elif first_marker > 0:
                self.chat_display.insertPlainText(chars[:first_marker])
                chars = chars[first_marker:]
                continue

            # Handle markdown link
            if link_match and link_pos == 0:
                text, url = link_match.groups()
                # Set link blue, underlined, then revert back to original color
                ccf = self.chat_display.currentCharFormat()
                current_foreground = ccf.foreground().color()
                ccf.setForeground(QColor("blue"))
                ccf.setAnchor(True)
                ccf.setAnchorHref(url)
                ccf.setToolTip(url)
                ccf.setFontUnderline(True)
                self.chat_display.setCurrentCharFormat(ccf)
                self.chat_display.insertPlainText(text)
                ccf = self.chat_display.currentCharFormat()
                ccf.setForeground(current_foreground)
                ccf.setAnchor(False)
                ccf.setAnchorHref("")
                ccf.setToolTip("")
                ccf.setFontUnderline(False)
                self.chat_display.setCurrentCharFormat(ccf)
                chars = chars[link_match.end() :]
            # Handle formatting markers
            elif chars.startswith("**"):
                ccf = self.chat_display.currentCharFormat()
                ccf.setFontWeight(
                    QFont.Bold if ccf.fontWeight() == QFont.Normal else QFont.Normal
                )
                self.chat_display.setCurrentCharFormat(ccf)
                chars = chars[2:]
            elif chars.startswith("*"):
                ccf = self.chat_display.currentCharFormat()
                ccf.setFontItalic(not ccf.fontItalic())
                self.chat_display.setCurrentCharFormat(ccf)
                chars = chars[1:]
        if start_color:
            end_color = self.palette().color(self.palette().Text)
            ccf.setForeground(end_color)
            self.chat_display.setCurrentCharFormat(ccf)

    def resetTextCursor(self):
        cursor = self.chat_display.textCursor()
        cursor.movePosition(cursor.End)
        self.chat_display.setTextCursor(cursor)

    def onChatButtonClicked(self, msg):
        # Handle button click
        from console import console
        from PyQt5.QtWidgets import QApplication

        self.iface.actionShowPythonDialog().trigger()
        console._console.console.toggleEditor(True)

        QApplication.clipboard().setText(msg["msg"])
        console._console.console.pasteEditor()

    def appendHtmlToBottom(self, html, break_line=True):
        # Append and scroll to bottom
        cursor = self.chat_display.textCursor()
        cursor.movePosition(cursor.End)
        self.chat_display.setTextCursor(cursor)

        if break_line:
            self.chat_display.append("")
        self.chat_display.insertHtml(html)
        self.chat_display.verticalScrollBar().setValue(
            self.chat_display.verticalScrollBar().maximum()
        )

    def onEnterClicked(self):
        if self.textbox.toPlainText().startswith("/find"):
            return
        text = self.textbox.toPlainText()
        if text.strip() == "":
            return

        self.messageSent(text, True)
        self.textbox.clear()

    def reset(self):
        self.chat_display.clear()
        self.above_mb_widget.setCurrentIndex(0)
        self.setChatMessageID(None)
        self.kue_cloud_list.clear()

        for msg in self.starter_messages:
            self.addMessage({"role": "assistant", "msg": msg})

    def openRasterFile(self, path: str):
        rlayer = QgsRasterLayer(path, os.path.basename(path))
        QgsProject.instance().addMapLayer(rlayer)

    def openVectorFile(self, path: str):
        vlayer = QgsVectorLayer(path, os.path.basename(path), "ogr")
        QgsProject.instance().addMapLayer(vlayer)

    def onTextUpdate(self, text):
        if text.startswith("/find"):
            self.above_mb_widget.setCurrentIndex(1)

            query = text[5:].strip()
            self.find_results.clear()

            # Search with checkbox state
            results = self.kue_find.search(
                query, filter_for_map_canvas=self.map_canvas_filter.isChecked()
            )
            for path, atime, file_type, geom_type, location in results:
                item = QListWidgetItem()
                item.setData(
                    Qt.UserRole,
                    {
                        "path": path.replace(os.path.expanduser("~"), "~"),
                        "atime": atime,
                        "location": location,
                    },
                )
                if file_type == "vector":
                    if geom_type == "Point":
                        item.setIcon(QgsIconUtils.iconPoint())
                    elif geom_type == "Line String":
                        item.setIcon(QgsIconUtils.iconLine())
                    else:
                        item.setIcon(QgsIconUtils.iconPolygon())
                elif file_type == "raster":
                    item.setIcon(QgsIconUtils.iconRaster())
                else:
                    item.setIcon(QgsIconUtils.iconDefaultLayer())
                self.find_results.addItem(item)
        else:
            self.above_mb_widget.setCurrentIndex(0)

    def maybeUpdateFindResults(self, only_for_bbox: bool = True):
        if only_for_bbox and not self.map_canvas_filter.isChecked():
            return
        # Only update if find widget is visible and has a filter
        if (
            self.isVisible()
            and self.above_mb_widget.currentIndex() == 1
            and self.textbox.toPlainText().startswith("/find")
        ):
            self.onTextUpdate(self.textbox.toPlainText())


class FileListWidget(QListWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        # Keep default drag settings for file results
        # (dragging out of Kue to OS or elsewhere).
        # We do not handle layer drops here, so just remain read-only for that.
        # If you wanted to accept layer drops specifically in this list,
        # you would set setAcceptDrops(True), etc.
        # But for now, we only do drag in this widget:
        #   self.setDragEnabled(True)
        #   self.setDragDropMode(QListWidget.DragOnly)
        # as set in the constructor of KueSidebar above.


class KueFileResult(QAbstractItemDelegate):
    def __init__(self, open_vector=None, open_raster=None):
        super().__init__()
        self.open_vector = open_vector
        self.open_raster = open_raster

    def editorEvent(self, event, model, option, index):
        if event.type() == event.MouseButtonDblClick:
            path = index.data(Qt.UserRole)["path"]
            path = path.replace("~", os.path.expanduser("~"))

            # Trigger appropriate open
            if path.lower().endswith(VECTOR_EXTENSIONS) and self.open_vector:
                self.open_vector(path)
            elif path.lower().endswith(RASTER_EXTENSIONS) and self.open_raster:
                self.open_raster(path)
            return True
        return False

    def paint(self, painter, option, index):
        # Draw background if selected
        if option.state & QStyle.State_Selected:
            painter.fillRect(option.rect, option.palette.highlight())

        # Draw bottom line
        painter.setPen(option.palette.dark().color())
        painter.drawLine(
            option.rect.left(),
            option.rect.bottom(),
            option.rect.right(),
            option.rect.bottom(),
        )

        # Text color depends on select state
        if option.state & QStyle.State_Selected:
            painter.setPen(option.palette.highlightedText().color())
        else:
            painter.setPen(option.palette.text().color())

        # Get the icon, draw on top of bg
        icon = index.data(Qt.DecorationRole)
        if icon:
            icon_rect = option.rect.adjusted(4, 4, -option.rect.width() + 24, -4)
            icon.paint(painter, icon_rect)

        path = index.data(Qt.UserRole)["path"]
        filename = os.path.basename(path)
        dirname = os.path.dirname(path)

        atime = index.data(Qt.UserRole)["atime"]
        location = index.data(Qt.UserRole)["location"]

        # Draw filename on first line with offset for icon
        font = painter.font()
        font.setBold(False)
        painter.setFont(font)
        text_rect = option.rect.adjusted(28, 4, -4, -int(option.rect.height() / 2))
        painter.drawText(
            text_rect, Qt.AlignLeft | Qt.AlignVCenter, f"{dirname} (opened {atime})"
        )

        # Draw dirname on second line
        font.setBold(True)
        painter.setFont(font)
        painter.drawText(
            option.rect.adjusted(28, int(option.rect.height() / 2), -4, -4),
            Qt.AlignLeft | Qt.AlignVCenter,
            filename,
        )
        # Location is lighter gray
        if option.state & QStyle.State_Selected:
            painter.setPen(option.palette.highlightedText().color().lighter())
        else:
            painter.setPen(option.palette.text().color().lighter())

        painter.drawText(
            option.rect.adjusted(28, int(option.rect.height() / 2), -4, -4),
            Qt.AlignRight | Qt.AlignVCenter,
            location,
        )

    def sizeHint(self, option, index):
        return QSize(option.rect.width(), 40)


class TextEditWithButtons(QTextEdit):
    """Our custom chat text area that also supports drag/drop from QGIS layer panel."""

    def __init__(self, parent=None):
        super().__init__(parent)
        self.setReadOnly(True)
        self.setAcceptDrops(True)
        self.sidebar_parent = None  # will be assigned after creation
        self.anchor = None

    def dragEnterEvent(self, event):
        # Force a copy action so the layer doesn't vanish from QGIS
        if event.mimeData().hasFormat("application/qgis.layertreemodeldata"):
            event.setDropAction(Qt.CopyAction)
            event.acceptProposedAction()
        else:
            event.ignore()

    def dragMoveEvent(self, event):
        if event.mimeData().hasFormat("application/qgis.layertreemodeldata"):
            event.setDropAction(Qt.CopyAction)
            event.acceptProposedAction()
        else:
            event.ignore()

    def dropEvent(self, event):
        if not self.sidebar_parent:
            event.ignore()
            return

        if event.mimeData().hasFormat("application/qgis.layertreemodeldata"):
            root = ET.fromstring(
                event.mimeData()
                .data("application/qgis.layertreemodeldata")
                .data()
                .decode()
            )
            layer_elems = root.findall(".//layer-tree-layer")
            for layer_elem in layer_elems:
                layer_id = layer_elem.get("id")
                source = layer_elem.get("source")
                if source:
                    self.sidebar_parent.addLayerToKueCloud(layer_id)
            event.ignore()
        else:
            event.ignore()

    def mousePressEvent(self, e):
        self.anchor = self.anchorAt(e.pos())
        if self.anchor:
            QApplication.setOverrideCursor(Qt.PointingHandCursor)
        super().mousePressEvent(e)

    def mouseReleaseEvent(self, e):
        if self.anchor:
            QDesktopServices.openUrl(QUrl(self.anchor))
            QApplication.setOverrideCursor(Qt.ArrowCursor)
            self.anchor = None
        super().mouseReleaseEvent(e)
