from qgis.PyQt.QtWidgets import (
    QDialog, QPushButton, QLineEdit, QComboBox,
    QFileDialog, QMessageBox, QVBoxLayout,
    QHBoxLayout, QProgressBar
)
from qgis.core import (
    QgsVectorLayer, QgsVectorFileWriter,
    QgsProject, QgsCoordinateTransformContext
)
from qgis.gui import QgsProjectionSelectionWidget
import os


class KmlKmzConverterDialog(QDialog):

    def __init__(self, iface):
        super().__init__(iface.mainWindow())
        self.iface = iface
        self.setWindowTitle("KML / KMZ Converter")
        self.resize(550, 450)          # Initial dialog size
        self.setMinimumSize(480, 400)  # Minimum size to avoid too small panel

        self.cancel_requested = False

        # ====================
        # Widgets
        # ====================
        self.btnInput = QPushButton("Select KML/KMZ Files")
        self.inputFiles = QLineEdit()
        self.inputFiles.setReadOnly(True)
        self.inputFiles.setPlaceholderText("Selected files will appear here")

        self.btnOutput = QPushButton("Select Output Folder")
        self.outputFolder = QLineEdit()
        self.outputFolder.setReadOnly(True)
        self.outputFolder.setPlaceholderText("Output folder path")

        self.formatCombo = QComboBox()
        self.formatCombo.addItems(["Shapefile", "GeoPackage", "GeoJSON"])

        self.crsWidget = QgsProjectionSelectionWidget()

        self.progressBar = QProgressBar()
        self.progressBar.setRange(0, 100)
        self.progressBar.setValue(0)

        # Buttons
        self.btnConvert = QPushButton("Convert")
        self.btnCancel = QPushButton("Cancel")
        self.btnCancel.setEnabled(False)
        self.btnClose = QPushButton("Close")

        # ====================
        # Layouts
        # ====================
        button_layout = QHBoxLayout()
        button_layout.addWidget(self.btnConvert)
        button_layout.addWidget(self.btnCancel)
        button_layout.addWidget(self.btnClose)

        main_layout = QVBoxLayout()
        main_layout.addWidget(self.btnInput)
        main_layout.addWidget(self.inputFiles)
        main_layout.addWidget(self.btnOutput)
        main_layout.addWidget(self.outputFolder)
        main_layout.addWidget(self.formatCombo)
        main_layout.addWidget(self.crsWidget)
        main_layout.addWidget(self.progressBar)
        main_layout.addLayout(button_layout)

        self.setLayout(main_layout)

        # ====================
        # Signals
        # ====================
        self.btnInput.clicked.connect(self.select_inputs)
        self.btnOutput.clicked.connect(self.select_output)
        self.btnConvert.clicked.connect(self.convert_files)
        self.btnCancel.clicked.connect(self.cancel_conversion)
        self.btnClose.clicked.connect(self.close_dialog)

    # ====================
    # Slots
    # ====================
    def select_inputs(self):
        files, _ = QFileDialog.getOpenFileNames(
            self,
            "Select KML/KMZ files",
            "",
            "KML/KMZ (*.kml *.kmz)"
        )
        if files:
            self.inputFiles.setText(";".join(files))

    def select_output(self):
        folder = QFileDialog.getExistingDirectory(
            self,
            "Select Output Folder"
        )
        if folder:
            self.outputFolder.setText(folder)

    def cancel_conversion(self):
        self.cancel_requested = True
        self.btnCancel.setEnabled(False)

    def close_dialog(self):
        self.close()

    def convert_files(self):
        input_files = [f for f in self.inputFiles.text().split(";") if f]
        output_dir = self.outputFolder.text()
        output_format = self.formatCombo.currentText()
        target_crs = self.crsWidget.crs()

        if not input_files or not output_dir:
            QMessageBox.warning(self, "Error", "Please select input files and output folder.")
            return

        self.cancel_requested = False
        self.btnConvert.setEnabled(False)
        self.btnCancel.setEnabled(True)
        self.progressBar.setValue(0)

        total_files = len(input_files)

        for idx, file_path in enumerate(input_files, start=1):

            if self.cancel_requested:
                QMessageBox.information(self, "Cancelled", "Conversion was cancelled.")
                break

            layer = QgsVectorLayer(file_path, os.path.basename(file_path), "ogr")

            if not layer.isValid():
                QMessageBox.warning(self, "Error", f"Failed to load {file_path}")
                continue

            base_name = os.path.splitext(os.path.basename(file_path))[0]

            if output_format == "Shapefile":
                out_path = os.path.join(output_dir, f"{base_name}.shp")
                driver = "ESRI Shapefile"
            elif output_format == "GeoPackage":
                out_path = os.path.join(output_dir, f"{base_name}.gpkg")
                driver = "GPKG"
            else:
                out_path = os.path.join(output_dir, f"{base_name}.geojson")
                driver = "GeoJSON"

            options = QgsVectorFileWriter.SaveVectorOptions()
            options.driverName = driver
            options.destCRS = target_crs
            options.fileEncoding = "UTF-8"

            result, error_message = QgsVectorFileWriter.writeAsVectorFormatV3(
                layer,
                out_path,
                QgsCoordinateTransformContext(),
                options
            )

            if result == QgsVectorFileWriter.NoError:
                new_layer = QgsVectorLayer(out_path, base_name, "ogr")
                QgsProject.instance().addMapLayer(new_layer)
            else:
                QMessageBox.warning(self, "Error", error_message)

            # Update progress
            progress = int((idx / total_files) * 100)
            self.progressBar.setValue(progress)
            self.progressBar.repaint()

        self.btnConvert.setEnabled(True)
        self.btnCancel.setEnabled(False)
