# -*- coding: utf-8 -*-
"""
/***************************************************************************
 KictRainPredictor
                                 A QGIS plugin
 A QGIS plugin for quantitative precipitation forecasting using deep learning models. This plugin utilizes radar data to predict rainfall patterns up to 180 minutes ahead with three different model options: standard Keras model, TFLite optimized model, and ensemble TFLite models. Developed based on the Korean Institute of Civil Engineering and Building Technology (KICT) rainfall prediction system.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-07-10
        git sha              : $Format:%H$
        copyright            : (C) 2025 by KICT, HermeSys
        email                : sukmin28@hermesys.co.kr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import importlib
import os.path
import subprocess

from qgis.PyQt.QtCore import QCoreApplication, QSettings, QTranslator
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox

# Initialize Qt resources from file resources.py
from .resources import *


class KictRainPredictor:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "KictRainPredictor_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&KICT Rain AI")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("KictRainPredictor", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        # 필요한 패키지 설치 확인 및 설치
        self.check_and_install_dependencies()

        icon_path = ":/plugins/kict_rain_ai/icon.png"
        self.add_action(
            icon_path,
            text=self.tr("kict_rain_ai"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.first_start = True

    def check_and_install_dependencies(self):
        """필요한 패키지가 설치되어 있는지 확인하고, 없으면 설치합니다."""
        required_packages = {"gdown": "5.2.0", "tensorflow": "2.20.0"}
        missing_packages = []

        for package, version in required_packages.items():
            try:
                module = importlib.import_module(package)
                # 버전 확인 (모듈에 __version__ 속성이 있는 경우)
                if hasattr(module, "__version__"):
                    installed_version = module.__version__
                    if installed_version != version:
                        print(
                            f"{package} 버전이 {installed_version}이지만, {version}이 필요합니다. 업그레이드합니다."
                        )
                        missing_packages.append(f"{package}=={version}")
                    else:
                        print(
                            f"{package} {version} "
                            + self.tr("패키지가 이미 설치되어 있습니다.")
                        )
                else:
                    print(
                        f"{package} "
                        + self.tr(
                            "패키지가 설치되어 있습니다. 버전을 확인할 수 없습니다."
                        )
                    )
            except ImportError:
                missing_packages.append(f"{package}=={required_packages[package]}")

        if missing_packages:
            msg = QMessageBox()
            msg.setIcon(QMessageBox.Information)
            msg.setText(
                self.tr("다음 패키지를 설치해야 합니다:")
                + f" {', '.join(missing_packages)}"
            )
            msg.setInformativeText(self.tr("설치를 진행하시겠습니까?"))
            msg.setWindowTitle(self.tr("패키지 설치"))
            msg.setStandardButtons(QMessageBox.Yes | QMessageBox.No)

            if msg.exec_() == QMessageBox.Yes:
                try:
                    # 버전이 이미 missing_packages 내에서 지정되어 있음
                    # 예: ['gdown==5.2.0', 'tensorflow==2.20.0']
                    subprocess.check_call(
                        [
                            "pip",
                            "install",
                            *missing_packages,
                        ]
                    )
                    QMessageBox.information(
                        None,
                        self.tr("설치 완료"),
                        self.tr(
                            "모든 필요한 패키지가 설치되었습니다. QGIS를 재시작해주세요."
                        ),
                    )
                except Exception as e:
                    QMessageBox.warning(
                        None,
                        self.tr("설치 오류"),
                        self.tr("패키지 설치 중 오류가 발생했습니다:") + f" {str(e)}",
                    )

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&KICT Rain AI"), action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        # Import the code for the dialog
        from .kict_rain_ai_dialog import KictRainPredictorDialog

        if self.first_start:
            self.first_start = False
            self.dlg = KictRainPredictorDialog()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass
            # substitute with your code.
            pass
