# -*- coding: utf-8 -*-
"""
/***************************************************************************
 kataster
                                 A QGIS plugin
 kataster
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-12-02
        git sha              : $Format:%H$
        copyright            : (C) 2025 by jancabaj
        email                : jan807931@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QThread, pyqtSignal
from qgis.PyQt.QtGui import QIcon, QDesktopServices
from qgis.PyQt.QtCore import QUrl
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QFileDialog
from qgis.core import (
    QgsVectorLayer,
    QgsProject,
    QgsMessageLog,
    Qgis,
    QgsVectorFileWriter,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsFeature,
    QgsGeometry,
    QgsPointXY
)

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .kataster_dialog import katasterDialog
# Import download functions
from .api import download_cadastre
import os.path
import sys
import json


class DownloadWorker(QThread):
    """Worker thread for downloading cadastre data"""

    # Signals
    progress = pyqtSignal(str)  # Progress messages
    error = pyqtSignal(str)  # Error messages
    finished = pyqtSignal(list)  # List of downloaded GeoJSON files

    def __init__(self, query, output_path, output_name, fetch_parcel_c, fetch_parcel_e, fetch_zoning, plugin_dir):
        super().__init__()
        self.query = query
        self.output_path = output_path
        self.output_name = output_name
        self.fetch_parcel_c = fetch_parcel_c
        self.fetch_parcel_e = fetch_parcel_e
        self.fetch_zoning = fetch_zoning
        self.plugin_dir = plugin_dir
        self._is_killed = False

    def run(self):
        """Run the download in a separate thread"""
        try:
            # Create callback that emits signals
            callback = DownloadCallbackQt(self.progress, self.error)

            # Find cadastre code
            cadastre_code, cadastre_name = download_cadastre.find_cadastre_code(self.query)

            if not cadastre_code:
                self.error.emit(f"Could not find cadastre for query: '{self.query}'")
                self.finished.emit([])
                return

            self.progress.emit(f"\n{'='*60}")
            if cadastre_name:
                self.progress.emit(f"Cadastre: {cadastre_name.title()} (code: {cadastre_code})")
            else:
                self.progress.emit(f"Cadastre code: {cadastre_code}")
            self.progress.emit(f"Output directory: {self.output_path}")

            layers_text = []
            if self.fetch_parcel_c:
                layers_text.append("Parcel C")
            if self.fetch_parcel_e:
                layers_text.append("Parcel E")
            if self.fetch_zoning:
                layers_text.append("Zoning")
            self.progress.emit(f"Layers to fetch: {', '.join(layers_text)}")
            self.progress.emit(f"{'='*60}\n")

            output_files = []

            # Fetch C register parcels if requested
            if self.fetch_parcel_c and not self._is_killed:
                parcel_c_collection = download_cadastre.fetch_parcels_by_cadastre_code(
                    cadastre_code, output_dir=self.output_path, callback=callback
                )

                if parcel_c_collection['features']:
                    self.progress.emit(f"\nSaving C register parcel data...")
                    geojson_file = os.path.join(self.output_path, f"{self.output_name}_parcel_c.geojson")
                    download_cadastre.save_to_geojson(parcel_c_collection, geojson_file)
                    output_files.append(geojson_file)
                    self.progress.emit(f"✓ Saved to {os.path.basename(geojson_file)}")
                else:
                    self.progress.emit("\n⚠ No C register parcels found (may be API issue or empty cadastre)")

            # Fetch E register parcels if requested
            if self.fetch_parcel_e and not self._is_killed:
                if self.fetch_parcel_c:
                    self.progress.emit("\n")  # Add spacing
                parcel_e_collection = download_cadastre.fetch_parcel_e_by_cadastre_code(
                    cadastre_code, output_dir=self.output_path, callback=callback
                )

                if parcel_e_collection['features']:
                    self.progress.emit(f"\nSaving E register parcel data...")
                    geojson_file = os.path.join(self.output_path, f"{self.output_name}_parcel_e.geojson")
                    download_cadastre.save_to_geojson(parcel_e_collection, geojson_file)
                    output_files.append(geojson_file)
                    self.progress.emit(f"✓ Saved to {os.path.basename(geojson_file)}")
                else:
                    self.progress.emit("\n⚠ No E register parcels found (may be API issue or empty cadastre)")

            # Fetch zoning if requested
            if self.fetch_zoning and not self._is_killed:
                if self.fetch_parcel_c or self.fetch_parcel_e:
                    self.progress.emit("\n")  # Add spacing
                zoning_collection = download_cadastre.fetch_cadastral_zoning_by_code(
                    cadastre_code, output_dir=self.output_path, callback=callback
                )

                if zoning_collection['features']:
                    self.progress.emit(f"\nSaving zoning data...")
                    geojson_file = os.path.join(self.output_path, f"{self.output_name}_zoning.geojson")
                    download_cadastre.save_to_geojson(zoning_collection, geojson_file)
                    output_files.append(geojson_file)
                    self.progress.emit(f"✓ Saved to {os.path.basename(geojson_file)}")
                else:
                    self.progress.emit("\n⚠ No zoning features found (may be API issue or empty cadastre)")

            self.progress.emit(f"\n{'='*60}")
            if output_files:
                self.progress.emit(f"Downloaded {len(output_files)} layer(s)")
                for layer_file in output_files:
                    self.progress.emit(f"  - {os.path.basename(layer_file)}")
            else:
                self.progress.emit("No data downloaded")
            self.progress.emit(f"{'='*60}")

            # Emit finished with list of files
            self.finished.emit(output_files)

        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            self.error.emit(f"Download failed: {error_details}")
            self.finished.emit([])

    def kill(self):
        """Stop the download"""
        self._is_killed = True


class DownloadCallbackQt(download_cadastre.DownloadCallback):
    """Callback that emits Qt signals"""

    def __init__(self, progress_signal, error_signal):
        super().__init__()
        self.progress_signal = progress_signal
        self.error_signal = error_signal

    def on_progress(self, message):
        """Emit progress signal"""
        self.progress_signal.emit(message)

    def on_error(self, message):
        """Emit error signal"""
        self.error_signal.emit(message)


class kataster:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'kataster_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&kataster')
        self.toolbar = None  # Custom toolbar for plugin buttons

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Worker thread for downloading data
        self.worker = None
        self.current_query = None

        # Settings key for output path
        self.settings = QSettings()
        self.settings_key = 'kataster/output_path'

        # Add plugin's proj directory to PROJ search path for bundled grid file
        proj_dir = os.path.join(self.plugin_dir, 'proj')
        if os.path.exists(proj_dir):
            current_proj_data = os.environ.get('PROJ_DATA', '')
            if proj_dir not in current_proj_data:
                if current_proj_data:
                    os.environ['PROJ_DATA'] = f"{proj_dir}{os.pathsep}{current_proj_data}"
                else:
                    os.environ['PROJ_DATA'] = proj_dir
                QgsMessageLog.logMessage(f"Using bundled grid file from: {proj_dir}", "Kataster", Qgis.Info)

    def get_default_output_path(self):
        """Get default output path (KN directory inside plugin folder)"""
        default_path = os.path.join(self.plugin_dir, "KN")

        # Create directory if it doesn't exist
        if not os.path.exists(default_path):
            try:
                os.makedirs(default_path)
                QgsMessageLog.logMessage(f"Created default output directory: {default_path}", "Kataster", Qgis.Info)
            except Exception as e:
                QgsMessageLog.logMessage(f"Failed to create default directory: {e}", "Kataster", Qgis.Warning)
                # Fallback to plugin directory
                default_path = self.plugin_dir

        return default_path

    def get_output_path(self):
        """Get configured output path from settings or return default"""
        saved_path = self.settings.value(self.settings_key, None)

        if saved_path and os.path.exists(saved_path):
            return saved_path

        # If no saved path or path doesn't exist, use default
        return self.get_default_output_path()

    def set_output_path(self, path):
        """Save output path to settings"""
        if path and os.path.isdir(path):
            self.settings.setValue(self.settings_key, path)
            QgsMessageLog.logMessage(f"Output path set to: {path}", "Kataster", Qgis.Info)
            return True
        return False

    def browse_output_path(self):
        """Open directory browser dialog"""
        current_path = self.dlg.output_path_input.text()
        if not current_path or not os.path.exists(current_path):
            current_path = self.get_output_path()

        folder = QFileDialog.getExistingDirectory(
            self.dlg,
            "Select Output Folder",
            current_path
        )

        if folder:
            self.dlg.output_path_input.setText(folder)
            self.set_output_path(folder)

    def remove_diacritics(self, text):
        """
        Remove diacritics from Slovak text for safe filenames

        Args:
            text: Text with diacritics

        Returns:
            Text without diacritics
        """
        # Slovak diacritics mapping
        diacritic_map = {
            'á': 'a', 'ä': 'a', 'č': 'c', 'ď': 'd', 'é': 'e', 'ě': 'e',
            'í': 'i', 'ĺ': 'l', 'ľ': 'l', 'ň': 'n', 'ó': 'o', 'ô': 'o',
            'ŕ': 'r', 'š': 's', 'ť': 't', 'ú': 'u', 'ů': 'u', 'ý': 'y',
            'ž': 'z',
            'Á': 'A', 'Ä': 'A', 'Č': 'C', 'Ď': 'D', 'É': 'E', 'Ě': 'E',
            'Í': 'I', 'Ĺ': 'L', 'Ľ': 'L', 'Ň': 'N', 'Ó': 'O', 'Ô': 'O',
            'Ŕ': 'R', 'Š': 'S', 'Ť': 'T', 'Ú': 'U', 'Ů': 'U', 'Ý': 'Y',
            'Ž': 'Z'
        }

        result = text
        for diacritic, replacement in diacritic_map.items():
            result = result.replace(diacritic, replacement)

        return result

    def fix_swapped_coordinates(self, layer, layer_name):
        """
        Fix parcels with swapped coordinates (X and Y reversed)

        Args:
            layer: QgsVectorLayer to check and fix
            layer_name: Name of layer (for logging)

        Returns:
            QgsVectorLayer: Memory layer with fixed geometries
        """
        # Get source CRS
        source_crs = layer.crs()
        if not source_crs.isValid():
            source_crs = QgsCoordinateReferenceSystem('EPSG:4326')
            QgsMessageLog.logMessage(f"  ⚠ Source CRS not defined, assuming EPSG:4326", "Kataster", Qgis.Warning)

        # Create memory layer with same geometry type
        geom_type = layer.geometryType()
        geom_type_str = "MultiPolygon" if geom_type == 2 else "Polygon"

        memory_layer = QgsVectorLayer(
            f"{geom_type_str}?crs={source_crs.authid()}",
            "temp",
            "memory"
        )

        # Add fields
        memory_layer.dataProvider().addAttributes(layer.fields())
        memory_layer.updateFields()

        # Process features
        fixed_count = 0
        total_count = 0

        for feature in layer.getFeatures():
            total_count += 1
            geom = feature.geometry()

            if geom.isNull():
                # Copy feature without geometry
                new_feature = QgsFeature(memory_layer.fields())
                new_feature.setAttributes(feature.attributes())
                memory_layer.dataProvider().addFeature(new_feature)
                continue

            bbox = geom.boundingBox()

            # Check for swapped coordinates (X > 40 and Y < 40 in geographic CRS)
            # Normal Slovak coordinates: X (longitude) ~17-22°, Y (latitude) ~47-49°
            if bbox.xMinimum() > 40 and bbox.yMaximum() < 40:
                # Swap X and Y coordinates
                fixed_count += 1

                if geom.isMultipart():
                    multipolygon = geom.asMultiPolygon()
                    swapped_multipolygon = []

                    for polygon in multipolygon:
                        swapped_polygon = []
                        for ring in polygon:
                            swapped_ring = []
                            for point in ring:
                                swapped_ring.append(QgsPointXY(point.y(), point.x()))
                            swapped_polygon.append(swapped_ring)
                        swapped_multipolygon.append(swapped_polygon)

                    geom = QgsGeometry.fromMultiPolygonXY(swapped_multipolygon)
                else:
                    polygon = geom.asPolygon()
                    swapped_polygon = []
                    for ring in polygon:
                        swapped_ring = []
                        for point in ring:
                            swapped_ring.append(QgsPointXY(point.y(), point.x()))
                        swapped_polygon.append(swapped_ring)
                    geom = QgsGeometry.fromPolygonXY(swapped_polygon)

            # Create new feature with fixed geometry
            new_feature = QgsFeature(memory_layer.fields())
            new_feature.setAttributes(feature.attributes())
            new_feature.setGeometry(geom)
            memory_layer.dataProvider().addFeature(new_feature)

        if fixed_count > 0:
            QgsMessageLog.logMessage(f"  ⚠ Fixed {fixed_count}/{total_count} parcels with swapped coordinates", "Kataster", Qgis.Warning)
        else:
            QgsMessageLog.logMessage(f"  ✓ All {total_count} parcels have correct coordinates", "Kataster", Qgis.Info)

        return memory_layer

    def convert_geojsons_to_gpkg(self, geojson_files, output_gpkg, cadastre_name, transform_to_5514=False):
        """
        Convert multiple GeoJSON files to a single GeoPackage with multiple layers

        Args:
            geojson_files: List of GeoJSON file paths
            output_gpkg: Output GeoPackage file path
            cadastre_name: Name of cadastre (for logging)
            transform_to_5514: If True, transform from EPSG:4258 to EPSG:5514

        Returns:
            bool: True if successful, False otherwise
        """
        try:
            # Remove existing GPKG if it exists
            if os.path.exists(output_gpkg):
                try:
                    os.remove(output_gpkg)
                    QgsMessageLog.logMessage(f"Removed existing GPKG file", "Kataster", Qgis.Info)
                except Exception as e:
                    QgsMessageLog.logMessage(f"Warning: Could not remove existing GPKG: {e}", "Kataster", Qgis.Warning)
                    return False

            # Source CRS - Slovak cadastre API serves data in EPSG:4326 (WGS84)
            # GeoJSON from the API has no explicit CRS, defaults to EPSG:4326
            source_crs = QgsCoordinateReferenceSystem('EPSG:4326')
            transform_context = QgsProject.instance().transformContext()

            # Determine target CRS based on transformation option
            if transform_to_5514:
                target_crs = QgsCoordinateReferenceSystem('EPSG:5514')

                # Set explicit coordinate operation for EPSG:4326 -> EPSG:5514 transformation
                # Using official Slovak geodesy transformation with grid shift
                # This is the official transformation from Slovak government (GKU)
                proj_string = ("+proj=pipeline "
                               "+step +proj=unitconvert +xy_in=deg +xy_out=rad "
                               "+step +proj=push +v_3 "
                               "+step +proj=cart +ellps=WGS84 "
                               "+step +inv +proj=helmert +x=485.021 +y=169.465 +z=483.839 "
                               "+rx=-7.786342 +ry=-4.397554 +rz=-4.102655 +s=0 +convention=coordinate_frame "
                               "+step +inv +proj=cart +ellps=bessel "
                               "+step +proj=pop +v_3 "
                               "+step +proj=krovak +lat_0=49.5 +lon_0=24.8333333333333 "
                               "+alpha=30.2881397527778 +k=0.9999 +x_0=0 +y_0=0 +ellps=bessel "
                               "+step +inv +proj=krovak +lat_0=49.5 +lon_0=24.8333333333333 "
                               "+alpha=30.2881397527778 +k=0.9999 +x_0=0 +y_0=0 +ellps=bessel "
                               "+step +proj=hgridshift +grids=sk_gku_JTSK03_to_JTSK.tif "
                               "+step +proj=krovak +lat_0=49.5 +lon_0=24.8333333333333 "
                               "+alpha=30.2881397527778 +k=0.9999 +x_0=0 +y_0=0 +ellps=bessel")

                # Try to set the coordinate operation explicitly
                try:
                    transform_context.addCoordinateOperation(source_crs, target_crs, proj_string)
                    QgsMessageLog.logMessage(f"CRS transformation enabled: EPSG:4326 → EPSG:5514 (using official SK GKU transformation)", "Kataster", Qgis.Info)
                except Exception as e:
                    QgsMessageLog.logMessage(f"Could not set explicit coordinate operation, using default: {e}", "Kataster", Qgis.Warning)
                    QgsMessageLog.logMessage(f"CRS transformation enabled: EPSG:4326 → EPSG:5514 (default)", "Kataster", Qgis.Info)
            else:
                target_crs = source_crs
                QgsMessageLog.logMessage(f"Keeping original CRS: EPSG:4326", "Kataster", Qgis.Info)

            # Define layer configurations
            layer_configs = []
            for geojson_file in geojson_files:
                if not os.path.exists(geojson_file):
                    continue

                filename = os.path.basename(geojson_file)

                # Determine layer name and style based on filename
                if '_parcel_c' in filename:
                    layer_name = 'ParcelC'
                    qml_file = 'kn_parcelC.qml'
                elif '_parcel_e' in filename:
                    layer_name = 'ParcelE'
                    qml_file = 'kn_parcelE.qml'
                elif '_zoning' in filename:
                    layer_name = 'CadastralUnit'
                    qml_file = 'kn_cadastralunit.qml'
                else:
                    continue

                layer_configs.append({
                    'geojson': geojson_file,
                    'layer_name': layer_name,
                    'qml': qml_file
                })

            if not layer_configs:
                QgsMessageLog.logMessage("No valid GeoJSON files to convert", "Kataster", Qgis.Warning)
                return False

            # Convert each GeoJSON to GPKG layer
            for idx, config in enumerate(layer_configs):
                QgsMessageLog.logMessage(f"Converting {config['layer_name']}...", "Kataster", Qgis.Info)

                # Load GeoJSON layer
                layer = QgsVectorLayer(config['geojson'], config['layer_name'], "ogr")

                if not layer.isValid():
                    QgsMessageLog.logMessage(f"ERROR: Could not load {config['geojson']}", "Kataster", Qgis.Critical)
                    continue

                feature_count = layer.featureCount()
                QgsMessageLog.logMessage(f"  Loaded {feature_count} features", "Kataster", Qgis.Info)

                # Fix swapped coordinates for parcel layers
                if config['layer_name'] in ['ParcelC', 'ParcelE']:
                    layer = self.fix_swapped_coordinates(layer, config['layer_name'])

                # Set up save options
                save_options = QgsVectorFileWriter.SaveVectorOptions()
                save_options.driverName = 'GPKG'
                save_options.fileEncoding = 'UTF-8'
                save_options.layerName = config['layer_name']
                save_options.layerOptions = ['SPATIAL_INDEX=YES']

                # Add coordinate transformation if requested
                if transform_to_5514:
                    save_options.ct = QgsCoordinateTransform(
                        layer.crs(),
                        target_crs,
                        transform_context
                    )

                # Determine file action (create new file or add layer to existing)
                if idx > 0 and os.path.exists(output_gpkg):
                    file_action = QgsVectorFileWriter.CreateOrOverwriteLayer
                else:
                    file_action = QgsVectorFileWriter.CreateOrOverwriteFile

                save_options.actionOnExistingFile = file_action

                # Write layer to GPKG
                error = QgsVectorFileWriter.writeAsVectorFormatV3(
                    layer,
                    output_gpkg,
                    transform_context,
                    save_options
                )

                if error[0] != QgsVectorFileWriter.NoError:
                    QgsMessageLog.logMessage(f"ERROR: Failed to write {config['layer_name']}: {error[1]}", "Kataster", Qgis.Critical)
                    return False

                QgsMessageLog.logMessage(f"  ✓ {config['layer_name']} written to GPKG", "Kataster", Qgis.Success)

                # Apply style
                self.apply_style_to_gpkg_layer(output_gpkg, config['layer_name'], config['qml'])

            return True

        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            QgsMessageLog.logMessage(f"ERROR during GPKG conversion: {error_details}", "Kataster", Qgis.Critical)
            return False

    def apply_style_to_gpkg_layer(self, gpkg_path, layer_name, qml_file):
        """Apply QML style to GPKG layer and set as default"""
        try:
            # Load layer from GPKG
            layer = QgsVectorLayer(f"{gpkg_path}|layername={layer_name}", layer_name, "ogr")
            if not layer.isValid():
                QgsMessageLog.logMessage(f"  Warning: Could not load {layer_name} for styling", "Kataster", Qgis.Warning)
                return

            # Load style
            style_path = os.path.join(self.plugin_dir, 'styles', qml_file)
            if os.path.exists(style_path):
                msg, success = layer.loadNamedStyle(style_path)
                if success:
                    # Save to database as DEFAULT style
                    error_msg = layer.saveStyleToDatabase(
                        "",  # Empty name = default style
                        "",  # Empty description
                        True,  # useAsDefault = True
                        ""  # Empty UI file path
                    )
                    if error_msg:
                        QgsMessageLog.logMessage(f"  Warning: Style save warning: {error_msg}", "Kataster", Qgis.Warning)
                    else:
                        QgsMessageLog.logMessage(f"  ✓ Style saved as default for {layer_name}", "Kataster", Qgis.Info)
                else:
                    QgsMessageLog.logMessage(f"  Warning: Could not load style: {msg}", "Kataster", Qgis.Warning)
            else:
                QgsMessageLog.logMessage(f"  Warning: Style file not found: {qml_file}", "Kataster", Qgis.Warning)
        except Exception as e:
            QgsMessageLog.logMessage(f"  Warning: Style error: {str(e)}", "Kataster", Qgis.Warning)

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('kataster', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
        use_custom_toolbar=False):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            if use_custom_toolbar:
                # Add to custom toolbar
                if self.toolbar is None:
                    self.toolbar = self.iface.addToolBar(u'Kataster SR')
                self.toolbar.addAction(action)
            else:
                # Adds plugin icon to Plugins toolbar
                self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        # Add Kataster SR button to custom toolbar
        icon_path = ':/plugins/kataster/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Kataster SR'),
            callback=self.run,
            parent=self.iface.mainWindow(),
            status_tip=self.tr(u'Load cadastral data from Slovak Cadastre'),
            use_custom_toolbar=True)  # Use custom toolbar

        # Add ZBGIS button to same custom toolbar
        zbgis_icon_path = ':/plugins/kataster/zbgis_icon.png'
        self.add_action(
            zbgis_icon_path,
            text=self.tr(u'Open ZBGIS'),
            callback=self.open_zbgis,
            parent=self.iface.mainWindow(),
            status_tip=self.tr(u'Open zbgis.skgeodesy.sk at current map extent'),
            add_to_menu=False,  # Don't add to plugin menu
            use_custom_toolbar=True)  # Use custom toolbar

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&kataster'),
                action)
            self.iface.removeToolBarIcon(action)

        # Remove custom toolbar
        if self.toolbar is not None:
            del self.toolbar
            self.toolbar = None


    def worker_progress(self, message):
        """Handle progress messages from worker"""
        QgsMessageLog.logMessage(message, "Kataster", Qgis.Info)

    def worker_error(self, message):
        """Handle error messages from worker"""
        QgsMessageLog.logMessage(message, "Kataster", Qgis.Critical)

    def worker_finished(self, geojson_files):
        """Called when download worker finishes"""
        try:
            # Check if any GeoJSON files were created
            existing_geojsons = [f for f in geojson_files if os.path.exists(f)]

            if not existing_geojsons:
                raise Exception("No GeoJSON files were created")

            self.dlg.progress_bar.setValue(70)
            self.dlg.status_label.setText("Converting to GeoPackage...")

            # Determine output GPKG path based on cadastre name/code
            output_dir = os.path.dirname(existing_geojsons[0])
            # Remove diacritics and replace spaces with underscores for safe filename
            safe_name = self.remove_diacritics(self.current_query).replace(' ', '_')
            gpkg_filename = f"{safe_name}.gpkg"
            output_gpkg = os.path.join(output_dir, gpkg_filename)

            # Check if CRS transformation is requested
            transform_to_5514 = self.dlg.transform_crs_checkbox.isChecked()

            # Convert GeoJSON files to GPKG
            QgsMessageLog.logMessage(f"\nConverting GeoJSON files to GeoPackage: {gpkg_filename}", "Kataster", Qgis.Info)
            conversion_success = self.convert_geojsons_to_gpkg(
                existing_geojsons,
                output_gpkg,
                self.current_query,
                transform_to_5514
            )

            if not conversion_success:
                raise Exception("Failed to convert GeoJSON files to GeoPackage")

            self.dlg.progress_bar.setValue(85)
            self.dlg.status_label.setText("Loading layers into QGIS...")

            # Load all layers from GPKG
            loaded_layers = []
            total_features = 0

            # Layer names in GPKG
            gpkg_layer_names = ['ParcelC', 'ParcelE', 'CadastralUnit']

            for layer_name in gpkg_layer_names:
                layer_uri = f"{output_gpkg}|layername={layer_name}"
                # Use layer name as display name to match style naming
                layer = QgsVectorLayer(layer_uri, layer_name, "ogr")

                if not layer.isValid():
                    # Layer might not exist (e.g., only ParcelC was downloaded)
                    continue

                num_features = layer.featureCount()

                if num_features == 0:
                    QgsMessageLog.logMessage(f"Warning: {layer_name} has no features", "Kataster", Qgis.Warning)
                    continue

                # Add to project
                QgsProject.instance().addMapLayer(layer)
                loaded_layers.append((layer_name, num_features))
                total_features += num_features

                QgsMessageLog.logMessage(
                    f"✓ Loaded {num_features} features for {layer_name}",
                    "Kataster",
                    Qgis.Success
                )

            if not loaded_layers:
                QMessageBox.warning(
                    self.dlg,
                    "No Data",
                    f"No data found for {self.current_query}.\n\nThis may be due to:\n• API returning invalid data (government server issue)\n• Empty cadastre\n• Try a different cadastre"
                )
                self.dlg.status_label.setText("No data found")
                return

            # Delete GeoJSON files after successful GPKG creation
            self.dlg.status_label.setText("Cleaning up temporary files...")
            for geojson_file in existing_geojsons:
                try:
                    if os.path.exists(geojson_file):
                        os.remove(geojson_file)
                        QgsMessageLog.logMessage(f"  Removed {os.path.basename(geojson_file)}", "Kataster", Qgis.Info)
                except Exception as e:
                    QgsMessageLog.logMessage(f"  Warning: Could not remove {os.path.basename(geojson_file)}: {e}", "Kataster", Qgis.Warning)

            self.dlg.progress_bar.setValue(100)

            # Build success message
            success_msg = f"Successfully loaded {len(loaded_layers)} layer(s):\n\n"
            for layer_name, num_features in loaded_layers:
                success_msg += f"• {layer_name}: {num_features} features\n"
            success_msg += f"\nData saved to: {gpkg_filename}"

            self.dlg.status_label.setText(f"Success! Loaded {total_features} features in {len(loaded_layers)} layer(s)")

            QMessageBox.information(
                self.dlg,
                "Success",
                success_msg
            )

        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            QgsMessageLog.logMessage(f"ERROR: {error_details}", "Kataster", Qgis.Critical)

            QMessageBox.critical(
                self.dlg,
                "Error",
                f"Failed to load layers:\n{str(e)}\n\nCheck View → Panels → Log Messages (Kataster tab) for details"
            )
            self.dlg.status_label.setText(f"Error: {str(e)}")

        finally:
            self.dlg.load_button.setEnabled(True)
            self.dlg.progress_bar.setValue(0)
            if self.worker:
                self.worker.quit()
                self.worker.wait()
                self.worker = None

    def load_parcels(self):
        """Load selected layers by calling standalone download script asynchronously"""
        # Get cadastre query
        query = self.dlg.cadastre_input.text().strip()

        if not query:
            QMessageBox.warning(
                self.dlg,
                "Input Required",
                "Please enter a cadastre name or code"
            )
            return

        # Get output path
        output_path = self.dlg.output_path_input.text().strip()
        if not output_path or not os.path.exists(output_path):
            QMessageBox.warning(
                self.dlg,
                "Invalid Path",
                "Please select a valid output folder"
            )
            return

        # Check which layers are selected
        fetch_parcel_c = self.dlg.parcel_c_checkbox.isChecked()
        fetch_parcel_e = self.dlg.parcel_e_checkbox.isChecked()
        fetch_zoning = self.dlg.zoning_checkbox.isChecked()

        if not fetch_parcel_c and not fetch_parcel_e and not fetch_zoning:
            QMessageBox.warning(
                self.dlg,
                "No Layers Selected",
                "Please select at least one layer to fetch"
            )
            return

        # Check if already running
        if self.worker is not None and self.worker.isRunning():
            QMessageBox.warning(
                self.dlg,
                "Already Running",
                "A download is already in progress. Please wait."
            )
            return

        # Update UI
        self.dlg.load_button.setEnabled(False)
        self.dlg.progress_bar.setValue(10)

        layers_text = []
        if fetch_parcel_c:
            layers_text.append("Parcel C")
        if fetch_parcel_e:
            layers_text.append("Parcel E")
        if fetch_zoning:
            layers_text.append("Zoning")
        self.dlg.status_label.setText(f"Downloading {', '.join(layers_text)} for {query}...")

        try:
            # Remove diacritics and replace spaces with underscores for safe filename
            safe_query = self.remove_diacritics(query).replace(' ', '_')
            output_name = f"cadastre_{safe_query}"

            self.current_query = query

            # Remove old files if they exist
            expected_files = []
            if fetch_parcel_c:
                expected_files.append(os.path.join(output_path, f"{output_name}_parcel_c.geojson"))
            if fetch_parcel_e:
                expected_files.append(os.path.join(output_path, f"{output_name}_parcel_e.geojson"))
            if fetch_zoning:
                expected_files.append(os.path.join(output_path, f"{output_name}_zoning.geojson"))

            for geojson_file in expected_files:
                if os.path.exists(geojson_file):
                    os.remove(geojson_file)

            QgsMessageLog.logMessage(f"Query: {query}", "Kataster", Qgis.Info)
            QgsMessageLog.logMessage(f"Output path: {output_path}", "Kataster", Qgis.Info)
            QgsMessageLog.logMessage(f"Layers: {layers_text}", "Kataster", Qgis.Info)

            self.dlg.progress_bar.setValue(30)
            self.dlg.status_label.setText("Starting download...")

            # Create and start worker thread
            self.worker = DownloadWorker(
                query=query,
                output_path=output_path,
                output_name=output_name,
                fetch_parcel_c=fetch_parcel_c,
                fetch_parcel_e=fetch_parcel_e,
                fetch_zoning=fetch_zoning,
                plugin_dir=self.plugin_dir
            )

            # Connect signals
            self.worker.progress.connect(self.worker_progress)
            self.worker.error.connect(self.worker_error)
            self.worker.finished.connect(self.worker_finished)

            # Start the worker
            self.worker.start()

            self.dlg.status_label.setText(f"Downloading {query} (this may take a minute)...")
            QgsMessageLog.logMessage("Download started", "Kataster", Qgis.Info)

        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            QgsMessageLog.logMessage(f"ERROR: {error_details}", "Kataster", Qgis.Critical)

            QMessageBox.critical(
                self.dlg,
                "Error",
                f"Failed to start download:\n{str(e)}"
            )
            self.dlg.status_label.setText(f"Error: {str(e)}")
            self.dlg.load_button.setEnabled(True)
            self.dlg.progress_bar.setValue(0)
            self.worker = None

    def open_zbgis(self):
        """Open zbgis.skgeodesy.sk with current map extent coordinates"""
        try:
            # Get the map canvas
            canvas = self.iface.mapCanvas()

            # Get current extent
            extent = canvas.extent()

            # Get project CRS
            project_crs = QgsProject.instance().crs()

            # Define WGS84 CRS (EPSG:4326)
            wgs84_crs = QgsCoordinateReferenceSystem("EPSG:4326")

            # Create coordinate transform
            transform = QgsCoordinateTransform(project_crs, wgs84_crs, QgsProject.instance())

            # Transform extent center to WGS84
            center = extent.center()
            center_wgs84 = transform.transform(center)

            # Get lat, lon
            lat = center_wgs84.y()
            lon = center_wgs84.x()

            # Calculate zoom level based on QGIS map scale
            # zbgis uses standard web map zoom levels where zoom correlates with scale
            import math

            # Get map scale (e.g., 1:50000 returns 50000)
            scale = canvas.scale()

            # Convert scale to zoom level
            # Web map zoom formula: scale = 591657550.5 / (2^zoom)
            # Therefore: zoom = log2(591657550.5 / scale)
            zoom = math.log2(591657550.5 / scale)

            # Adjust offset so that 1:100 scale gives zoom 21
            zoom = zoom - 1.5

            # Round to nearest integer
            zoom = round(zoom)

            # Clamp zoom to reasonable range (8-21 for zbgis, 21 is max)
            zoom = int(max(8, min(21, zoom)))

            # Build URL with pos parameter (format: lat,lon,zoom)
            # Use 6 decimal places for coordinate precision
            url = f"https://zbgis.skgeodesy.sk/mapka/sk/kataster?pos={lat:.6f},{lon:.6f},{zoom}"

            # Open in default browser
            QDesktopServices.openUrl(QUrl(url))

            QgsMessageLog.logMessage(f"QGIS Scale: 1:{scale:.0f}, Calculated zoom: {zoom}, URL: {url}", "Kataster", Qgis.Info)

            # Also show message to user
            self.iface.messageBar().pushMessage("ZBGIS", f"Opening at zoom {zoom} (QGIS scale 1:{scale:.0f})", Qgis.Info, 3)

        except Exception as e:
            QgsMessageLog.logMessage(f"Error opening ZBGIS: {str(e)}", "Kataster", Qgis.Critical)
            self.iface.messageBar().pushMessage("Error", f"Failed to open ZBGIS: {str(e)}", Qgis.Critical, 5)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = katasterDialog()
            # Connect the load button
            self.dlg.load_button.clicked.connect(self.load_parcels)
            # Connect the browse button
            self.dlg.browse_button.clicked.connect(self.browse_output_path)

        # Reset UI
        self.dlg.cadastre_input.clear()
        self.dlg.progress_bar.setValue(0)
        self.dlg.status_label.setText("Ready")

        # Set output path to saved value or default
        output_path = self.get_output_path()
        self.dlg.output_path_input.setText(output_path)

        # show the dialog
        self.dlg.show()
