/*
 *  (C) 2018, Roman Geistho:vel
 *  
 *  This program is free software; you can redistribute it and/or modify it 
 *  under the terms of the GNU General Public License as published by the 
 *  Free Software Foundation; either version 3 of the License, or    
 *  (at your option) any later version.
 *
 */

#include <stdlib.h>
#include <math.h>
#include <time.h>

#if defined(_MSC_VER)
#	if _MSC_VER <= 1500
#		error "Need at least MSVC 2010"
#	endif
    #ifndef __cplusplus
        typedef _Bool bool;
    #endif /* __cplusplus */
#endif /* _MSC_VER */

#include <assert.h>
#include <string.h>

#ifdef __cplusplus
#define EXTERNC extern "C"
#else
#define EXTERNC 
#endif

#ifdef __cplusplus
#define EXTERNC_BEGIN EXTERNC {
#else
#define EXTERNC_BEGIN
#endif

#ifdef __cplusplus
#define EXTERNC_END }
#else
#define EXTERNC_END
#endif

#if defined(_MSC_VER)
#	define DLLPUBLIC    __declspec(dllexport) 
#	define inline		_inline
#	define RESTRICT		__restrict
#elif defined(__GNUC__)
#   include <stdbool.h>
#	define DLLPUBLIC
#	define RESTRICT		__restrict
#else
#   include <stdbool.h>
#	define DLLPUBLIC
#	define RESTRICT
#endif

#define ASSERTNN(x)             (assert((x) != NULL))
#define ISNULL(x)               (NULL == (x))
#define NOTNULL(x)              (NULL != (x))

#define CAST(to_type, expr)     ((to_type)(expr))
    
#define ZEROMEM(ptr, nbytes)    (memset(ptr, 0, nbytes))

#define ZALLOC(TYPE, NUM)       CAST(TYPE *, calloc(NUM, sizeof(TYPE)))
#define FREE(PTR)               (free(PTR))

#ifdef DEBUG
    #define ARR_IDX(nrows, ncols, i, j)     ((i)*(ncols) + (j))
#else
    #define ARR_IDX(nrows, ncols, i, j)     ((i)*(ncols) + (j))
#endif


typedef double real_t;

//--------------------------------------------------------------------------//

bool
pixel_at(
    /*@out@*/ int * RESTRICT i,
    /*@out@*/ int * RESTRICT j,
    real_t x, 
    real_t y,
    int nrows, 
    int ncols) 
{
    ASSERTNN(i);
    ASSERTNN(j);

    const int m = CAST(int, y);
    const int n = CAST(int, x);

    *i = m;
    *j = n;

    return  0 <= m && m < nrows     
        &&  0 <= n && n < ncols;
}

void
free_arrays(real_t * x, real_t * y)
{
    if (NOTNULL(x)) { FREE(x); }
    if (NOTNULL(y)) { FREE(y); }
}

int 
dummy_progress(double f)
{
    return 1;
}

EXTERNC_BEGIN

/* Progress callback type */
typedef  
int 
(*callback_t)(double);

/* 
    Line integral convolution

    result                  array to hold output
    texture                 array holding texture 
    vec_x                   normalized x direction
    vec_y                   normalized y direction
    progress                progress callback for calling programm
    nrows                   number of rows
    ncols                   number of columns
    L                       integration length
*/
DLLPUBLIC int
lic(    /*@out@*/   real_t *        RESTRICT result,
                    const real_t *  RESTRICT texture,
                    const real_t *  RESTRICT vec_x,
                    const real_t *  RESTRICT vec_y,
        /*@null@*/  callback_t      progress,
                    const int       nrows, 
                    const int       ncols,
                    const int       L)
{

    ASSERTNN(result);
    ASSERTNN(texture);
    ASSERTNN(vec_x);   
    ASSERTNN(vec_y);   

    if (ISNULL(progress))
    {
        progress = dummy_progress;
    }

    const size_t nelems = CAST(size_t, nrows*ncols);

    const real_t DS = 0.5; // step size

    // Multiply vector field by step size
    real_t * nvec_x = ZALLOC(real_t, nelems);
    real_t * nvec_y = ZALLOC(real_t, nelems);
    {
        if (ISNULL(nvec_x) || ISNULL(nvec_y)) 
        {
            free_arrays(nvec_x, nvec_y);
            return 0;
        }

        const real_t * vxp = vec_x;
        const real_t * vyp = vec_y;

        real_t * nvxp = nvec_x;
        real_t * nvyp = nvec_y;

        const real_t * end = vec_x + nelems;

        while (vxp < end) 
        {
            *nvxp = DS * (*vxp);
            *nvyp = DS * (*vyp);
            ++vxp;
            ++vyp;
            ++nvxp;
            ++nvyp;
        }
    }

    ZEROMEM(result, nelems*sizeof(real_t));
    real_t * rp = result;

    const real_t * tp = texture;

    srand(CAST(unsigned, time(NULL)));

    int p=0, q=0;
    real_t x=0, y=0;

    int idx=0;

    for (int i=0; i<nrows; ++i) 
    {
        for (int j=0; j<ncols; ++j) 
        {
            
            real_t sum = *tp;
            int n = 1; // count of line steps

            // Start from origin and step forwards
            p=i, q=j;
            idx = ARR_IDX(nrows, ncols, p, q);
            x=j+0.5, y=i+0.5;
            for (int s=0; s<L; ++s) 
            {
                x += nvec_x[idx];
                y += nvec_y[idx];
                if (pixel_at(&p, &q, x, y, nrows, ncols)) 
                {
                    idx = ARR_IDX(nrows, ncols, p, q);
                    sum += texture[idx]; 
                    ++n;
                } 
                else 
                {
                    break;
                }
            }
            
            // Return to origin and step backwards
            p=i, q=j;
            idx = ARR_IDX(nrows, ncols, p, q);
            x=j+0.5, y=i+0.5;
            for (int s=0; s<L; ++s) 
            {
                x -= nvec_x[idx];
                y -= nvec_y[idx];
                if (pixel_at(&p, &q, x, y, nrows, ncols)) 
                {
                    idx = ARR_IDX(nrows, ncols, p, q);
                    sum += texture[idx]; 
                    ++n;
                } 
                else 
                {
                    break;
                }
            }
            
            // Normalize and assign
            *rp = sum / n;

            ++rp;
            ++tp;
        }

        if (0 == progress(CAST(double, i+1) / nrows))
        {
            free_arrays(nvec_x, nvec_y);
            return 0;
        }
    }
    
    free_arrays(nvec_x, nvec_y);
    return 1;
}

EXTERNC_END
