"""
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__    = 'Roman Geisthövel'
__date__      = '2018-03-05'
__copyright__ = '(C) 2018 by Roman Geisthövel'

from glob import glob
import numpy as np
import karika.appinter as appi

__MOD = None

def getmod():
    global __MOD
    if __MOD is None:
        C = appi.Common
        F = appi.FFI
        files = glob(C.mkpath(C.folder(), "lib", "ext_lic*"))
        for ext in files:
            __MOD = F.load(ext)
            if __MOD is not None:
                break
    return __MOD
    
def lic(grid, cellsize, int_steps, **opt):

    if getmod() is None:
        raise NotImplementedError

    # Generalization by Line Integral Convolution of the grid values
    # along the gradient of the grid, as described in the PhD thesis
    # Geisthoevel (2017) "Automatic Swiss style rock depiction"
    # https://doi.org/10.3929/ethz-b-000201368

    assert cellsize > 0,  "Cellsize must be greater than zero."
    assert int_steps > 0, "Integration length must be greater 0."
    F = appi.FFI

    Real = np.dtype("f8")

    # Normalized gradient
    dy, dx = np.gradient(grid, cellsize)
    out = np.nan_to_num(np.hypot(dx,dy))
    out[out==0] = 1
    dx /= out
    dy /= out

    H,W  = grid.shape

    ameta = dict(dtype=Real, ndim=1, shape=H*W, flags=["C"])
    arr_t = np.ctypeslib.ndpointer(**ameta)
    
    del ameta["flags"]
    arrw_t = np.ctypeslib.ndpointer(flags=("C", "W"), **ameta)

    Int = F.c_t("int")
    CB  = F.func_t(Int, F.c_t("double")) # Progress callback
    lic = getmod().lic
    lic.argtypes = [arrw_t, arr_t, arr_t, arr_t, CB, Int, Int, Int ]
    lic.restype  = Int

    arrs = [_.reshape(grid.size, order="C") for _ in [out, grid, dx, dy]]
    for i, a in enumerate(arrs):
        if a.dtype != Real:
            arrs[i] = a.astype(Real)

    progress_cb = CB(opt.get("prog", None))
    res = lic(arrs[0], arrs[1], arrs[2], arrs[3], progress_cb, H, W, int_steps)

    if res == 0:
        return None

    out.resize((H,W))

    return out

