# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Karika
                                 A QGIS plugin
 Terrain generalization
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-03-05
        copyright            : (C) 2018 by Roman Geisthövel
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__    = 'Roman Geisthövel'
__date__      = '2018-03-05'
__copyright__ = '(C) 2018 by Roman Geisthövel'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

#=============================================================================
from functools import partial
from os.path import splitext

from PyQt5.QtCore import QCoreApplication

from qgis.PyQt.QtGui import QIcon
from qgis.core import (QgsProcessingAlgorithm,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterNumber,
                       QgsProcessingException,
                       QgsRasterFileWriter,
                       QgsProcessingParameterRasterDestination)

import karika.appinter as appi
from karika.implementation import lic

#=============================================================================
def progress(fraction, feedback, scale=1, translate=0):
    if feedback.isCanceled():
        return 0
    else:
        feedback.setProgress(int((scale*fraction + translate)*100))
        return 1

#=============================================================================
class KarikaAlgorithm(QgsProcessingAlgorithm):

    IN_RASTER  = "IN_RASTER"
    LIC_LENGTH = "LIC_LENGTH"
    Z_FACTOR   = "Z_FACTOR"
    BAND_NO    = "BAND_NO"
    OUT_RASTER = "OUT_RASTER"

    #-------------------------------------------------------------------------
    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm.
        """

        self.addParameter(QgsProcessingParameterRasterLayer(
                            self.IN_RASTER,
                            self.tr('Input raster')))

        self.addParameter(QgsProcessingParameterNumber(
                            self.LIC_LENGTH,
                            self.tr("Integration length"),
                            QgsProcessingParameterNumber.Integer,
                            5, False, 1))

        self.addParameter(QgsProcessingParameterNumber(
                            self.BAND_NO,
                            self.tr("Band number"),
                            QgsProcessingParameterNumber.Integer,
                            1, False, 1))

        self.addParameter(QgsProcessingParameterNumber(
                            self.Z_FACTOR,
                            self.tr("Z factor"),
                            QgsProcessingParameterNumber.Double,
                            1, False, 1e-10))

        self.addParameter(QgsProcessingParameterRasterDestination(
                            self.OUT_RASTER,
                            self.tr('Output raster'), 
                            ".tif"))
            
            

    #-------------------------------------------------------------------------
    def processAlgorithm(self, args, ctx, feedback):
        """
        Here is where the processing itself takes place.
        """
        Ras = appi.Raster
        App = appi.App

        PROG_OFF = 0.01

        log = feedback.setProgressText

        out_raster = self.parameterAsOutputLayer(args, self.OUT_RASTER, ctx)
        res = { self.OUT_RASTER : out_raster }

        # Check output format
        out_fmt = QgsRasterFileWriter.driverForExtension(splitext(out_raster)[1])
        if not out_fmt or out_fmt.lower() != "gtiff":
            log("CRITICAL: Currently only GeoTIFF output format allowed, exiting!")
            return res

        if not progress(0, feedback):
            log("Canceled!")
            return res

        # Process args
        in_raster  = self.parameterAsRasterLayer(args,  self.IN_RASTER,  ctx)
        int_length = self.parameterAsInt(args,          self.LIC_LENGTH, ctx)
        band_no    = self.parameterAsInt(args,          self.BAND_NO,    ctx)
        z_factor   = self.parameterAsDouble(args,       self.Z_FACTOR,   ctx)


        if not (1 <= band_no <= Ras.num_bands(in_raster)):
            log(("WARNING: invalid band number selected ({}), "
                  "using band 1 instead.").format(band_no))
            band_no = 1

        if z_factor <= 0:
            log(("WARNING: invalid z-factor selected ({}), "
                  "using 1 instead.").format(z_factor))
            z_factor = 1
        
        log(self.tr("Reading input raster ..."))
        grid = Ras.to_numpy(in_raster, band=band_no)

        if not progress(PROG_OFF, feedback): 
            log("Canceled!")
            return res

        if z_factor != 1 and z_factor > 0:
            grid *= z_factor

        dy, dx = Ras.cellsize(in_raster)

        if dx != dy:
            cs = (dx+dy) * 0.5
            log(("WARNING: cell dimensions for X and Y differ ({}, {}), "
                  "using mean {} instead.").format(dx,dy,cs))
        else:
            cs = dx

        
        log(self.tr("Processing ..."))

        prog_cb = partial(progress, feedback=feedback, scale=1-2*PROG_OFF, 
                                                            translate=PROG_OFF)
        try:
            smoothed = lic(grid, cs, int_length, prog=prog_cb)
        except NotImplementedError:
            raise QgsProcessingException(
                ("Could not load the C extension module. Take a look at "
                "the HELP on how to compile it for your platform."))

        if smoothed is None:
            log("Canceled or error")
            return res

        log(self.tr("Saving output raster ..."))
        Ras.numpy_to_file(smoothed, out_raster, src=str(in_raster.source()))

        size = appi.Common.file_size(out_raster)
        if (size >> 20):
            log("Size: %i MB" % (size>>20))
        else:
            log("Size: %i bytes" % size)

        if not progress(1, feedback):
            log("Canceled!")
            return res

        log(self.tr("Done!\n"))

        return res

    #-------------------------------------------------------------------------
    def icon(self):
        return QIcon(self.svgIconPath())

    #-------------------------------------------------------------------------
    def svgIconPath(self):
        C = appi.Common
        return C.mkpath(C.folder(), "img_gear.svg")

    #-------------------------------------------------------------------------
    def name(self):
        return 'karika'

    #-------------------------------------------------------------------------
    def displayName(self):
        return "Karika"

    #-------------------------------------------------------------------------
    def group(self):
        return self.tr("Generalization")

    #-------------------------------------------------------------------------
    def groupId(self):
        return "generalization"

    #-------------------------------------------------------------------------
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    #-------------------------------------------------------------------------
    def tags(self):
        return [self.tr(_) for _ in ("Raster", "Generalization", "Processing")]

    #-------------------------------------------------------------------------
    def createInstance(self):
        return KarikaAlgorithm()

    #-------------------------------------------------------------------------
    def helpUrl(self):
        return "file:///%s/help/index.html" % appi.Common.folder()
