# -*- coding: utf-8 -*-
# -*- coding: utf-8 -*-

# KAN Imagery Catalog QGIS plugin
# Satellite image catalog manager

# * Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
# * begin      : 2023-07-14
# * copyright  : (C) 2023 by Fernando D. Gómez :: KAN Territory & IT
# * email      : fgomezdev@gmail.com
# * git sha    : $Format:%H$


# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.


import os

from PyQt5 import QtWidgets, uic
from PyQt5.QtCore import QDate, Qt, QVariant, pyqtSignal
from PyQt5.QtGui import QIcon, QIntValidator
from PyQt5.QtWidgets import QApplication, QListWidgetItem

from core.catalogs import get_catalog, get_thumbnail
from core.settings import PluginSettings
from gui.custom_widgets import CustomWidgetListItem
from gui.form_default_collections import FormDefaultCollections
from gui.form_settings import FormSettings
from gui.helpers import forms
from utils import qgis_helper
from utils.constants import RESULTS_GROUP_NAME, RESULTS_LAYER_NAME
from utils.exceptions import (
    AuthorizationError,
    DataNotFoundError,
    HostError,
    PluginError,
    SettingsError,
    ProviderError
)

FORM_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'kan_imagery_catalog_dock.ui'))


class KANImageryCatalogDock(QtWidgets.QDockWidget, FORM_CLASS):
    """KANImageryCatalogDock Dockwidget class."""

    closing_plugin = pyqtSignal()

    def __init__(self, parent=None):
        """Constructor."""

        super(KANImageryCatalogDock, self).__init__(parent)

        self.parent = parent

        self.setupUi(self)
        forms.set_form_stylesheet(self)

        self.btn_settings.clicked.connect(self.btn_settings_clicked)
        self.btn_get_data.clicked.connect(self.btn_get_data_clicked)
        self.btn_select_catalogs.clicked.connect(self.btn_select_catalogs_clicked)

        self.btn_show_hide_search_area.clicked.connect(self.btn_show_hide_search_area_clicked)
        self.btn_show_hide_catalogs.clicked.connect(self.btn_show_hide_catalogs_clicked)
        self.btn_show_hide_filters.clicked.connect(self.btn_show_hide_filters_clicked)
        self.btn_sort_results.clicked.connect(self.btn_sort_results_clicked)
        # self.btn_sort_results.clicked.connect(lambda: self.sort_list_widget_1(True))

        self.btn_update_layers_list.clicked.connect(self.btn_update_layers_list_clicked)
        self.chk_search_by_dataframe.stateChanged.connect(self.chk_search_by_dataframe_update)
        self.slider_cloud_coverage.valueChanged.connect(self.update_cloud_coverage_label)

        self.btn_select_catalogs.setGraphicsEffect(forms.get_shadow_effect())
        self.btn_get_data.setGraphicsEffect(forms.get_shadow_effect())
        self.btn_get_data.setText('Search')

        int_validator = QIntValidator()
        self.txt_max_catalog_results.setValidator(int_validator)
        self.txt_max_catalog_results.textChanged.connect(self.txt_max_catalog_results_text_changed)

        self.chk_search_by_dataframe.setChecked(True)
        self.chk_search_by_dataframe_update()

        self.lbl_search_area_tooltip.setToolTip("Dataframe or project layers (only 'single polygon' type layers).")
        self.lbl_catalogs_tooltip.setToolTip('Defines the catalogs of each supplier that will be used for searches.')
        self.lbl_filters_tooltip.setToolTip('Perform the search according to the indicated filters.')
        # self.cbo_sort_by.setStyleSheet("QComboBox{border : 0px;}")

        sort_by = [
            {'key': 'angle', 'value': 'Angle'},
            {'key': 'date', 'value': 'Date'},
            {'key': 'cloud_coverage', 'value': 'Cloud coverage'},
            {'key': 'name', 'value': 'Name'},
        ]

        forms.load_combobox(self.cbo_sort_by, 'key', 'value', sort_by)
        self.cbo_sort_by.setHidden(True)
        self.lst_data.setSortingEnabled(True)

        self.settings = PluginSettings()

        default_date_to = QDate.currentDate()
        self.dt_date_to.setDate(default_date_to)

        default_date_from = default_date_to.addDays(int(self.settings.back_days) * -1)
        self.dt_date_from.setDate(default_date_from)
        self.slider_cloud_coverage.setValue(int(self.settings.cloud_coverage or 0))
        self.txt_max_catalog_results.setText(str(self.settings.max_catalog_results or 0))
        self.collections = []

        self.sort_ascending = True

    def closeEvent(self, event):
        """Run close plugin event."""

        self.closing_plugin.emit()
        event.accept()

    def txt_max_catalog_results_text_changed(self):
        """Event handler for textbox 'txt_max_catalog_results'."""

        forms.check_int_not_empty(self.txt_max_catalog_results)

    def update_cloud_coverage_label(self):
        """Event handler for slider 'slider_cloud_coverage'."""

        self.lbl_cloud_coverage.setText(f'Max cloud coverage ({self.slider_cloud_coverage.value()} %)')

    def chk_search_by_dataframe_update(self):
        """Event handler for checkbox 'chk_search_by_dataframe'."""

        self.cbo_layer.setHidden(self.chk_search_by_dataframe.isChecked())
        self.btn_update_layers_list.setHidden(self.chk_search_by_dataframe.isChecked())

        if not self.chk_search_by_dataframe.isChecked():
            self.btn_update_layers_list_clicked()

    def btn_update_layers_list_clicked(self):
        """Event handler for button 'btn_update_layers_list'."""

        self.cbo_layer.clear()

        layer_list = ['  -----  ']
        if not self.chk_search_by_dataframe.isChecked():
            layer_list = qgis_helper.get_single_polygon_layers()

        self.cbo_layer.addItems(layer_list)

    def btn_settings_clicked(self):
        """Event handler for button 'btn_settings'."""

        frm = FormSettings(parent=self, closing_plugin=self.closing_plugin)
        frm.exec()
        self.settings = PluginSettings()

    def btn_select_catalogs_clicked(self):
        """Event handler for button 'btn_select_catalogs'."""

        frm = FormDefaultCollections(parent=self, closing_plugin=self.closing_plugin)
        frm.exec()

    def btn_sort_results_clicked(self):
        """Event handler for button 'btn_sort_results'."""

        icon_path = ':/resources/icons/sort-ascending.svg'
        if not self.sort_ascending:
            icon_path = ':/resources/icons/sort-descending.svg'

        self.btn_sort_results.setIcon(QIcon(icon_path))
        self.lst_data.sortItems(Qt.AscendingOrder if self.sort_ascending else Qt.DescendingOrder)
        self.sort_ascending = not self.sort_ascending

    def btn_show_hide_search_area_clicked(self):
        """Event handler for button 'btn_show_hide_search_area'."""

        icon_path = ':/resources/icons/down.svg'
        if not self.frame_search_area.isVisible():
            icon_path = ':/resources/icons/up.svg'

        self.btn_show_hide_search_area.setIcon(QIcon(icon_path))
        self.frame_search_area.setVisible(not self.frame_search_area.isVisible())

    def btn_show_hide_catalogs_clicked(self):
        """Event handler for button 'btn_show_hide_catalogs'."""

        icon_path = ':/resources/icons/down.svg'
        if not self.frame_catalogs.isVisible():
            icon_path = ':/resources/icons/up.svg'

        self.btn_show_hide_catalogs.setIcon(QIcon(icon_path))
        self.frame_catalogs.setVisible(not self.frame_catalogs.isVisible())

    def btn_show_hide_filters_clicked(self):
        """Event handler for button 'btn_show_hide_filters'."""

        icon_path = ':/resources/icons/down.svg'
        if not self.frame_filters.isVisible():
            icon_path = ':/resources/icons/up.svg'

        self.btn_show_hide_filters.setIcon(QIcon(icon_path))
        self.frame_filters.setVisible(not self.frame_filters.isVisible())

    def btn_get_data_clicked(self):
        """Event handler for button 'btn_get_data'."""

        # Check if there are providers configured...
        if not self.settings.get_active_providers():
            qgis_helper.warning_message(
                'Warning',
                'There are no providers defined in the plugin settings.',
            )
            return

        self.frame_catalog.setDisabled(True)
        self.btn_settings.setDisabled(True)
        self.btn_get_data.setText('Getting results...')

        self.lst_data.clear()

        cloud_coverage = self.slider_cloud_coverage.value()
        date_from = self.dt_date_from.date()
        date_to = self.dt_date_to.date()

        try:
            max_catalog_results = int(self.txt_max_catalog_results.text().strip())
        except Exception:
            max_catalog_results = self.settings.max_catalog_results
            self.txt_max_catalog_results.setText(str(max_catalog_results))

        QApplication.processEvents()

        try:
            if not self.chk_search_by_dataframe.isChecked() and not self.cbo_layer.currentText():
                raise DataNotFoundError('The project has no layers available to use as a reference for searching.')

            layer_name = '' if self.chk_search_by_dataframe.isChecked() else self.cbo_layer.currentText()

            self.get_results(
                layer_name=layer_name,
                cloud_coverage=cloud_coverage,
                date_from=date_from,
                date_to=date_to,
                max_catalog_results=max_catalog_results,
            )
            qgis_helper.success_message('', 'The catalog search has ended.')

        except SettingsError as ex:
            qgis_helper.warning_message('Warning', str(ex))

        except DataNotFoundError as ex:
            qgis_helper.info_message('Warning', str(ex))

        except AuthorizationError as ex:
            qgis_helper.warning_message('Warning', str(ex))

        except HostError as ex:
            qgis_helper.error_message('Error', str(ex))

        except (PluginError, Exception) as ex:
            qgis_helper.error_message('Error', str(ex))

        finally:
            self.frame_catalog.setDisabled(False)
            self.btn_settings.setDisabled(False)
            self.btn_get_data.setText('Search')

    def get_results(self, layer_name, cloud_coverage, date_from, date_to, max_catalog_results):
        """Get results from selected catalogs with selected filters."""

        if layer_name:
            dict_bbox = qgis_helper.get_bounding_box_selected_feature(layer_name)
        else:
            dict_bbox = qgis_helper.get_bounding_box_canvas()

        catalogs = []
        limit_features = self.settings.max_features_results

        if not (isinstance(self.settings.selected_collections, list)) or len(self.settings.selected_collections) == 0:
            raise SettingsError('There are no catalogs selected.')

        catalog_counter = 0

        # Group collections by host
        dict_collections = {}
        collection_aux = {}

        active_providers = self.settings.get_active_providers()

        for collection in self.settings.selected_collections:
            if collection['provider'] not in active_providers:
                continue

            # Save collection name and title to use in results list
            collection_aux[collection['name']] = collection['title']
            host_name = collection['hostName']
            if host_name not in dict_collections:
                dict_collections[host_name] = {'provider': collection['provider'], 'collections': []}

            dict_collections[host_name]['collections'].append(collection['name'])

        for host, values in dict_collections.items():
            provider = values['provider']
            collections = values['collections']

            if catalog_counter >= max_catalog_results:
                break

            datetime_params = f"{date_from.toString('yyyy-MM-ddT00:00:00Z')}/{date_to.toString('yyyy-MM-ddT23:59:59Z')}"

            search_params = {
                'collections': collections,
                'bbox': [dict_bbox['x_min'], dict_bbox['y_min'], dict_bbox['x_max'], dict_bbox['y_max']],
                'datetime': datetime_params,
                'limit': limit_features,
            }

            try:
                catalogs = get_catalog(
                    provider=provider,
                    host_name=host,
                    search_params=search_params,
                    max_cloud_coverage=cloud_coverage,
                    collection_names=collection_aux,
                )
            except AuthorizationError as ex:
                qgis_helper.info_message('Warning', str(ex))
                continue

            except (ProviderError, HostError) as ex:
                qgis_helper.error_message('Error', str(ex))
                continue

            features_counter = 0
            for catalog in catalogs:  # ['features']:
                if features_counter >= limit_features:
                    break

                item_date = catalog['aux_date']
                item_angle = catalog['aux_angle']
                item_cloud_coverage = catalog['aux_cloud_coverage']
                collection_name = catalog['aux_collection_name']
                image_id = catalog['aux_image_id']

                coordinates = catalog['aux_coordinates']
                self.add_feature_to_footprints_layer(
                    coordinates=coordinates,
                    footprint_id=image_id,
                )

                self.add_item_to_results(
                    provider_name=provider,
                    host_name=host,
                    collection_name=collection_name,
                    feature_data=catalog,
                    acquisition_date=item_date,
                    incidence_angle=item_angle,
                    cloud_coverage=item_cloud_coverage,
                    image_id=image_id,
                    feature_index=features_counter,
                )

                features_counter += 1
                QApplication.processEvents()

            catalog_counter += 1
        qgis_helper.success_message('', 'The catalog search has ended.')

    def add_item_to_results(
        self,
        provider_name,
        host_name,
        collection_name,
        feature_data,
        acquisition_date,
        incidence_angle,
        cloud_coverage,
        image_id,
        feature_index,
    ):
        """Add item to results list."""

        thumbnail = get_thumbnail(
            provider=provider_name,
            host_name=host_name,
            collection_name=collection_name,
            image_id=image_id,
            feature_data=feature_data,
        )

        custom_item = CustomWidgetListItem(
            parent=self.lst_data,
            provider_name=provider_name,
            host_name=host_name,
            collection_name=collection_name,
            feature_data=feature_data,
            thumbnail=thumbnail,
            acquisition_date=acquisition_date,
            incidence_angle=incidence_angle,
            cloud_coverage=cloud_coverage,
            image_id=image_id,
            feature_index=feature_index,
            closing_plugin=self.closing_plugin,
        )

        item = QListWidgetItem(self.lst_data)
        item.setSizeHint(custom_item.sizeHint())
        item.setData(Qt.UserRole, QVariant(custom_item))
        item.setText(acquisition_date or '')
        self.lst_data.addItem(item)
        self.lst_data.setItemWidget(item, custom_item)

    def add_feature_to_footprints_layer(self, coordinates, footprint_id):
        footprints_layer = qgis_helper.get_or_create_footprints_layer(RESULTS_LAYER_NAME, RESULTS_GROUP_NAME)
        qgis_helper.add_feature_to_layer(coordinates, footprint_id, footprints_layer)
