##############################################
#
#Name: Alexandra Crowe
#Date: 5 July 2020
#Description:
# This is the class hierarchy script that defines the
# waterbody classes. There are nine waterbody classes.
# waterbody is the root class that defines the name and
# geometry of each object. The linear and areal waterbody
# classes are sub objects of the waterbody class. Stream,
# river and canal classes are object of the linear waterbody
# class and lake, pond and reservoir are sub classes of the
# areal waterbody class. Each body of water are defined by
# their own specific tags. For new bodies of water, this code
# can be modified with the tag criteria of entity.
#
###############################################

import datetime
import qgis
from qgis.core import *
import qgis.core


class waterbody():
    """ abstract root class that derives more specific bodies of water. Each derived
    class has an associate name and geometry object. Each instantiable  class should
    implement fromOSMWay() and toQgsFeature() methods. """
    def __init__(self, name, geometry):
        """ each waterbody object should have an instance variable called name and called geometry """
        self.name = name
        self.geometry = geometry

    def fromOSMWAY(way, allNodes):
        pass

    def toQgsFeature(self):
        pass

class linearWaterbody(waterbody):
    """abstract class for all bodies of water that are linear. Each object of a linear
    waterbody should have an instance variable called length"""
    def __init__(self, name, geometry):
        super(linearWaterbody, self).__init__(name, geometry)

        qda = qgis.core.QgsDistanceArea()
        qda.setEllipsoid("WGS84")
        length = qda.measureLength(geometry)

        self.length = qda.convertLengthMeasurement(length, qgis.core.QgsUnitTypes.DistanceMeters) #each linear waterbody object should have an instance variable called length

    def savePolylineToFile(features, filename, fileFormat):
        """ use feature list from the linear instances toQgsFeature() methods to create
        a new polyline layer"""
        layer = qgis.core.QgsVectorLayer('LineString?crs=EPSG:4326&field=Name:string(255)&field=Type:string(255)&field=Length:integer', 'tracks' , 'memory' )
        prov = layer.dataProvider()
        prov.addFeatures(features)
        qgis.core.QgsVectorFileWriter.writeAsVectorFormat( layer, filename, "utf-8", layer.crs(), fileFormat )
        currentProject = qgis.core.QgsProject.instance()
        currentProject.addMapLayer(layer)

class arealWaterbody(waterbody):
    """abstract class for all bodies of water that are polygons. Each object of a areal
    waterbody should have an instance variable called area"""
    def __init__(self, name, geometry):
        super(arealWaterbody, self).__init__(name, geometry)

        # calculate area of this areal waterbody
        qda = qgis.core.QgsDistanceArea()
        qda.setEllipsoid('WGS84')
        area = qda.measureArea(geometry)

        # instance variable for storing the area of this areal waterbody
        self.area = qda.convertAreaMeasurement(area, qgis.core.QgsUnitTypes.AreaSquareMeters) #each areal waterbody object should have an instance variable called area

    def savePolygonToFile(features, filename, fileFormat):
        """ use feature list from the areal instances toQgsFeature() methods to create
        a new polygon layer"""
        layer = qgis.core.QgsVectorLayer('Polygon?crs=EPSG:4326&field=Name:string(255)&field=Type:string(255)&field=Area:integer', 'tracks' , 'memory' )
        prov = layer.dataProvider()
        prov.addFeatures(features)
        qgis.core.QgsVectorFileWriter.writeAsVectorFormat( layer, filename, "utf-8", layer.crs(), fileFormat )
        currentProject = qgis.core.QgsProject.instance()
        currentProject.addMapLayer(layer)

class stream(linearWaterbody):
    """sub class of the linearWaterbody class. Stream is defined by a way with a "tag"
    dictionary and a waterway element named stream."""
    def __init__(self, name, geometry):
        super(stream, self).__init__(name, geometry)

    def fromOSMWay(way, allNodes):
        """This method checks the tags of the way element to see whether this element describes
        a stream. Then creates geometry by matching the nodes with node IDs from the allNodes
        dictionary. Identifies the name and then creates an object of the class stream with the
        name and geometry"""
        if "tags" in way:
            tag = way["tags"]
            if "name" in tag:
                name = tag["name"]
            else:
                name = "unknown"
            if "waterway" in tag and tag["waterway"] == "stream":
                node = way["nodes"]
                points = [ qgis.core.QgsPointXY((allNodes[n]["lon"]),(allNodes[n]["lat"])) for n in node]
                lineGeometry = qgis.core.QgsGeometry.fromPolylineXY(points)
                return stream(name,lineGeometry)

    def toQgsFeature(self):
        """ Creates a new QgsFeature object of the polyline geometry for a stream with name, type and length as attributes.
        Returns the feature """
        feature = qgis.core.QgsFeature()
        feature.setAttributes([self.name, "Stream", self.length])
        feature.setGeometry(self.geometry)
        return feature

    def __str__(self):
        return "Stream {} (length: {}m)".format(self.name, self.length)

class river(linearWaterbody):
    """sub class of the linearWaterbody class. River is defined by a way with a "tag"
    dictionary and a waterway element named river."""
    def __init__(self, name, geometry):
        super(river, self).__init__(name, geometry)

    def fromOSMWay(way, allNodes):
        """This method checks the tags of the way element to see whether this element describes
        a river. Then creates geometry by matching the nodes with node IDs from the allNodes
        dictionary. Identifies the name and then creates an object of the class river with the
        name and geometry"""
        if "tags" in way:
            tag = way["tags"]
            if "name" in tag:
                name = tag["name"]
            else:
                name = "unknown"
            if "waterway" in tag and tag["waterway"] == "river":
                node = way["nodes"]
                points = [ qgis.core.QgsPointXY((allNodes[n]["lon"]),(allNodes[n]["lat"])) for n in node]
                lineGeometry = qgis.core.QgsGeometry.fromPolylineXY(points)
                return river(name,lineGeometry)

    def toQgsFeature(self):
        """ Creates a new QgsFeature object of the polyline geometry for a river with name, type and length as attributes.
        Returns the feature """
        feature = qgis.core.QgsFeature()
        feature.setAttributes([self.name, "River", self.length])
        feature.setGeometry(self.geometry)
        return feature

    def __str__(self):
        return "River {} (length: {}m)".format(self.name, self.length)

class canal(linearWaterbody):
    """sub class of the linearWaterbody class. Canal is defined by a way with a "tag"
    dictionary and a waterway element named canal."""
    def __init__(self, name, geometry):
        super(canal, self).__init__(name, geometry)

    def fromOSMWay(way, allNodes):
        """This method checks the tags of the way element to see whether this element describes
        a canal. Then creates geometry by matching the nodes with node IDs from the allNodes
        dictionary. Identifies the name and then creates an object of the class canal with the
        name and geometry"""
        if "tags" in way:
            tag = way["tags"]
            if "name" in tag:
                name = tag["name"]
            else:
                name = "unknown"
            if "waterway" in tag and tag["waterway"] == "canal":
                node = way["nodes"]
                points = [ qgis.core.QgsPointXY((allNodes[n]["lon"]),(allNodes[n]["lat"])) for n in node]
                lineGeometry = qgis.core.QgsGeometry.fromPolylineXY(points)
                return canal(name,lineGeometry)

    def toQgsFeature(self):
        """ Creates a new QgsFeature object of the polyline geometry for a canal with name, type and length as attributes.
        Returns the feature """
        feature = qgis.core.QgsFeature()
        feature.setAttributes([self.name, "Canal", self.length])
        feature.setGeometry(self.geometry)
        return feature

    def __str__(self):
        return "Canal {} (length: {}m)".format(self.name, self.length)

"""sub class of the arealWaterbody class. Lake is defined by a way with a "tag" dictionary and a natural and  water keys with values water and lake respectively."""
class lake(arealWaterbody):
    def __init__(self, name, geometry):
        super(lake, self).__init__(name, geometry)

    def fromOSMWay(way, allNodes):
        """This method checks the tags of the way element to see whether this element describes
        a lake. Then creates geometry by matching the nodes with node IDs from the allNodes
        dictionary. Identifies the name and then creates an object of the class lake with the
        name and geometry"""
        if "tags" in way:
            tag = way["tags"]
            if "name" in tag:
                name = tag["name"]
            else:
                name = "unknown"
            if "natural" and "water" in tag and tag["natural"] == "water" and tag["water"] == "lake":
                node = way["nodes"]
                points = [ qgis.core.QgsPointXY((allNodes[n]["lon"]),(allNodes[n]["lat"])) for n in node]
                areaGeometry = qgis.core.QgsGeometry.fromPolygonXY([points])
                return lake(name,areaGeometry)

    def toQgsFeature(self):
        """ Creates a new QgsFeature object of the polygon geometry for a lake with name, type and area as attributes.
        Returns the feature """
        feature = qgis.core.QgsFeature()
        feature.setAttributes([self.name, "Lake", self.area])
        feature.setGeometry(self.geometry)
        return feature

    def __str__(self):
        return "Lake {} (area: {}m)".format(self.name, self.area)

"""sub class of the arealWaterbody class. Pond is defined by a way with a "tag" dictionary and a natural and  water keys with values water and pond respectively."""
class pond(arealWaterbody):

    def __init__(self, name, geometry):
        super(pond, self).__init__(name, geometry)

    def fromOSMWay(way, allNodes):
        """This method checks the tags of the way element to see whether this element describes
        a pond. Then creates geometry by matching the nodes with node IDs from the allNodes
        dictionary. Identifies the name and then creates an object of the class pond with the
        name and geometry"""
        if "tags" in way:
            tag = way["tags"]
            if "name" in tag:
                name = tag["name"]
            else:
                name = "unknown"
            if "natural" and "water" in tag and tag["natural"] == "water" and tag["water"] == "pond":
                node = way["nodes"]
                points = [ qgis.core.QgsPointXY((allNodes[n]["lon"]),(allNodes[n]["lat"])) for n in node]
                areaGeometry = qgis.core.QgsGeometry.fromPolygonXY([points])
                return pond(name,areaGeometry)

    def toQgsFeature(self):
        """ Creates a new QgsFeature object of the polygon geometry for a pond with name, type and area as attributes.
        Returns the feature """
        feature = qgis.core.QgsFeature()
        feature.setAttributes([self.name, "Pond", self.area])
        feature.setGeometry(self.geometry)
        return feature

    def __str__(self):
        return "Pond {} (area: {}m)".format(self.name, self.area)

"""sub class of the arealWaterbody class. Reservoir is defined by a way with a "tag" dictionary and a natural and  water keys with values water and reservoir respectively."""
class reservoir(arealWaterbody):
    def __init__(self, name, geometry):
        super(reservoir, self).__init__(name, geometry)

    def fromOSMWay(way, allNodes):
        """This method checks the tags of the way element to see whether this element describes
        a reservoir. Then creates geometry by matching the nodes with node IDs from the allNodes
        dictionary. Identifies the name and then creates an object of the class reservoir with the
        name and geometry"""
        if "tags" in way:
            tag = way["tags"]
            if "name" in tag:
                name = tag["name"]
            else:
                name = "unknown"
            if "natural" and "water" in tag and tag["natural"] == "water" and tag["water"] == "reservoir":
                node = way["nodes"]
                points = [ (qgis.core.QgsPointXY((allNodes[n]["lon"]),(allNodes[n]["lat"]))) for n in node]
                areaGeometry = qgis.core.QgsGeometry.fromPolygonXY([points])
                return reservoir(name, areaGeometry)

    def toQgsFeature(self):
        """ Creates a new QgsFeature object of the polygon geometry for a reservoir with name, type and area as attributes.
        Returns the feature """
        feature = qgis.core.QgsFeature()
        feature.setAttributes([self.name, "Reservoir", self.area])
        feature.setGeometry(self.geometry)
        return feature

    def __str__(self):
        return "Reservoir {} (area: {}m)".format(self.name, self.area)
