from qgis.PyQt.QtWidgets import (
    QDockWidget, QWidget, QVBoxLayout, QLabel, QPushButton, QComboBox, QDialog,
    QMessageBox, QHBoxLayout, QFormLayout, QLineEdit, QListWidget
)
from qgis.PyQt.QtCore import Qt
from qgis.core import (
    QgsProject, QgsSpatialIndex, QgsWkbTypes, QgsGeometry, QgsPointXY
)
from qgis.core import (
    QgsMapLayerType,QgsVectorLayer, QgsField, QgsFeature, QgsGeometry, QgsPointXY, QgsWkbTypes
)
from PyQt6.QtCore import QVariant
from .auto_address_plugin import LayerSelectionDialog
from .processing_logic import process_layers

class joTOOLSWindow(QDockWidget):
    def __init__(self, iface):
        super().__init__()
        self.iface = iface
        self.setWindowTitle("joTOOLS")

        self.main_widget = QWidget()
        self.setWidget(self.main_widget)

        layout = QVBoxLayout()
        self.label = QLabel("Welcome to the Toolbox!")
        self.combo = QComboBox()
        self.combo.addItems([
            "Auto Address Creator",
            "Delete Points Without External IDs",
            "Centralize Address Points",
            "Unit Counter",
            "Duplicate Detector",
            "Grid Generator"
        ])
        self.button = QPushButton("Run")

        layout.addWidget(self.label)
        layout.addWidget(self.combo)
        layout.addWidget(self.button)
        self.main_widget.setLayout(layout)

        self.button.clicked.connect(self.on_button_click)


    def on_button_click(self):
        selected_option = self.combo.currentText()
        self.label.setText(f"You selected: {selected_option}")
        if selected_option == "Auto Address Creator":
            self.launch_auto_address_dialog()
        elif selected_option == "Delete Points Without External IDs":
            self.launch_delete_points_without_external_ids_dialog()
        elif selected_option == "Centralize Address Points":
            self.launch_centralize_address_points_dialog()
        elif selected_option == "Unit Counter":
            self.launch_unit_counter()
        elif selected_option == "Duplicate Detector":
            self.launch_duplicate_detector_dialog()
        elif selected_option == "Grid Generator":
            self.launch_grid_generator_dialog()
        else:
            self.launch_generic_dialog(f"{selected_option} Window")



    def launch_auto_address_dialog(self):
        dialog = LayerSelectionDialog(self)
        if dialog.exec() != QDialog.DialogCode.Accepted:
            return

        parcel_name, address_name, da_name, hut_name, da_project = dialog.get_selected_options()
        try:
            added_count = process_layers(parcel_name, address_name, da_name, hut_name, da_project)
            QMessageBox.information(self, "Done", f"{added_count} new address point(s) added with full attributes.")
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Processing failed:\\n{str(e)}")


    def launch_delete_points_without_external_ids_dialog(self):
        dialog = QDialog(self)
        dialog.setWindowTitle("Select Address Layer")
        layout = QVBoxLayout()

        # Get active vector layers
        layers = QgsProject.instance().mapLayers().values()
        vector_layers = [lyr for lyr in layers if lyr.type() == lyr.VectorLayer]
        layer_names = [lyr.name() for lyr in vector_layers]

        if not layer_names:
            QMessageBox.warning(self, "No Layers", "No vector layers found.")
            return

        combo = QComboBox()
        combo.addItems(layer_names)
        layout.addWidget(QLabel("Select Address Layer:"))
        layout.addWidget(combo)

        # Buttons
        button_layout = QHBoxLayout()
        ok_btn = QPushButton("Delete")
        cancel_btn = QPushButton("Cancel")
        button_layout.addWidget(ok_btn)
        button_layout.addWidget(cancel_btn)
        layout.addLayout(button_layout)

        dialog.setLayout(layout)

        def on_ok():
            selected_name = combo.currentText()
            try:
                address_layer = QgsProject.instance().mapLayersByName(selected_name)[0]
                address_layer.startEditing()
                ids_to_delete = [
                    feature.id()
                    for feature in address_layer.getFeatures()
                    if not feature['External ID'] or str(feature['External ID']).strip() == ''
                ]
                if ids_to_delete:
                    address_layer.deleteFeatures(ids_to_delete)
                address_layer.commitChanges()

                # Show result in a new dialog
                result_dialog = QDialog(self)
                result_dialog.setWindowTitle("Deletion Summary")
                result_layout = QVBoxLayout()
                result_layout.addWidget(QLabel(f"{len(ids_to_delete)} address point(s) without External ID were deleted."))
                close_btn = QPushButton("Close")
                close_btn.clicked.connect(result_dialog.accept)
                result_layout.addWidget(close_btn)
                result_dialog.setLayout(result_layout)
                result_dialog.exec()

            except Exception as e:
                QMessageBox.critical(self, "Error", f"Deletion failed:\n{str(e)}")
            dialog.accept()

        ok_btn.clicked.connect(on_ok)
        cancel_btn.clicked.connect(dialog.reject)
        dialog.exec()

    def launch_centralize_address_points_dialog(self):
        try:
            # Get active vector layers
            layers = QgsProject.instance().mapLayers().values()
            vector_layers = [lyr for lyr in layers if lyr.type() == lyr.VectorLayer]

            # Identify address and parcel layers based on geometry type
            address_layer = None
            parcel_layer = None
            for lyr in vector_layers:
                if lyr.geometryType() == QgsWkbTypes.PointGeometry and address_layer is None:
                    address_layer = lyr
                elif lyr.geometryType() == QgsWkbTypes.PolygonGeometry and parcel_layer is None:
                    parcel_layer = lyr

            if not address_layer or not parcel_layer:
                QMessageBox.warning(self, "Layer Not Found", "Active address or parcel layer not found.")
                return

            # Build spatial index for parcels
            parcel_index = QgsSpatialIndex(parcel_layer.getFeatures())

            # Start editing address layer
            if not address_layer.isEditable():
                address_layer.startEditing()

            centralized_count = 0

            for address_feat in address_layer.getFeatures():
                address_geom = address_feat.geometry()
                if address_geom.type() != QgsWkbTypes.PointGeometry:
                    continue

                # Find intersecting parcel
                intersecting_ids = parcel_index.intersects(address_geom.boundingBox())
                for parcel_id in intersecting_ids:
                    parcel_feat = parcel_layer.getFeature(parcel_id)
                    if parcel_feat.geometry().contains(address_geom):
                        # Get centroid of parcel
                        centroid = parcel_feat.geometry().centroid().asPoint()
                        new_geom = QgsGeometry.fromPointXY(QgsPointXY(centroid))
                        if address_layer.changeGeometry(address_feat.id(), new_geom):
                            centralized_count += 1
                        break  # Stop after first match

            # Commit changes
            if address_layer.commitChanges():
                QMessageBox.information(self, "Success", f"{centralized_count} address point(s) centralized within parcels.")
            else:
                address_layer.rollBack()
                QMessageBox.critical(self, "Failure", "Failed to commit geometry changes.")

        except Exception as e:
            QMessageBox.critical(self, "Error", f"Centralization failed:\n{str(e)}")

    def launch_unit_counter(self):
        class UnitNumberDialog(QDialog):
            def __init__(self):
                super().__init__()
                self.setWindowTitle("Fill Unit Numbers for Specific Address")
                layout = QVBoxLayout()

                self.layer_combo = QComboBox()
                self.layers_dict = {}
                for layer in QgsProject.instance().mapLayers().values():
                    self.layer_combo.addItem(layer.name())
                    self.layers_dict[layer.name()] = layer

                form_layout = QFormLayout()
                self.field_input = QLineEdit("Unit Number")
                self.units_input = QLineEdit("10")
                self.floors_input = QLineEdit("4")
                self.address_input = QLineEdit()
                form_layout.addRow("Layer:", self.layer_combo)
                form_layout.addRow("Field name:", self.field_input)
                form_layout.addRow("Units per floor:", self.units_input)
                form_layout.addRow("Total floors:", self.floors_input)
                form_layout.addRow("Target ID:", self.address_input)
                layout.addLayout(form_layout)

                self.status_label = QLabel("")
                layout.addWidget(self.status_label)

                self.run_button = QPushButton("Run")
                self.run_button.clicked.connect(self.run_script)
                layout.addWidget(self.run_button)

                self.setLayout(layout)

            def run_script(self):
                layer_name = self.layer_combo.currentText()
                layer = self.layers_dict.get(layer_name)
                if layer is None:
                    self.status_label.setText(f"Layer '{layer_name}' not found.")
                    return

                field_name = self.field_input.text().strip()
                if field_name not in [f.name() for f in layer.fields()]:
                    self.status_label.setText(f"Field '{field_name}' not found in {layer_name}.")
                    return

                try:
                    units_per_floor = int(self.units_input.text())
                    total_floors = int(self.floors_input.text())
                except ValueError:
                    self.status_label.setText("Units per floor and total floors must be integers.")
                    return

                target_address = self.address_input.text().strip()
                if not target_address:
                    self.status_label.setText("Please enter a target address.")
                    return

                if not layer.isEditable():
                    if not layer.startEditing():
                        self.status_label.setText("Failed to start editing layer.")
                        return

                field_index = layer.fields().indexOf(field_name)
                count = 0

                if "ID" not in [f.name() for f in layer.fields()]:
                    self.status_label.setText("Layer does not have an 'ID' field.")
                    return

                for feature in layer.getFeatures():
                    if feature["ID"] != target_address:
                        continue
                    floor = (count // units_per_floor) + 1
                    unit_index = (count % units_per_floor) + 1
                    if floor > total_floors:
                        break
                    unit_number = f"{floor}{str(unit_index).zfill(2)}"
                    if not layer.changeAttributeValue(feature.id(), field_index, unit_number):
                        self.status_label.setText(f"Failed to update feature {feature.id()}.")
                        layer.rollBack()
                        return
                    count += 1

                if count == 0:
                    self.status_label.setText(f"No features found for address '{target_address}'.")
                    layer.rollBack()
                    return

                if layer.commitChanges():
                    self.status_label.setText(f"Updated {count} features successfully.")
                else:
                    layer.rollBack()
                    self.status_label.setText("Failed to commit changes. Rolled back edits.")

        dlg = UnitNumberDialog()
        dlg.exec()

    def launch_duplicate_detector_dialog(self):
        dialog = QDialog(self)
        dialog.setWindowTitle("Duplicate Detector")
        layout = QVBoxLayout()

        layer_combo = QComboBox()
        field_combo = QComboBox()
        layers_dict = {}

        # Populate layer combo and dictionary
        
        for layer in QgsProject.instance().mapLayers().values():
            if layer.type() == QgsMapLayerType.VectorLayer:
                layer_combo.addItem(layer.name())
                layers_dict[layer.name()] = layer


        form_layout = QFormLayout()
        form_layout.addRow("Select Layer:", layer_combo)
        form_layout.addRow("Field to Check:", field_combo)
        layout.addLayout(form_layout)

        result_list = QListWidget()
        layout.addWidget(result_list)

        run_btn = QPushButton("Detect Duplicates")
        layout.addWidget(run_btn)

        dialog.setLayout(layout)

        # Function to update field combo when layer changes
        def update_field_combo():
            layer_name = layer_combo.currentText()
            layer = layers_dict.get(layer_name)
            field_combo.clear()
            if layer:
                field_combo.addItems([f.name() for f in layer.fields()])

        layer_combo.currentIndexChanged.connect(update_field_combo)
        update_field_combo()  # Initialize field list for the first layer

        # Function to run duplicate detection
        
        def run_detection():
            layer_name = layer_combo.currentText()
            field_name = field_combo.currentText()
            layer = layers_dict.get(layer_name)

            if not layer or field_name not in [f.name() for f in layer.fields()]:
                QMessageBox.warning(dialog, "Invalid Input", "Please select a valid layer and field.")
                return

            value_counts = {}
            for feature in layer.getFeatures():
                value = str(feature[field_name])
                value_counts[value] = value_counts.get(value, 0) + 1

            duplicates = [val for val, count in value_counts.items() if count > 1]

            result_list.clear()
            result_list.addItem(f"Found {len(duplicates)} duplicate value(s):")
            for val in duplicates:
                result_list.addItem(f"{val} ({value_counts[val]} occurrences)")

            # Select duplicate features in the attribute table
            duplicate_ids = [feature.id() for feature in layer.getFeatures() if str(feature[field_name]) in duplicates]
            layer.selectByIds(duplicate_ids)

        run_btn.clicked.connect(run_detection)
        dialog.exec()

    def launch_generic_dialog(self, title):
        dialog = QDialog(self)
        dialog.setWindowTitle(title)
        layout = QVBoxLayout()
        layout.addWidget(QLabel(f"This is the {title}"))
        dialog.setLayout(layout)
        dialog.exec()
