# -*- coding: utf-8 -*-
"""
Japanese Area Unit - QGIS Plugin
Main plugin class for area unit conversion.
"""

import os
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QSize, QVariant
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import (
    QAction, QDialog, QVBoxLayout, QHBoxLayout, QLabel,
    QLineEdit, QComboBox, QPushButton, QGridLayout, QGroupBox,
    QMessageBox
)
from qgis.core import (
    QgsProject, QgsWkbTypes, QgsDistanceArea,
    QgsPalLayerSettings, QgsTextFormat, QgsVectorLayerSimpleLabeling,
    QgsField, QgsExpression
)


class JapaneseAreaUnitDialog(QDialog):
    """Dialog for area unit conversion."""

    # Conversion factors to square meters
    UNITS = {
        'm2': ('Square Meters', 1.0),
        'ha': ('Hectares', 10000.0),
        'a': ('Ares', 100.0),
        'tsubo': ('Tsubo', 3.305785),
        'se': ('Se', 99.17355),
        'tan': ('Tan', 991.7355),
        'cho': ('Cho', 9917.355),
    }

    def __init__(self, parent=None, iface=None):
        """Initialize the dialog."""
        super().__init__(parent)
        self.iface = iface
        self.setWindowTitle(self.tr('Japanese Area Unit Converter'))
        self.setMinimumWidth(450)
        self.setup_ui()

    def setup_ui(self):
        """Set up the user interface."""
        layout = QVBoxLayout()

        # Layer group
        layer_group = QGroupBox(self.tr('Layer Area'))
        layer_layout = QGridLayout()

        self.layer_combo = QComboBox()
        self.layer_combo.currentIndexChanged.connect(self.on_layer_changed)

        refresh_btn = QPushButton(self.tr('Refresh'))
        refresh_btn.clicked.connect(self.refresh_layers)

        # Feature selection
        self.feature_combo = QComboBox()
        self.feature_combo.addItem(self.tr('All Features'), 'all')

        # Unit selection for label
        self.label_unit_combo = QComboBox()
        for key, (name, _) in self.UNITS.items():
            display_name = self.get_display_name(key, name)
            self.label_unit_combo.addItem(display_name, key)

        get_area_btn = QPushButton(self.tr('Get Area'))
        get_area_btn.clicked.connect(self.get_layer_area)

        add_label_btn = QPushButton(self.tr('Add Label'))
        add_label_btn.clicked.connect(self.add_area_label)

        layer_layout.addWidget(QLabel(self.tr('Layer:')), 0, 0)
        layer_layout.addWidget(self.layer_combo, 0, 1)
        layer_layout.addWidget(refresh_btn, 0, 2)
        layer_layout.addWidget(QLabel(self.tr('Feature:')), 1, 0)
        layer_layout.addWidget(self.feature_combo, 1, 1, 1, 2)
        layer_layout.addWidget(QLabel(self.tr('Label Unit:')), 2, 0)
        layer_layout.addWidget(self.label_unit_combo, 2, 1, 1, 2)
        layer_layout.addWidget(get_area_btn, 3, 0, 1, 2)
        layer_layout.addWidget(add_label_btn, 3, 2)
        layer_group.setLayout(layer_layout)

        # Input group
        input_group = QGroupBox(self.tr('Manual Input'))
        input_layout = QGridLayout()

        self.input_value = QLineEdit()
        self.input_value.setPlaceholderText(self.tr('Enter value'))
        self.input_value.textChanged.connect(self.convert)

        self.input_unit = QComboBox()
        for key, (name, _) in self.UNITS.items():
            display_name = self.get_display_name(key, name)
            self.input_unit.addItem(display_name, key)
        self.input_unit.currentIndexChanged.connect(self.convert)

        input_layout.addWidget(QLabel(self.tr('Value:')), 0, 0)
        input_layout.addWidget(self.input_value, 0, 1)
        input_layout.addWidget(QLabel(self.tr('Unit:')), 1, 0)
        input_layout.addWidget(self.input_unit, 1, 1)
        input_group.setLayout(input_layout)

        # Output group
        output_group = QGroupBox(self.tr('Conversion Results'))
        output_layout = QGridLayout()

        self.result_labels = {}
        row = 0
        for key, (name, _) in self.UNITS.items():
            display_name = self.get_display_name(key, name)
            label = QLabel(f'{display_name}:')
            value_label = QLabel('0')
            value_label.setTextInteractionFlags(Qt.TextSelectableByMouse)
            self.result_labels[key] = value_label
            output_layout.addWidget(label, row, 0)
            output_layout.addWidget(value_label, row, 1)
            row += 1

        output_group.setLayout(output_layout)

        # Buttons
        button_layout = QHBoxLayout()
        clear_btn = QPushButton(self.tr('Clear'))
        clear_btn.clicked.connect(self.clear_input)
        close_btn = QPushButton(self.tr('Close'))
        close_btn.clicked.connect(self.close)
        button_layout.addWidget(clear_btn)
        button_layout.addStretch()
        button_layout.addWidget(close_btn)

        # Add to main layout
        layout.addWidget(layer_group)
        layout.addWidget(input_group)
        layout.addWidget(output_group)
        layout.addLayout(button_layout)

        self.setLayout(layout)

    def refresh_layers(self):
        """Refresh the layer combo box with polygon layers."""
        self.layer_combo.clear()
        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            if layer.type() == layer.VectorLayer:
                if layer.geometryType() == QgsWkbTypes.PolygonGeometry:
                    self.layer_combo.addItem(layer.name(), layer.id())
        self.on_layer_changed()

    def on_layer_changed(self):
        """Update feature combo when layer changes."""
        self.feature_combo.clear()
        self.feature_combo.addItem(self.tr('All Features'), 'all')

        layer_id = self.layer_combo.currentData()
        if not layer_id:
            return

        layer = QgsProject.instance().mapLayer(layer_id)
        if not layer:
            return

        # Add individual features
        for feature in layer.getFeatures():
            fid = feature.id()
            # Try to get a name field
            name = None
            for field_name in ['name', 'Name', 'NAME', 'id', 'ID', 'fid', 'FID']:
                if field_name in [f.name() for f in layer.fields()]:
                    name = feature[field_name]
                    break

            if name:
                display = f"ID:{fid} - {name}"
            else:
                display = f"ID:{fid}"

            self.feature_combo.addItem(display, fid)

    def get_layer_area(self):
        """Get the area of the selected layer or feature."""
        layer_id = self.layer_combo.currentData()
        if not layer_id:
            return

        layer = QgsProject.instance().mapLayer(layer_id)
        if not layer:
            return

        # Calculate area
        d = QgsDistanceArea()
        d.setSourceCrs(layer.crs(), QgsProject.instance().transformContext())
        d.setEllipsoid(QgsProject.instance().ellipsoid())

        feature_selection = self.feature_combo.currentData()

        total_area = 0.0
        if feature_selection == 'all':
            # All features
            for feature in layer.getFeatures():
                geom = feature.geometry()
                if geom and not geom.isNull():
                    total_area += d.measureArea(geom)
        else:
            # Single feature
            feature = layer.getFeature(feature_selection)
            if feature:
                geom = feature.geometry()
                if geom and not geom.isNull():
                    total_area = d.measureArea(geom)

        # Set value in input field (in square meters)
        self.input_value.setText(str(total_area))
        self.input_unit.setCurrentIndex(0)  # Set to m2

    def add_area_label(self):
        """Add area labels to the layer."""
        layer_id = self.layer_combo.currentData()
        if not layer_id:
            QMessageBox.warning(self, self.tr('Warning'), self.tr('Please select a layer.'))
            return

        layer = QgsProject.instance().mapLayer(layer_id)
        if not layer:
            return

        # Get selected unit
        unit_key = self.label_unit_combo.currentData()
        _, unit_factor = self.UNITS[unit_key]
        unit_name = self.get_display_name(unit_key, self.UNITS[unit_key][0])

        # Create expression for area calculation
        # $area returns area in layer CRS units, need to convert
        if unit_key == 'm2':
            expression = "format_number($area, 2) || ' m²'"
        elif unit_key == 'ha':
            expression = "format_number($area / 10000, 4) || ' ha'"
        elif unit_key == 'a':
            expression = "format_number($area / 100, 4) || ' a'"
        elif unit_key == 'tsubo':
            expression = "format_number($area / 3.305785, 2) || ' 坪'"
        elif unit_key == 'se':
            expression = "format_number($area / 99.17355, 4) || ' 畝'"
        elif unit_key == 'tan':
            expression = "format_number($area / 991.7355, 4) || ' 反'"
        elif unit_key == 'cho':
            expression = "format_number($area / 9917.355, 4) || ' 町'"

        # Set up labeling
        settings = QgsPalLayerSettings()
        settings.fieldName = expression
        settings.isExpression = True

        text_format = QgsTextFormat()
        text_format.setSize(10)
        settings.setFormat(text_format)

        labeling = QgsVectorLayerSimpleLabeling(settings)
        layer.setLabeling(labeling)
        layer.setLabelsEnabled(True)
        layer.triggerRepaint()

        QMessageBox.information(
            self,
            self.tr('Label Added'),
            self.tr('Area labels have been added to the layer in {}.').format(unit_name)
        )

    def get_display_name(self, key, english_name):
        """Get display name with Japanese translation for Japanese units."""
        display_names = {
            'm2': '㎡',
            'ha': 'ha',
            'a': 'a (アール)',
            'tsubo': '坪',
            'se': '畝',
            'tan': '反',
            'cho': '町',
        }
        return display_names.get(key, english_name)

    def convert(self):
        """Perform the conversion."""
        try:
            value = float(self.input_value.text())
        except ValueError:
            for label in self.result_labels.values():
                label.setText('0')
            return

        input_key = self.input_unit.currentData()
        _, input_factor = self.UNITS[input_key]

        # Convert to square meters first
        value_in_m2 = value * input_factor

        # Convert to all units
        for key, (_, factor) in self.UNITS.items():
            result = value_in_m2 / factor
            if result == int(result):
                self.result_labels[key].setText(f'{int(result):,}')
            else:
                self.result_labels[key].setText(f'{result:,.6f}'.rstrip('0').rstrip('.'))

    def clear_input(self):
        """Clear input field."""
        self.input_value.clear()
        for label in self.result_labels.values():
            label.setText('0')

    def tr(self, message):
        """Get the translation for a string."""
        return QCoreApplication.translate('JapaneseAreaUnit', message)


class JapaneseAreaUnit:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = self.tr('&Japanese Area Unit')
        self.toolbar = self.iface.addToolBar('JapaneseAreaUnit')
        self.toolbar.setObjectName('JapaneseAreaUnit')
        self.toolbar.setToolButtonStyle(Qt.ToolButtonTextOnly)
        self.dialog = None

        # Initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir, 'i18n', f'{locale}.qm')

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        return QCoreApplication.translate('JapaneseAreaUnit', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None
    ):
        """Add a toolbar icon to the toolbar."""
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = os.path.join(self.plugin_dir, 'icon.png')

        self.add_action(
            icon_path,
            text=self.tr('日本面積単位'),
            callback=self.run,
            parent=self.iface.mainWindow(),
            status_tip=self.tr('Convert between metric and Japanese area units')
        )

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr('&Japanese Area Unit'), action)
            self.iface.removeToolBarIcon(action)

        del self.toolbar

    def run(self):
        """Run method that performs all the real work."""
        if self.dialog is None:
            self.dialog = JapaneseAreaUnitDialog(self.iface.mainWindow(), self.iface)

        self.dialog.refresh_layers()
        self.dialog.show()
        self.dialog.raise_()
        self.dialog.activateWindow()
