# -*- coding: utf-8 -*-
"""
Japan Fude Polygon Loader - Main Dialog
メインダイアログ（地域選択・ダウンロード）
"""

from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QGroupBox, QLabel,
    QComboBox, QListWidget, QListWidgetItem, QPushButton,
    QProgressBar, QMessageBox, QCheckBox, QAbstractItemView,
    QFrame, QLineEdit, QInputDialog, QFileDialog
)
from qgis.PyQt.QtCore import Qt, QThread, pyqtSignal
from qgis.core import (
    QgsProject, QgsVectorLayer, QgsCoordinateReferenceSystem,
    QgsCoordinateTransform
)
import json
import urllib.request

from ..core import RegionMaster, DataDownloader, CacheManager, LayerStyler


class DownloadThread(QThread):
    """ダウンロード処理用スレッド"""

    progress = pyqtSignal(int, int)  # downloaded, total
    finished_one = pyqtSignal(str, str, str, str)  # pref_code, city_code, pref_name, city_name
    error = pyqtSignal(str, str, str)  # pref_code, city_code, error_message
    all_finished = pyqtSignal()

    def __init__(self, downloader, tasks, year, format_type):
        super().__init__()
        self.downloader = downloader
        self.tasks = tasks  # List of (pref_code, city_code, pref_name, city_name)
        self.year = year
        self.format_type = format_type
        self._is_cancelled = False

    def run(self):
        for pref_code, city_code, pref_name, city_name in self.tasks:
            if self._is_cancelled:
                break

            file_path, error_message = self.downloader.download(
                pref_code, city_code, self.year, self.format_type,
                progress_callback=lambda d, t: self.progress.emit(d, t)
            )

            if file_path:
                self.finished_one.emit(pref_code, city_code, pref_name, city_name)
            else:
                self.error.emit(pref_code, city_code, error_message or "ダウンロードに失敗しました")

        self.all_finished.emit()

    def cancel(self):
        self._is_cancelled = True


class MainDialog(QDialog):
    """メインダイアログ"""

    def __init__(self, iface, parent=None):
        super().__init__(parent)
        self.iface = iface
        self.region_master = RegionMaster()
        self.cache_manager = CacheManager()
        self.downloader = DataDownloader(self.cache_manager)
        self.layer_styler = LayerStyler()
        self.download_thread = None

        self.setup_ui()
        self.connect_signals()

    def setup_ui(self):
        """UIを構築"""
        self.setWindowTitle("Japan Fude Polygon Loader - 筆ポリゴン読込")
        self.setMinimumSize(450, 350)
        layout = QVBoxLayout(self)

        # 現在地から取得グループ（メイン機能）
        current_group = QGroupBox("現在の表示位置から筆ポリゴンを取得")
        current_layout = QVBoxLayout(current_group)

        self.current_location_label = QLabel("マップの中心座標から市区町村を特定し、筆ポリゴンを取得します")
        current_layout.addWidget(self.current_location_label)

        current_btn_layout = QHBoxLayout()
        self.btn_get_current = QPushButton("現在地の筆ポリゴンを取得")
        self.btn_get_current.setMinimumHeight(45)
        self.btn_get_current.setStyleSheet("font-weight: bold; font-size: 14px;")
        current_btn_layout.addWidget(self.btn_get_current)
        current_layout.addLayout(current_btn_layout)

        # 進捗表示
        self.progress_label_current = QLabel("")
        current_layout.addWidget(self.progress_label_current)

        layout.addWidget(current_group)

        # 区切り線
        line1 = QFrame()
        line1.setFrameShape(QFrame.HLine)
        line1.setFrameShadow(QFrame.Sunken)
        layout.addWidget(line1)

        # ローカルファイル読込グループ
        load_group = QGroupBox("ローカルファイルから読み込み")
        load_layout = QVBoxLayout(load_group)

        # ファイル読込ボタン
        self.btn_load_file = QPushButton("ZIPファイルを選択して読み込み")
        self.btn_load_file.setMinimumHeight(35)
        load_layout.addWidget(self.btn_load_file)

        self.load_status_label = QLabel("")
        load_layout.addWidget(self.load_status_label)

        layout.addWidget(load_group)

        # 地域選択グループ（自動ダウンロード非対応のため非表示）
        region_group = QGroupBox("地域を指定して取得")
        region_group.setVisible(False)  # 自動ダウンロードは機能しないため非表示
        region_layout = QVBoxLayout(region_group)

        # 都道府県選択
        pref_layout = QHBoxLayout()
        pref_layout.addWidget(QLabel("都道府県:"))
        self.pref_combo = QComboBox()
        self.pref_combo.setMinimumWidth(200)
        pref_layout.addWidget(self.pref_combo)
        pref_layout.addStretch()
        region_layout.addLayout(pref_layout)

        # 市区町村選択（複数選択可）
        region_layout.addWidget(QLabel("市区町村（複数選択可）:"))
        self.city_list = QListWidget()
        self.city_list.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.city_list.setMinimumHeight(200)
        region_layout.addWidget(self.city_list)

        # 選択ボタン
        select_btn_layout = QHBoxLayout()
        self.btn_select_all = QPushButton("全選択")
        self.btn_deselect_all = QPushButton("選択解除")
        self.btn_manual_code = QPushButton("コード入力")
        self.btn_manual_code.setToolTip("市区町村コード（5桁）を直接入力")
        select_btn_layout.addWidget(self.btn_select_all)
        select_btn_layout.addWidget(self.btn_deselect_all)
        select_btn_layout.addWidget(self.btn_manual_code)
        select_btn_layout.addStretch()
        region_layout.addLayout(select_btn_layout)

        layout.addWidget(region_group)

        # オプショングループ（自動ダウンロード非対応のため非表示）
        option_group = QGroupBox("オプション")
        option_group.setVisible(False)
        option_layout = QVBoxLayout(option_group)

        # 年度選択
        year_layout = QHBoxLayout()
        year_layout.addWidget(QLabel("データ年度:"))
        self.year_combo = QComboBox()
        self.year_combo.addItems(["2024", "2023", "2022"])
        year_layout.addWidget(self.year_combo)
        year_layout.addStretch()
        option_layout.addLayout(year_layout)

        # データ形式選択
        format_layout = QHBoxLayout()
        format_layout.addWidget(QLabel("データ形式:"))
        self.format_combo = QComboBox()
        self.format_combo.addItem("GeoJSON", "geojson")
        self.format_combo.addItem("FlatGeobuf", "fgb")
        format_layout.addWidget(self.format_combo)
        format_layout.addStretch()
        option_layout.addLayout(format_layout)

        # キャッシュ使用
        self.cb_use_cache = QCheckBox("キャッシュを使用（ダウンロード済みデータを再利用）")
        self.cb_use_cache.setChecked(True)
        option_layout.addWidget(self.cb_use_cache)

        layout.addWidget(option_group)

        # 進捗表示（自動ダウンロード非対応のため非表示）
        progress_group = QGroupBox("進捗")
        progress_group.setVisible(False)
        progress_layout = QVBoxLayout(progress_group)
        self.progress_label = QLabel("待機中")
        progress_layout.addWidget(self.progress_label)
        self.progress_bar = QProgressBar()
        self.progress_bar.setRange(0, 100)
        self.progress_bar.setValue(0)
        progress_layout.addWidget(self.progress_bar)
        layout.addWidget(progress_group)

        # ボタン（非表示の互換性用）
        btn_layout = QHBoxLayout()
        self.btn_download = QPushButton("ダウンロード")
        self.btn_download.setVisible(False)  # 機能しないため非表示
        self.btn_load_local = QPushButton("ファイル読込")
        self.btn_load_local.setVisible(False)  # 上部に移動したため非表示
        self.btn_cancel = QPushButton("キャンセル")
        self.btn_cancel.setVisible(False)
        self.btn_close = QPushButton("閉じる")
        self.btn_close.setMinimumHeight(35)
        btn_layout.addStretch()
        btn_layout.addWidget(self.btn_download)
        btn_layout.addWidget(self.btn_load_local)
        btn_layout.addWidget(self.btn_cancel)
        btn_layout.addWidget(self.btn_close)
        layout.addLayout(btn_layout)

        # キャッシュ情報（自動ダウンロード非対応のため非表示）
        cache_layout = QHBoxLayout()
        self.cache_info_label = QLabel("")
        self.cache_info_label.setVisible(False)
        self.btn_clear_cache = QPushButton("キャッシュクリア")
        self.btn_clear_cache.setVisible(False)
        cache_layout.addWidget(self.cache_info_label)
        cache_layout.addStretch()
        cache_layout.addWidget(self.btn_clear_cache)
        layout.addLayout(cache_layout)

    def connect_signals(self):
        """シグナルを接続"""
        self.btn_get_current.clicked.connect(self.download_current_location)
        self.btn_load_file.clicked.connect(self.load_local_file)
        self.pref_combo.currentIndexChanged.connect(self.on_prefecture_changed)
        self.btn_select_all.clicked.connect(self.select_all_cities)
        self.btn_deselect_all.clicked.connect(self.deselect_all_cities)
        self.btn_manual_code.clicked.connect(self.add_manual_city_code)
        self.btn_download.clicked.connect(self.start_download)
        self.btn_load_local.clicked.connect(self.load_local_file)
        self.btn_cancel.clicked.connect(self.cancel_download)
        self.btn_close.clicked.connect(self.close)
        self.btn_clear_cache.clicked.connect(self.clear_cache)

    def load_prefectures(self):
        """都道府県一覧を読み込み"""
        self.pref_combo.clear()
        self.pref_combo.addItem("-- 選択してください --", "")
        for code, name in self.region_master.get_prefectures():
            self.pref_combo.addItem(f"{name} ({code})", code)

    def download_current_location(self):
        """現在の表示位置から筆ポリゴンをダウンロードして読み込み"""
        canvas = self.iface.mapCanvas()
        center = canvas.center()
        extent = canvas.extent()
        crs = canvas.mapSettings().destinationCrs()

        # WGS84に変換
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        if crs.authid() != "EPSG:4326":
            transform = QgsCoordinateTransform(crs, wgs84, QgsProject.instance())
            center = transform.transform(center)
            extent = transform.transformBoundingBox(extent)

        lat = center.y()
        lon = center.x()

        self.current_location_label.setText(f"座標: {lat:.6f}, {lon:.6f}")
        self.progress_label_current.setText("市区町村を検索中...")
        self.repaint()

        # 国土地理院APIで市区町村を特定
        city_info = self._reverse_geocode(lat, lon)

        if not city_info:
            self.progress_label_current.setText("この地域にはデータがありません")
            QMessageBox.warning(self, "データなし",
                "この地域には筆ポリゴンデータがありません。\n\n"
                "筆ポリゴンは日本国内の農地のみを対象としています。\n"
                "日本国内の農地がある地域を表示してください。")
            return

        pref_code = city_info['pref_code']
        city_code = city_info['city_code']
        pref_name = city_info['pref_name']
        city_name = city_info['city_name']

        self.current_location_label.setText(f"{pref_name} {city_name} ({city_code})")
        self.progress_label_current.setText("筆ポリゴンをダウンロード中...")
        self.btn_get_current.setEnabled(False)
        self.repaint()

        # ダウンロード実行
        def progress_callback(downloaded, total):
            if total > 0:
                percent = int(downloaded / total * 100)
                self.progress_label_current.setText(f"ダウンロード中... {percent}%")
                self.repaint()

        file_path, error = self.downloader.download(
            pref_code, city_code, "2025", "fgb",
            progress_callback=progress_callback
        )

        if not file_path:
            self.progress_label_current.setText("この地域にはデータがありません")
            self.btn_get_current.setEnabled(True)

            # 404エラーの場合はデータなしメッセージ
            if "見つかりません" in error or "404" in error:
                QMessageBox.information(self, "データなし",
                    f"この地域（{pref_name}）には筆ポリゴンデータが公開されていません。")
            else:
                QMessageBox.warning(self, "ダウンロードエラー", f"筆ポリゴンのダウンロードに失敗しました。\n\n{error}")
            return

        self.progress_label_current.setText(f"レイヤを読み込み中（{city_name}のみ）...")
        self.repaint()

        # レイヤを読み込み（市区町村コードでフィルタリング）
        # 実際にダウンロードされた年度を使用
        actual_year = self.downloader.actual_year or "2025"
        self._load_fgb_layer_with_extent(file_path, pref_code, city_code, pref_name, city_name, extent, actual_year)

        self.progress_label_current.setText("読込完了")
        self.btn_get_current.setEnabled(True)

    def _load_fgb_layer_with_extent(self, file_path, pref_code, city_code, pref_name, city_name, extent, year="2025"):
        """FlatGeobufファイルをレイヤとして読み込み（表示範囲でフィルタリング）"""
        layer_name = f"筆ポリゴン_{city_name}_{year}"

        # レイヤを作成
        layer = QgsVectorLayer(file_path, layer_name, "ogr")

        if not layer.isValid():
            QMessageBox.warning(self, "エラー", "レイヤの読み込みに失敗しました")
            return

        fields = layer.fields()

        # CRSを設定（EPSG:6668 - JGD2011）
        crs = QgsCoordinateReferenceSystem("EPSG:6668")
        layer.setCrs(crs)

        # point_lng, point_lat フィールドで空間フィルタリング
        filter_applied = False

        # バッファを追加（表示範囲より少し広め）
        buffer = 0.005  # 約500m
        xmin = extent.xMinimum() - buffer
        xmax = extent.xMaximum() + buffer
        ymin = extent.yMinimum() - buffer
        ymax = extent.yMaximum() + buffer

        if fields.indexOf('point_lng') >= 0 and fields.indexOf('point_lat') >= 0:
            # point_lng, point_lat で範囲フィルタ
            filter_str = (
                f'"point_lng" >= {xmin} AND "point_lng" <= {xmax} AND '
                f'"point_lat" >= {ymin} AND "point_lat" <= {ymax}'
            )
            layer.setSubsetString(filter_str)
            count = layer.featureCount()

            if count > 0:
                filter_applied = True
            else:
                layer.setSubsetString("")

        if not filter_applied:
            QMessageBox.information(self, "データなし",
                "この表示範囲には筆ポリゴンデータがありません。\n\n"
                "農地がある地域を表示してください。")
            return

        feature_count = layer.featureCount()

        if feature_count == 0:
            QMessageBox.information(self, "データなし",
                "この表示範囲には筆ポリゴンデータがありません。\n\n"
                "農地がある地域を表示してください。")
            return

        if feature_count > 10000:
            reply = QMessageBox.question(self, "確認",
                f"フィーチャ数が {feature_count:,} 件あります。\n"
                "読み込みに時間がかかる場合があります。\n\n"
                "表示範囲を狭くすると軽くなります。\n続行しますか？",
                QMessageBox.Yes | QMessageBox.No)
            if reply == QMessageBox.No:
                return

        # スタイルを適用
        self.layer_styler.apply_style(layer)

        # プロジェクトに追加
        QgsProject.instance().addMapLayer(layer)

        # 表示範囲は変更しない（現在の範囲を維持）
        self.iface.mapCanvas().refresh()

        QMessageBox.information(self, "完了",
            f"筆ポリゴンを読み込みました。\n\n"
            f"レイヤ名: {layer_name}\n"
            f"データ年度: {year}年\n"
            f"フィーチャ数: {feature_count:,}\n\n"
            f"表示範囲のデータのみ読み込まれています。")

    def _load_fgb_layer(self, file_path, pref_code, city_code, pref_name, city_name, year="2025"):
        """FlatGeobufファイルをレイヤとして読み込み（後方互換用）"""
        from qgis.core import QgsRectangle
        # 現在の表示範囲を取得
        extent = self.iface.mapCanvas().extent()
        self._load_fgb_layer_with_extent(file_path, pref_code, city_code, pref_name, city_name, extent, year)

    def identify_current_location(self):
        """互換性のため残す"""
        self.download_current_location()

    def get_current_location(self):
        """互換性のため残す"""
        self.download_current_location()

    def on_current_location_finished(self, pref_code, city_code, pref_name, city_name):
        """現在地のダウンロードが完了"""
        self.progress_bar.setValue(1)
        self.current_location_label.setText(f"{pref_name} {city_name} の筆ポリゴンを読込完了")

        # レイヤとして追加
        self.load_layer(pref_code, city_code, pref_name, city_name)

        # 表示範囲を維持（ズームしない）
        self.iface.mapCanvas().refresh()

    def _reverse_geocode(self, lat, lon):
        """
        国土地理院APIで逆ジオコーディング

        Args:
            lat: 緯度
            lon: 経度

        Returns:
            dict with pref_code, city_code, pref_name, city_name or None
        """
        url = f"https://mreversegeocoder.gsi.go.jp/reverse-geocoder/LonLatToAddress?lat={lat}&lon={lon}"

        try:
            with urllib.request.urlopen(url, timeout=10) as response:
                data = json.loads(response.read().decode('utf-8'))

            results = data.get('results')
            if not results:
                return None

            muni_cd = results.get('muniCd', '')
            if not muni_cd or len(muni_cd) < 5:
                return None

            # 市区町村コードから都道府県コード・市区町村コードを分離
            pref_code = muni_cd[:2]
            city_code = muni_cd[:5]

            # 都道府県名を取得
            pref_name = self.region_master.get_prefecture_name(pref_code)

            # 市区町村名を取得（APIレスポンスから）
            city_name = results.get('lv01Nm', '')

            return {
                'pref_code': pref_code,
                'city_code': city_code,
                'pref_name': pref_name,
                'city_name': city_name
            }

        except Exception:
            return None

    def on_prefecture_changed(self, index):
        """都道府県が変更された時の処理"""
        self.city_list.clear()
        pref_code = self.pref_combo.currentData()
        if not pref_code:
            return

        self.progress_label.setText("市区町村一覧を取得中...")
        self.repaint()

        # 市区町村一覧を取得
        cities = self.region_master.get_cities(pref_code)

        for code, name in cities:
            item = QListWidgetItem(f"{name} ({code})")
            item.setData(Qt.UserRole, code)
            item.setData(Qt.UserRole + 1, name)
            self.city_list.addItem(item)

        if len(cities) == 0:
            self.progress_label.setText("市区町村リストを取得できません。「コード入力」で直接入力するか、「現在地から取得」をご利用ください。")
        else:
            self.progress_label.setText(f"{len(cities)} 市区町村")

    def add_manual_city_code(self):
        """市区町村コードを手動で追加"""
        pref_code = self.pref_combo.currentData()
        if not pref_code:
            QMessageBox.warning(self, "選択エラー", "先に都道府県を選択してください。")
            return

        code, ok = QInputDialog.getText(
            self, "市区町村コード入力",
            "市区町村コード（5桁）を入力してください:\n例: 01100（札幌市）、13101（千代田区）",
            text=pref_code
        )

        if ok and code:
            code = code.strip()
            if len(code) != 5 or not code.isdigit():
                QMessageBox.warning(self, "入力エラー", "5桁の数字を入力してください。")
                return

            if not code.startswith(pref_code):
                QMessageBox.warning(self, "入力エラー", f"コードは選択した都道府県（{pref_code}）で始まる必要があります。")
                return

            # 既存の項目をチェック
            for i in range(self.city_list.count()):
                if self.city_list.item(i).data(Qt.UserRole) == code:
                    self.city_list.item(i).setSelected(True)
                    return

            # 新規追加
            item = QListWidgetItem(f"手動入力 ({code})")
            item.setData(Qt.UserRole, code)
            item.setData(Qt.UserRole + 1, f"コード{code}")
            self.city_list.addItem(item)
            item.setSelected(True)

    def select_all_cities(self):
        """全市区町村を選択"""
        self.city_list.selectAll()

    def deselect_all_cities(self):
        """選択を解除"""
        self.city_list.clearSelection()

    def get_selected_cities(self):
        """選択された市区町村を取得"""
        selected = []
        pref_code = self.pref_combo.currentData()
        pref_name = self.region_master.get_prefecture_name(pref_code)

        for item in self.city_list.selectedItems():
            city_code = item.data(Qt.UserRole)
            city_name = item.data(Qt.UserRole + 1)
            selected.append((pref_code, city_code, pref_name, city_name))

        return selected

    def start_download(self):
        """ダウンロードを開始"""
        tasks = self.get_selected_cities()
        if not tasks:
            QMessageBox.warning(self, "選択エラー", "市区町村を選択してください。")
            return

        year = self.year_combo.currentText()
        format_type = self.format_combo.currentData()

        # UI状態を更新
        self.btn_download.setEnabled(False)
        self.btn_cancel.setEnabled(True)
        self.progress_bar.setRange(0, len(tasks))
        self.progress_bar.setValue(0)

        # ダウンロードスレッドを開始
        self.download_thread = DownloadThread(
            self.downloader, tasks, year, format_type
        )
        self.download_thread.progress.connect(self.on_download_progress)
        self.download_thread.finished_one.connect(self.on_download_finished_one)
        self.download_thread.error.connect(self.on_download_error)
        self.download_thread.all_finished.connect(self.on_all_downloads_finished)
        self.download_thread.start()

    def on_download_progress(self, downloaded, total):
        """ダウンロード進捗を更新"""
        if total > 0:
            percent = int(downloaded / total * 100)
            self.progress_label.setText(f"ダウンロード中... {downloaded:,} / {total:,} bytes ({percent}%)")

    def on_download_finished_one(self, pref_code, city_code, pref_name, city_name):
        """1つのダウンロードが完了"""
        self.progress_bar.setValue(self.progress_bar.value() + 1)
        self.progress_label.setText(f"読込中: {pref_name} {city_name}")

        # レイヤとして追加
        self.load_layer(pref_code, city_code, pref_name, city_name)

    def on_download_error(self, pref_code, city_code, error_message):
        """ダウンロードエラー"""
        self.progress_bar.setValue(self.progress_bar.value() + 1)
        self.progress_label.setText(f"エラー: {city_code}")
        self.current_location_label.setText(f"エラー: {error_message}")

        # ネットワークエラーの場合は代替手段を案内
        help_text = ""
        if "getaddrinfo" in error_message or "ネットワーク" in error_message:
            help_text = (
                "\n\n【代替手段】\n"
                "1. 筆ポリゴン公開サイト(https://download.fude.maff.go.jp/)から\n"
                "   手動でZIPファイルをダウンロード\n"
                "2. 「ローカルファイル読込」ボタンで読み込み"
            )

        # エラーメッセージを表示
        QMessageBox.warning(
            self, "ダウンロードエラー",
            f"市区町村コード {city_code} のデータ取得に失敗しました。\n\n{error_message}{help_text}"
        )

    def on_all_downloads_finished(self):
        """全てのダウンロードが完了"""
        self.btn_get_current.setEnabled(True)
        self.btn_download.setEnabled(True)
        self.btn_cancel.setEnabled(False)
        self.progress_label.setText("完了")
        self.update_cache_info()

        # 表示範囲をレイヤに合わせる
        self.iface.mapCanvas().refresh()

    def cancel_download(self):
        """ダウンロードをキャンセル"""
        if self.download_thread:
            self.download_thread.cancel()
            self.progress_label.setText("キャンセル中...")

    def load_layer(self, pref_code, city_code, pref_name, city_name):
        """レイヤをQGISに追加"""
        year = self.year_combo.currentText()
        format_type = self.format_combo.currentData()

        # キャッシュからファイルパスを取得
        file_path = self.cache_manager.get_cached_file(
            pref_code, city_code, year, format_type
        )

        if not file_path:
            return

        # レイヤを作成
        layer_name = f"筆ポリゴン_{pref_name}_{city_name}_{year}"
        layer = QgsVectorLayer(file_path, layer_name, "ogr")

        if not layer.isValid():
            QMessageBox.warning(self, "エラー", f"レイヤの読み込みに失敗: {city_name}")
            return

        # CRSを設定（EPSG:6668 - JGD2011）
        crs = QgsCoordinateReferenceSystem("EPSG:6668")
        layer.setCrs(crs)

        # スタイルを適用
        self.layer_styler.apply_style(layer)

        # プロジェクトに追加
        QgsProject.instance().addMapLayer(layer)

    def update_cache_info(self):
        """キャッシュ情報を更新"""
        count = self.cache_manager.get_cache_count()
        size = self.cache_manager.get_cache_size()
        size_mb = size / (1024 * 1024)
        self.cache_info_label.setText(f"キャッシュ: {count} ファイル ({size_mb:.1f} MB)")

    def load_local_file(self):
        """ローカルファイルから筆ポリゴンを読み込み"""
        file_path, _ = QFileDialog.getOpenFileName(
            self,
            "筆ポリゴンファイルを選択",
            "",
            "GeoJSON/ZIP Files (*.geojson *.json *.zip *.fgb);;All Files (*)"
        )

        if not file_path:
            return

        import os
        import zipfile
        import tempfile

        try:
            actual_file = file_path

            # ZIPファイルの場合は解凍
            if file_path.lower().endswith('.zip'):
                self.load_status_label.setText("ZIPファイルを解凍中...")
                self.repaint()

                with zipfile.ZipFile(file_path, 'r') as zf:
                    file_list = zf.namelist()

                    # GeoJSONまたはFGBファイルを探す
                    target_file = None
                    for name in file_list:
                        if name.lower().endswith(('.geojson', '.json', '.fgb')):
                            target_file = name
                            break

                    if not target_file:
                        self.load_status_label.setText("エラー: GeoJSONファイルが見つかりません")
                        QMessageBox.warning(self, "エラー", "ZIPファイル内にGeoJSON/FGBファイルが見つかりません。")
                        return

                    # 一時ディレクトリに解凍
                    temp_dir = tempfile.mkdtemp()
                    zf.extract(target_file, temp_dir)
                    actual_file = os.path.join(temp_dir, target_file)

            # ファイル名から情報を推測
            base_name = os.path.basename(actual_file)
            layer_name = f"筆ポリゴン_{os.path.splitext(base_name)[0]}"

            self.load_status_label.setText(f"レイヤを読み込み中: {base_name}")
            self.repaint()

            # レイヤを作成
            layer = QgsVectorLayer(actual_file, layer_name, "ogr")

            if not layer.isValid():
                self.load_status_label.setText(f"エラー: 読み込み失敗")
                QMessageBox.warning(self, "エラー", f"ファイルの読み込みに失敗しました: {base_name}")
                return

            # CRSを設定（EPSG:6668 - JGD2011）
            crs = QgsCoordinateReferenceSystem("EPSG:6668")
            layer.setCrs(crs)

            # スタイルを適用
            self.layer_styler.apply_style(layer)

            # プロジェクトに追加
            QgsProject.instance().addMapLayer(layer)

            self.load_status_label.setText(f"読込完了: {layer_name}")
            self.iface.mapCanvas().refresh()

            QMessageBox.information(self, "完了", f"筆ポリゴンを読み込みました。\n\nレイヤ名: {layer_name}")

        except zipfile.BadZipFile:
            self.load_status_label.setText("エラー: 無効なZIPファイル")
            QMessageBox.warning(self, "エラー", "無効なZIPファイルです。")
        except Exception as e:
            self.load_status_label.setText(f"エラー: {str(e)[:30]}...")
            QMessageBox.warning(self, "エラー", f"ファイルの読み込み中にエラーが発生しました:\n{str(e)}")

    def clear_cache(self):
        """キャッシュをクリア"""
        reply = QMessageBox.question(
            self, "確認",
            "全てのキャッシュを削除しますか？",
            QMessageBox.Yes | QMessageBox.No
        )
        if reply == QMessageBox.Yes:
            self.cache_manager.clear_all_cache()
            self.update_cache_info()
            QMessageBox.information(self, "完了", "キャッシュを削除しました。")

    def closeEvent(self, event):
        """ダイアログを閉じる時の処理"""
        if self.download_thread and self.download_thread.isRunning():
            self.download_thread.cancel()
            self.download_thread.wait()
        super().closeEvent(event)
