# -*- coding: utf-8 -*-
"""
Japan Fude Polygon Loader - Cache Manager
ダウンロードデータのキャッシュ管理
"""

import os
import json
from typing import Optional, Dict
from pathlib import Path


class CacheManager:
    """ダウンロードデータのキャッシュ管理クラス"""

    CACHE_DIR_NAME = "japan_fude_polygon_loader"
    CACHE_INDEX_FILE = "cache_index.json"

    def __init__(self, base_dir: Optional[str] = None):
        """
        Initialize CacheManager.

        Args:
            base_dir: Base directory for cache. If None, uses user's cache directory.
        """
        if base_dir:
            self.base_dir = base_dir
        else:
            # デフォルトはQGISのプラグインデータディレクトリ
            self.base_dir = self._get_default_cache_dir()

        self._cache_index: Dict = {}
        self._load_cache_index()

    def _get_default_cache_dir(self) -> str:
        """デフォルトのキャッシュディレクトリを取得"""
        # QGISの設定ディレクトリを使用
        try:
            from qgis.core import QgsApplication
            qgis_dir = QgsApplication.qgisSettingsDirPath()
            return os.path.join(qgis_dir, "cache", self.CACHE_DIR_NAME)
        except ImportError:
            # QGIS外で実行された場合
            home = Path.home()
            return os.path.join(str(home), ".cache", self.CACHE_DIR_NAME)

    def _get_cache_index_path(self) -> str:
        """キャッシュインデックスファイルのパスを取得"""
        return os.path.join(self.base_dir, self.CACHE_INDEX_FILE)

    def _load_cache_index(self):
        """キャッシュインデックスを読み込み"""
        index_path = self._get_cache_index_path()
        if os.path.exists(index_path):
            try:
                with open(index_path, 'r', encoding='utf-8') as f:
                    self._cache_index = json.load(f)
            except Exception:
                self._cache_index = {}
        else:
            self._cache_index = {}

    def _save_cache_index(self):
        """キャッシュインデックスを保存"""
        os.makedirs(self.base_dir, exist_ok=True)
        index_path = self._get_cache_index_path()
        try:
            with open(index_path, 'w', encoding='utf-8') as f:
                json.dump(self._cache_index, f, ensure_ascii=False, indent=2)
        except Exception:
            pass

    def _make_cache_key(self, pref_code: str, city_code: str, year: str,
                        format_type: str) -> str:
        """キャッシュキーを生成"""
        return f"{pref_code}_{city_code}_{year}_{format_type}"

    def get_cache_dir(self, pref_code: str, city_code: str, year: str) -> str:
        """
        キャッシュディレクトリのパスを取得

        Args:
            pref_code: 都道府県コード
            city_code: 市区町村コード
            year: データ年度

        Returns:
            Cache directory path
        """
        return os.path.join(self.base_dir, year, pref_code, city_code)

    def get_cached_file(self, pref_code: str, city_code: str, year: str,
                        format_type: str) -> Optional[str]:
        """
        キャッシュされたファイルのパスを取得

        Args:
            pref_code: 都道府県コード
            city_code: 市区町村コード
            year: データ年度
            format_type: データ形式

        Returns:
            Cached file path or None if not cached
        """
        key = self._make_cache_key(pref_code, city_code, year, format_type)
        if key in self._cache_index:
            path = self._cache_index[key].get('path')
            if path and os.path.exists(path):
                return path
        return None

    def is_cached(self, pref_code: str, city_code: str, year: str,
                  format_type: str) -> bool:
        """
        データがキャッシュされているか確認

        Args:
            pref_code: 都道府県コード
            city_code: 市区町村コード
            year: データ年度
            format_type: データ形式

        Returns:
            True if cached, False otherwise
        """
        return self.get_cached_file(pref_code, city_code, year, format_type) is not None

    def register_cache(self, pref_code: str, city_code: str, year: str,
                       format_type: str, file_path: str):
        """
        キャッシュを登録

        Args:
            pref_code: 都道府県コード
            city_code: 市区町村コード
            year: データ年度
            format_type: データ形式
            file_path: ファイルパス
        """
        key = self._make_cache_key(pref_code, city_code, year, format_type)
        self._cache_index[key] = {
            'pref_code': pref_code,
            'city_code': city_code,
            'year': year,
            'format_type': format_type,
            'path': file_path
        }
        self._save_cache_index()

    def remove_cache(self, pref_code: str, city_code: str, year: str,
                     format_type: str) -> bool:
        """
        キャッシュを削除

        Args:
            pref_code: 都道府県コード
            city_code: 市区町村コード
            year: データ年度
            format_type: データ形式

        Returns:
            True if removed, False otherwise
        """
        key = self._make_cache_key(pref_code, city_code, year, format_type)
        if key in self._cache_index:
            path = self._cache_index[key].get('path')
            if path and os.path.exists(path):
                try:
                    os.remove(path)
                except Exception:
                    pass
            del self._cache_index[key]
            self._save_cache_index()
            return True
        return False

    def clear_all_cache(self):
        """全キャッシュを削除"""
        import shutil
        try:
            if os.path.exists(self.base_dir):
                shutil.rmtree(self.base_dir)
        except Exception:
            pass
        self._cache_index = {}

    def get_cache_size(self) -> int:
        """
        キャッシュの合計サイズを取得（バイト）

        Returns:
            Total cache size in bytes
        """
        total_size = 0
        for entry in self._cache_index.values():
            path = entry.get('path')
            if path and os.path.exists(path):
                total_size += os.path.getsize(path)
        return total_size

    def get_cache_count(self) -> int:
        """
        キャッシュされているファイル数を取得

        Returns:
            Number of cached files
        """
        return len(self._cache_index)
