# -*- coding: utf-8 -*-
"""
Japan Fude Polygon Loader - Main Plugin Module

This module provides the main plugin class for QGIS integration.
It handles toolbar creation, menu registration, and dialog management.

Copyright (C) 2026 Link Field
License: GPL-3.0
"""

import os
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtCore import Qt, QSettings, QTranslator, QCoreApplication


class JapanFudePolygonLoader:
    """
    Main plugin class for Japan Fude Polygon Loader.

    This plugin downloads and loads MAFF Fude Polygon (farmland parcel)
    data directly into QGIS.
    """

    def __init__(self, iface):
        """
        Initialize the plugin.

        :param iface: A QGIS interface instance.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = "&Japan Fude Polygon Loader"
        self.toolbar = self.iface.addToolBar("Japan Fude Polygon Loader")
        self.toolbar.setObjectName("JapanFudePolygonLoader")
        self.toolbar.setToolButtonStyle(Qt.ToolButtonTextOnly)
        self.main_dialog = None

        # Initialize locale
        locale_setting = QSettings().value('locale/userLocale')
        if locale_setting:
            locale = locale_setting[0:2]
        else:
            locale = 'en'
        locale_path = os.path.join(self.plugin_dir, 'i18n', f'{locale}.qm')

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        return QCoreApplication.translate('JapanFudePolygonLoader', message)

    def initGui(self):
        """
        Initialize the GUI elements.

        Creates toolbar buttons and menu entries for the plugin.
        """
        # Main action - download and load fude polygon
        self.action_main = QAction("筆ポリゴン読込", self.iface.mainWindow())
        self.action_main.triggered.connect(self.run)
        self.action_main.setStatusTip("筆ポリゴンデータをダウンロード・読込")
        self.toolbar.addAction(self.action_main)
        self.iface.addPluginToMenu(self.menu, self.action_main)
        self.actions.append(self.action_main)

    def unload(self):
        """
        Remove the plugin menu items and toolbar.

        Called when the plugin is unloaded from QGIS.
        """
        for action in self.actions:
            self.iface.removePluginMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)

        del self.toolbar

        # Clean up dialogs
        if self.main_dialog:
            self.main_dialog.close()
            self.main_dialog = None

    def run(self):
        """
        Show the main dialog for downloading fude polygon data.

        Creates the dialog if it doesn't exist, then shows it.
        """
        from .dialogs import MainDialog
        if self.main_dialog is None:
            self.main_dialog = MainDialog(self.iface, self.iface.mainWindow())
        self.main_dialog.show()
        self.main_dialog.raise_()
        self.main_dialog.activateWindow()
