# -*- coding: utf-8 -*-
"""
Japan Fude Polygon Loader - Region Master Data
都道府県・市区町村マスターデータ
"""

from typing import Dict, List, Tuple


class RegionMaster:
    """都道府県・市区町村マスターデータ管理クラス"""

    # 都道府県コード・名称
    PREFECTURES: Dict[str, str] = {
        "01": "北海道", "02": "青森県", "03": "岩手県", "04": "宮城県",
        "05": "秋田県", "06": "山形県", "07": "福島県", "08": "茨城県",
        "09": "栃木県", "10": "群馬県", "11": "埼玉県", "12": "千葉県",
        "13": "東京都", "14": "神奈川県", "15": "新潟県", "16": "富山県",
        "17": "石川県", "18": "福井県", "19": "山梨県", "20": "長野県",
        "21": "岐阜県", "22": "静岡県", "23": "愛知県", "24": "三重県",
        "25": "滋賀県", "26": "京都府", "27": "大阪府", "28": "兵庫県",
        "29": "奈良県", "30": "和歌山県", "31": "鳥取県", "32": "島根県",
        "33": "岡山県", "34": "広島県", "35": "山口県", "36": "徳島県",
        "37": "香川県", "38": "愛媛県", "39": "高知県", "40": "福岡県",
        "41": "佐賀県", "42": "長崎県", "43": "熊本県", "44": "大分県",
        "45": "宮崎県", "46": "鹿児島県", "47": "沖縄県",
    }

    def __init__(self):
        self._cities_cache: Dict[str, List[Tuple[str, str]]] = {}

    def get_prefectures(self) -> List[Tuple[str, str]]:
        """
        都道府県一覧を取得

        Returns:
            List of (code, name) tuples
        """
        return [(code, name) for code, name in sorted(self.PREFECTURES.items())]

    def get_prefecture_name(self, code: str) -> str:
        """都道府県コードから名称を取得"""
        return self.PREFECTURES.get(code, "")

    def get_cities(self, pref_code: str) -> List[Tuple[str, str]]:
        """
        市区町村一覧を取得（筆ポリゴンサイトから動的取得）

        Args:
            pref_code: 都道府県コード（2桁）

        Returns:
            List of (code, name) tuples
        """
        # キャッシュに有効なデータがある場合のみ使用
        if pref_code in self._cities_cache and self._cities_cache[pref_code]:
            return self._cities_cache[pref_code]

        # 市区町村一覧を取得
        cities = self._fetch_cities_from_site(pref_code)

        # 有効なデータのみキャッシュ
        if cities:
            self._cities_cache[pref_code] = cities

        return cities

    def _fetch_cities_from_site(self, pref_code: str) -> List[Tuple[str, str]]:
        """
        市区町村一覧を取得（国土交通省APIを使用）

        Args:
            pref_code: 都道府県コード（2桁）

        Returns:
            List of (code, name) tuples
        """
        import urllib.request
        import json

        pref_name = self.PREFECTURES.get(pref_code, '')
        if not pref_name:
            return []

        # 国土交通省 土地総合情報システムAPIを使用
        url = f"https://www.land.mlit.go.jp/webland/api/CitySearch?area={pref_code}"

        try:
            request = urllib.request.Request(url, headers={'User-Agent': 'QGIS Plugin'})
            with urllib.request.urlopen(request, timeout=15) as response:
                data = json.loads(response.read().decode('utf-8'))

            cities = []
            status = data.get('status', '')

            if status == 'OK':
                for city in data.get('data', []):
                    city_code = city.get('id', '')
                    city_name = city.get('name', '')
                    if city_code and city_name:
                        cities.append((city_code, city_name))

            if cities:
                return sorted(cities, key=lambda x: x[0])

        except Exception:

        # APIが失敗した場合は静的リストを返す
        return self._fetch_cities_fallback(pref_code)

    def _get_city_name_from_code(self, city_code: str) -> str:
        """市区町村コードから名称を取得（簡易版）"""
        return f"市区町村{city_code}"

    def _fetch_cities_fallback(self, pref_code: str) -> List[Tuple[str, str]]:
        """最終フォールバック: 主要都市の静的リスト"""
        # 各都道府県の県庁所在地を最低限提供
        CAPITAL_CITIES = {
            "01": [("01100", "札幌市")],
            "02": [("02201", "青森市")],
            "03": [("03201", "盛岡市")],
            "04": [("04100", "仙台市")],
            "05": [("05201", "秋田市")],
            "06": [("06201", "山形市")],
            "07": [("07201", "福島市")],
            "08": [("08201", "水戸市")],
            "09": [("09201", "宇都宮市")],
            "10": [("10201", "前橋市")],
            "11": [("11100", "さいたま市")],
            "12": [("12100", "千葉市")],
            "13": [("13101", "千代田区"), ("13102", "中央区"), ("13103", "港区"), ("13104", "新宿区")],
            "14": [("14100", "横浜市"), ("14130", "川崎市")],
            "15": [("15100", "新潟市")],
            "16": [("16201", "富山市")],
            "17": [("17201", "金沢市")],
            "18": [("18201", "福井市")],
            "19": [("19201", "甲府市")],
            "20": [("20201", "長野市")],
            "21": [("21201", "岐阜市")],
            "22": [("22100", "静岡市"), ("22130", "浜松市")],
            "23": [("23100", "名古屋市")],
            "24": [("24201", "津市")],
            "25": [("25201", "大津市")],
            "26": [("26100", "京都市")],
            "27": [("27100", "大阪市"), ("27140", "堺市")],
            "28": [("28100", "神戸市")],
            "29": [("29201", "奈良市")],
            "30": [("30201", "和歌山市")],
            "31": [("31201", "鳥取市")],
            "32": [("32201", "松江市")],
            "33": [("33100", "岡山市")],
            "34": [("34100", "広島市")],
            "35": [("35203", "山口市")],
            "36": [("36201", "徳島市")],
            "37": [("37201", "高松市")],
            "38": [("38201", "松山市")],
            "39": [("39201", "高知市")],
            "40": [("40100", "福岡市"), ("40130", "北九州市")],
            "41": [("41201", "佐賀市")],
            "42": [("42201", "長崎市")],
            "43": [("43100", "熊本市")],
            "44": [("44201", "大分市")],
            "45": [("45201", "宮崎市")],
            "46": [("46201", "鹿児島市")],
            "47": [("47201", "那覇市")],
        }
        return CAPITAL_CITIES.get(pref_code, [])

    def get_download_url(self, pref_code: str, city_code: str, year: str = "2025",
                         format_type: str = "fgb") -> str:
        """
        ダウンロードURLを生成（都道府県単位のFlatGeobuf形式）

        Args:
            pref_code: 都道府県コード（2桁）
            city_code: 市区町村コード（5桁）- 未使用（都道府県単位）
            year: データ年度
            format_type: データ形式（fgb推奨）

        Returns:
            Download URL
        """
        # 農林水産省の集落データサイトからFlatGeobuf形式でダウンロード
        return f"https://www.machimura.maff.go.jp/shurakudata/2020/mb/MB0001_{year}_2020_{pref_code}.zip"

    def clear_cache(self):
        """市区町村キャッシュをクリア"""
        self._cities_cache.clear()
