# -*- coding: utf-8 -*-
"""
Japan Fude Polygon Loader - Layer Styler
レイヤスタイル設定（耕地種別色分け）
"""

from qgis.core import (
    QgsVectorLayer, QgsCategorizedSymbolRenderer,
    QgsRendererCategory, QgsSymbol, QgsFillSymbol,
    QgsSimpleFillSymbolLayer
)
from qgis.PyQt.QtGui import QColor


class LayerStyler:
    """レイヤスタイル設定クラス"""

    # 耕地種別・名称・色の定義
    # land_type: 100=田（水田）, 200=畑
    LAND_TYPES = {
        '100': {
            'name': '田（水田）',
            'name_en': 'Rice paddy',
            'color': '#7ecef4',  # 水色
            'alpha': 180
        },
        '200': {
            'name': '畑',
            'name_en': 'Upland field',
            'color': '#f5d67a',  # 黄土色
            'alpha': 180
        }
    }

    # デフォルトの枠線色
    OUTLINE_COLOR = '#666666'
    OUTLINE_WIDTH = 0.26

    def __init__(self):
        pass

    def apply_style(self, layer: QgsVectorLayer, attribute_name: str = 'land_type'):
        """
        レイヤに耕地種別の色分けスタイルを適用

        Args:
            layer: QgsVectorLayer to style
            attribute_name: Attribute name for categorization (default: 'land_type')
        """
        if not layer or not layer.isValid():
            return

        categories = []

        for land_type, info in self.LAND_TYPES.items():
            # シンボルを作成
            symbol = self._create_fill_symbol(info['color'], info['alpha'])

            # カテゴリを作成
            category = QgsRendererCategory(
                land_type,
                symbol,
                info['name']
            )
            categories.append(category)

        # カテゴリ値レンダラーを作成・適用
        renderer = QgsCategorizedSymbolRenderer(attribute_name, categories)
        layer.setRenderer(renderer)
        layer.triggerRepaint()

    def _create_fill_symbol(self, color: str, alpha: int = 255) -> QgsSymbol:
        """
        塗りつぶしシンボルを作成

        Args:
            color: Fill color (hex string)
            alpha: Alpha value (0-255)

        Returns:
            QgsSymbol
        """
        symbol = QgsFillSymbol()

        # 塗りつぶし設定
        fill_color = QColor(color)
        fill_color.setAlpha(alpha)

        # 枠線設定
        outline_color = QColor(self.OUTLINE_COLOR)

        # シンボルレイヤーを設定
        symbol_layer = symbol.symbolLayer(0)
        if isinstance(symbol_layer, QgsSimpleFillSymbolLayer):
            symbol_layer.setFillColor(fill_color)
            symbol_layer.setStrokeColor(outline_color)
            symbol_layer.setStrokeWidth(self.OUTLINE_WIDTH)

        return symbol

    def get_land_type_name(self, land_type: str, lang: str = 'ja') -> str:
        """
        耕地種別から名称を取得

        Args:
            land_type: Land type code ('100' for rice paddy, '200' for upland field)
            lang: Language ('ja' or 'en')

        Returns:
            Land type name
        """
        land_type_str = str(land_type)
        if land_type_str in self.LAND_TYPES:
            if lang == 'en':
                return self.LAND_TYPES[land_type_str]['name_en']
            return self.LAND_TYPES[land_type_str]['name']
        return '不明'

    def set_layer_name(self, layer: QgsVectorLayer, pref_name: str, city_name: str,
                       year: str):
        """
        レイヤ名を設定

        Args:
            layer: QgsVectorLayer
            pref_name: Prefecture name
            city_name: City name
            year: Data year
        """
        if layer and layer.isValid():
            layer.setName(f"筆ポリゴン_{pref_name}_{city_name}_{year}")
