# -*- coding: utf-8 -*-
"""
Japan Fude Polygon Loader - Data Downloader
筆ポリゴンデータのダウンロード処理
"""

import os
import zipfile
import tempfile
from typing import Optional, Callable, Tuple
from urllib.request import urlopen, Request
from urllib.error import URLError, HTTPError

from .region_master import RegionMaster
from .cache_manager import CacheManager


class DownloadError(Exception):
    """ダウンロードエラー"""
    pass


class DataDownloader:
    """筆ポリゴンデータのダウンロード・解凍クラス"""

    USER_AGENT = "QGIS Japan Fude Polygon Loader/1.0"

    # ダウンロードURLパターン
    # FlatGeobuf形式（都道府県単位）- 農林水産省公式サイト
    # URL: https://www.machimura.maff.go.jp/shurakudata/2020/mb/MB0001_{year}_2020_{pref_code}.zip
    FGDB_URL_PATTERN = "https://www.machimura.maff.go.jp/shurakudata/2020/mb/MB0001_{year}_2020_{pref_code}.zip"

    # 利用可能な年度
    AVAILABLE_YEARS = ["2025", "2024", "2023", "2022"]

    def __init__(self, cache_manager: Optional[CacheManager] = None):
        """
        Initialize DataDownloader.

        Args:
            cache_manager: CacheManager instance for caching downloaded data
        """
        self.region_master = RegionMaster()
        self.cache_manager = cache_manager or CacheManager()
        self.last_error = ""
        self.actual_year = ""  # 実際にダウンロードされた年度

    def download(self, pref_code: str, city_code: str, year: str = "2025",
                 format_type: str = "fgb",
                 progress_callback: Optional[Callable[[int, int], None]] = None,
                 use_cache: bool = True) -> Tuple[Optional[str], str]:
        """
        筆ポリゴンデータをダウンロードして解凍（都道府県単位のFlatGeobuf形式）

        Args:
            pref_code: 都道府県コード（2桁）
            city_code: 市区町村コード（5桁）- フィルタリング用
            year: データ年度
            format_type: データ形式（fgb推奨）
            progress_callback: 進捗コールバック関数 (downloaded, total)
            use_cache: キャッシュを使用するか

        Returns:
            Tuple of (解凍されたファイルのパス or None, エラーメッセージ)
        """
        self.last_error = ""
        self.actual_year = ""

        # キャッシュを確認（都道府県単位）
        if use_cache:
            cached_path = self.cache_manager.get_cached_file(
                pref_code, "all", year, "fgb"
            )
            if cached_path and os.path.exists(cached_path):
                self.actual_year = year
                return (cached_path, "")

        # 年度を試行（最新から順に）
        years_to_try = [year] + [y for y in self.AVAILABLE_YEARS if y != year]

        last_error = ""
        for try_year in years_to_try:
            url = self.FGDB_URL_PATTERN.format(year=try_year, pref_code=pref_code)

            try:
                # ダウンロード試行
                zip_path = self._download_file(url, progress_callback)
                if not zip_path:
                    last_error = f"ダウンロード失敗: {url}"
                    continue

                # 解凍
                extracted_path = self._extract_fgb_zip(zip_path, pref_code, try_year)

                # 一時ZIPファイルを削除
                if os.path.exists(zip_path):
                    os.remove(zip_path)

                if extracted_path:
                    self.actual_year = try_year
                    return (extracted_path, "")
                else:
                    last_error = "ZIPファイルの解凍に失敗しました"

            except HTTPError as e:
                if e.code == 404:
                    last_error = f"データが見つかりません（年度: {try_year}）"
                    continue  # 次の年度を試す
                else:
                    last_error = f"HTTPエラー {e.code}: {url}"
            except URLError as e:
                last_error = f"ネットワークエラー: {e.reason}"
                break  # ネットワークエラーは再試行しない
            except Exception as e:
                last_error = f"エラー: {str(e)}"

        self.last_error = last_error
        return (None, last_error)

    def _extract_fgb_zip(self, zip_path: str, pref_code: str, year: str) -> Optional[str]:
        """
        FlatGeobuf ZIPファイルを解凍

        Args:
            zip_path: ZIPファイルのパス
            pref_code: 都道府県コード
            year: データ年度

        Returns:
            解凍されたFGBファイルのパス
        """
        try:
            # キャッシュディレクトリを取得
            cache_dir = self.cache_manager.get_cache_dir(pref_code, "all", year)
            os.makedirs(cache_dir, exist_ok=True)

            with zipfile.ZipFile(zip_path, 'r') as zf:
                file_list = zf.namelist()

                # FGBファイルを探す
                target_file = None
                for name in file_list:
                    if name.lower().endswith('.fgb'):
                        target_file = name
                        break

                if not target_file:
                    return None

                # ファイルを解凍
                zf.extract(target_file, cache_dir)
                extracted_path = os.path.join(cache_dir, target_file)

                # キャッシュに登録
                self.cache_manager.register_cache(
                    pref_code, "all", year, "fgb", extracted_path
                )

                return extracted_path

        except zipfile.BadZipFile:
            return None
        except Exception:
            return None

    def _get_download_urls(self, pref_code: str, year: str) -> list:
        """
        ダウンロードURLのリストを取得（都道府県単位）
        """
        urls = []

        # 複数の年度を試す
        years_to_try = [year] + [y for y in self.AVAILABLE_YEARS if y != year]

        for try_year in years_to_try:
            url = self.FGDB_URL_PATTERN.format(year=try_year, pref_code=pref_code)
            urls.append(url)

        return urls

    def _download_file(self, url: str,
                       progress_callback: Optional[Callable[[int, int], None]] = None) -> Optional[str]:
        """
        URLからファイルをダウンロード

        Args:
            url: Download URL
            progress_callback: Progress callback (downloaded_bytes, total_bytes)

        Returns:
            Downloaded file path or None

        Raises:
            HTTPError, URLError
        """
        request = Request(url, headers={'User-Agent': self.USER_AGENT})
        response = urlopen(request, timeout=60)

        # ファイルサイズを取得
        total_size = int(response.headers.get('Content-Length', 0))

        # 一時ファイルに保存
        fd, temp_path = tempfile.mkstemp(suffix='.zip')
        downloaded = 0
        chunk_size = 8192

        try:
            with os.fdopen(fd, 'wb') as f:
                while True:
                    chunk = response.read(chunk_size)
                    if not chunk:
                        break
                    f.write(chunk)
                    downloaded += len(chunk)
                    if progress_callback:
                        progress_callback(downloaded, total_size)

            return temp_path
        except Exception:
            # エラー時は一時ファイルを削除
            if os.path.exists(temp_path):
                os.remove(temp_path)
            raise

    def _extract_zip(self, zip_path: str, pref_code: str, city_code: str,
                     year: str, format_type: str) -> Optional[str]:
        """
        ZIPファイルを解凍

        Args:
            zip_path: ZIPファイルのパス
            pref_code: 都道府県コード
            city_code: 市区町村コード
            year: データ年度
            format_type: データ形式

        Returns:
            解凍されたGeoJSONファイルのパス
        """
        try:
            # キャッシュディレクトリを取得
            cache_dir = self.cache_manager.get_cache_dir(pref_code, city_code, year)
            os.makedirs(cache_dir, exist_ok=True)

            with zipfile.ZipFile(zip_path, 'r') as zf:
                # ZIPの中身を確認
                file_list = zf.namelist()

                # GeoJSONまたはFGBファイルを探す
                target_ext = '.fgb' if format_type == 'fgb' else '.geojson'
                target_file = None

                for name in file_list:
                    if name.lower().endswith(target_ext):
                        target_file = name
                        break

                if not target_file:
                    # 拡張子がない場合、.jsonも試す
                    for name in file_list:
                        if name.lower().endswith('.json'):
                            target_file = name
                            break

                if not target_file:
                    return None

                # ファイルを解凍
                zf.extract(target_file, cache_dir)
                extracted_path = os.path.join(cache_dir, target_file)

                # キャッシュに登録
                self.cache_manager.register_cache(
                    pref_code, city_code, year, format_type, extracted_path
                )

                return extracted_path

        except zipfile.BadZipFile:
            return None
        except Exception:
            return None

    def get_available_years(self, pref_code: str, city_code: str) -> list:
        """
        利用可能なデータ年度を取得

        Args:
            pref_code: 都道府県コード
            city_code: 市区町村コード

        Returns:
            List of available years (newest first)
        """
        available_years = ["2024", "2023", "2022"]
        return available_years
