# -*- coding: utf-8 -*-
"""
Japan Forest Tools - Main Dialog
"""

import json

from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QTabWidget, QWidget,
    QGroupBox, QLabel, QComboBox, QCheckBox, QPushButton,
    QMessageBox, QSlider, QSpinBox
)
from qgis.PyQt.QtCore import Qt, QUrl
from qgis.PyQt.QtNetwork import QNetworkRequest
from qgis.core import (
    QgsProject, QgsVectorLayer, QgsRasterLayer,
    QgsCoordinateReferenceSystem, QgsCoordinateTransform,
    QgsMessageLog, Qgis, QgsBlockingNetworkRequest
)

from ..core import DataSourceManager, ForestDataLoader


DEBUG_LOG = False


def log_info(message: str):
    """Info level log."""
    if DEBUG_LOG:
        QgsMessageLog.logMessage(message, "JapanForestTools", Qgis.Info)


def log_error(message: str):
    """Error level log."""
    QgsMessageLog.logMessage(message, "JapanForestTools", Qgis.Warning)


class MainDialog(QDialog):
    """Main dialog with tabs."""

    def __init__(self, iface, parent=None):
        super().__init__(parent)
        self.iface = iface
        self.data_manager = DataSourceManager()
        self.loader = ForestDataLoader()
        self._detected_location = ""  # Store detected location for region matching

        self.setup_ui()
        self.connect_signals()
        self.load_prefectures()

    def setup_ui(self):
        """Setup UI."""
        self.setWindowTitle("Japan Forest Tools - 森林データ取得")
        self.setMinimumSize(550, 700)
        layout = QVBoxLayout(self)

        # Tab widget
        self.tabs = QTabWidget()
        layout.addWidget(self.tabs)

        # Tab 1: Forest Agency Open Data
        self.tab_rinnya = QWidget()
        self.setup_rinnya_tab()
        self.tabs.addTab(self.tab_rinnya, "林野庁オープンデータ")

        # Tab 2: National Land Information
        self.tab_kokudo = QWidget()
        self.setup_kokudo_tab()
        self.tabs.addTab(self.tab_kokudo, "国土数値情報")

        # Tab 3: Forest Planning Data (林班・小班)
        self.tab_forest_plan = QWidget()
        self.setup_forest_plan_tab()
        self.tabs.addTab(self.tab_forest_plan, "森林計画図")

        # Tab 4: OSM Forest Roads
        self.tab_osm = QWidget()
        self.setup_osm_tab()
        self.tabs.addTab(self.tab_osm, "OSM林道データ")

        # Tab 5: Settings
        self.tab_settings = QWidget()
        self.setup_settings_tab()
        self.tabs.addTab(self.tab_settings, "設定")

        # Progress
        self.progress_label = QLabel("")
        layout.addWidget(self.progress_label)

        # Buttons
        btn_layout = QHBoxLayout()
        btn_layout.addStretch()
        self.btn_close = QPushButton("閉じる")
        self.btn_close.clicked.connect(self.close)
        btn_layout.addWidget(self.btn_close)
        layout.addLayout(btn_layout)

    def setup_rinnya_tab(self):
        """Setup Forest Agency open data tab."""
        layout = QVBoxLayout(self.tab_rinnya)

        # Current view info
        view_info = QLabel("※ 現在の表示範囲のデータのみ取得します")
        view_info.setStyleSheet("color: #006400; font-weight: bold;")
        layout.addWidget(view_info)

        # Prefecture detection
        detect_group = QGroupBox("都道府県")
        detect_layout = QVBoxLayout(detect_group)

        detect_btn_layout = QHBoxLayout()
        self.btn_detect_rinnya = QPushButton("現在地から検出")
        detect_btn_layout.addWidget(self.btn_detect_rinnya)
        self.pref_combo_rinnya = QComboBox()
        self.pref_combo_rinnya.setMinimumWidth(150)
        detect_btn_layout.addWidget(self.pref_combo_rinnya)
        detect_btn_layout.addStretch()
        detect_layout.addLayout(detect_btn_layout)

        self.rinnya_pref_note = QLabel("")
        self.rinnya_pref_note.setStyleSheet("color: gray; font-size: 10px;")
        detect_layout.addWidget(self.rinnya_pref_note)

        layout.addWidget(detect_group)

        # Data selection
        data_group = QGroupBox("データ選択")
        data_layout = QVBoxLayout(data_group)

        # Tree species
        self.cb_tree_species = QCheckBox("樹種ポリゴン")
        data_layout.addWidget(self.cb_tree_species)

        # Tree species format selection
        tree_format_layout = QHBoxLayout()
        tree_format_layout.setContentsMargins(20, 0, 0, 0)
        tree_format_layout.addWidget(QLabel("形式:"))
        self.combo_tree_format = QComboBox()
        self.combo_tree_format.addItem("ベクトルタイル（高速・凡例簡略）", "xyz")
        self.combo_tree_format.addItem("GeoPackage（低速・凡例完全）", "gpkg")
        self.combo_tree_format.setCurrentIndex(0)
        tree_format_layout.addWidget(self.combo_tree_format)
        tree_format_layout.addStretch()
        data_layout.addLayout(tree_format_layout)

        self.tree_format_note = QLabel(
            "　※ベクトルタイル：即時表示、凡例は樹種名のみ\n"
            "　※GeoPackage：ダウンロード必要、属性情報完全"
        )
        self.tree_format_note.setStyleSheet("color: gray; font-size: 9px;")
        data_layout.addWidget(self.tree_format_note)

        # Forest planning region selection (for GeoPackage)
        self.gpkg_region_group = QGroupBox("森林計画区（GeoPackage選択時）")
        gpkg_region_layout = QVBoxLayout(self.gpkg_region_group)

        gpkg_region_note = QLabel("※ ダウンロードする森林計画区を選択できます")
        gpkg_region_note.setStyleSheet("color: gray; font-size: 9px;")
        gpkg_region_layout.addWidget(gpkg_region_note)

        # Button to fetch region list
        self.btn_fetch_regions = QPushButton("森林計画区一覧を取得")
        self.btn_fetch_regions.setMaximumWidth(200)
        gpkg_region_layout.addWidget(self.btn_fetch_regions)

        # Region list (scrollable)
        from qgis.PyQt.QtWidgets import QScrollArea, QWidget
        self.region_scroll = QScrollArea()
        self.region_scroll.setWidgetResizable(True)
        self.region_scroll.setMaximumHeight(120)
        self.region_container = QWidget()
        self.region_list_layout = QVBoxLayout(self.region_container)
        self.region_list_layout.setContentsMargins(5, 5, 5, 5)
        self.region_list_layout.setSpacing(2)
        self.region_scroll.setWidget(self.region_container)
        gpkg_region_layout.addWidget(self.region_scroll)

        # Select all / none buttons
        region_btn_layout = QHBoxLayout()
        self.btn_select_all_regions = QPushButton("全選択")
        self.btn_select_all_regions.setMaximumWidth(80)
        self.btn_select_none_regions = QPushButton("全解除")
        self.btn_select_none_regions.setMaximumWidth(80)
        region_btn_layout.addWidget(self.btn_select_all_regions)
        region_btn_layout.addWidget(self.btn_select_none_regions)
        region_btn_layout.addStretch()
        gpkg_region_layout.addLayout(region_btn_layout)

        self.gpkg_region_group.setVisible(False)  # Hidden by default
        data_layout.addWidget(self.gpkg_region_group)

        # Store region checkboxes
        self.region_checkboxes = []
        self.available_gpkg_urls = []  # Store URLs for selected regions

        # Tile data
        self.cb_dem = QCheckBox("DEM（数値標高モデル）")
        self.cb_cs_map = QCheckBox("CS立体図（微地形表現図）")
        self.cb_slope = QCheckBox("傾斜区分図")

        data_layout.addWidget(self.cb_dem)
        data_layout.addWidget(self.cb_cs_map)
        data_layout.addWidget(self.cb_slope)

        layout.addWidget(data_group)

        # Download button
        self.btn_download_rinnya = QPushButton("選択データを取得")
        self.btn_download_rinnya.setMinimumHeight(40)
        self.btn_download_rinnya.setStyleSheet("font-weight: bold;")
        layout.addWidget(self.btn_download_rinnya)

        layout.addStretch()

    def setup_kokudo_tab(self):
        """Setup National Land Information tab."""
        layout = QVBoxLayout(self.tab_kokudo)

        # Current view info
        view_info = QLabel("※ 現在の表示範囲のデータのみ取得します")
        view_info.setStyleSheet("color: #006400; font-weight: bold;")
        layout.addWidget(view_info)

        # Prefecture detection
        detect_group = QGroupBox("都道府県")
        detect_layout = QVBoxLayout(detect_group)

        detect_btn_layout = QHBoxLayout()
        self.btn_detect_kokudo = QPushButton("現在地から検出")
        detect_btn_layout.addWidget(self.btn_detect_kokudo)
        self.pref_combo_kokudo = QComboBox()
        self.pref_combo_kokudo.setMinimumWidth(150)
        detect_btn_layout.addWidget(self.pref_combo_kokudo)
        detect_btn_layout.addStretch()
        detect_layout.addLayout(detect_btn_layout)

        layout.addWidget(detect_group)

        # Data selection
        data_group = QGroupBox("データ選択")
        data_layout = QVBoxLayout(data_group)

        self.cb_national_forest = QCheckBox("国有林野データ（林班・小班区画）")
        self.cb_forest_area = QCheckBox("森林地域データ")

        data_layout.addWidget(self.cb_national_forest)
        data_layout.addWidget(self.cb_forest_area)

        kokudo_note = QLabel("※ データは表示範囲に絞り込まれます\n※ 国有林野データには林班・小班の境界が含まれます")
        kokudo_note.setStyleSheet("color: gray; font-size: 10px;")
        data_layout.addWidget(kokudo_note)

        layout.addWidget(data_group)

        # Download button
        self.btn_download_kokudo = QPushButton("選択データを取得")
        self.btn_download_kokudo.setMinimumHeight(40)
        self.btn_download_kokudo.setStyleSheet("font-weight: bold;")
        layout.addWidget(self.btn_download_kokudo)

        layout.addStretch()

    def setup_forest_plan_tab(self):
        """Setup forest planning data (林班・小班) tab."""
        layout = QVBoxLayout(self.tab_forest_plan)

        # Current view info
        view_info = QLabel("※ 現在の表示範囲のデータのみ取得します")
        view_info.setStyleSheet("color: #006400; font-weight: bold;")
        layout.addWidget(view_info)

        # Prefecture selection
        pref_group = QGroupBox("都道府県")
        pref_layout = QVBoxLayout(pref_group)

        detect_btn_layout = QHBoxLayout()
        self.btn_detect_forest_plan = QPushButton("現在地から検出")
        detect_btn_layout.addWidget(self.btn_detect_forest_plan)
        self.pref_combo_forest_plan = QComboBox()
        self.pref_combo_forest_plan.setMinimumWidth(150)
        detect_btn_layout.addWidget(self.pref_combo_forest_plan)
        detect_btn_layout.addStretch()
        pref_layout.addLayout(detect_btn_layout)

        self.forest_plan_pref_note = QLabel("")
        self.forest_plan_pref_note.setStyleSheet("color: gray; font-size: 10px;")
        pref_layout.addWidget(self.forest_plan_pref_note)

        layout.addWidget(pref_group)

        # Data type selection - Radio buttons (one at a time)
        data_group = QGroupBox("表示タイプ")
        data_layout = QVBoxLayout(data_group)

        from qgis.PyQt.QtWidgets import QRadioButton, QButtonGroup

        self.rb_forest_shohan = QRadioButton("小班（小区画 - 個別ポリゴン表示）")
        self.rb_forest_rinpan = QRadioButton("林班（大区画 - 林班番号で集約表示）")
        self.rb_forest_shohan.setChecked(True)

        # Group radio buttons
        self.forest_plan_btn_group = QButtonGroup()
        self.forest_plan_btn_group.addButton(self.rb_forest_shohan)
        self.forest_plan_btn_group.addButton(self.rb_forest_rinpan)

        data_layout.addWidget(self.rb_forest_shohan)
        data_layout.addWidget(self.rb_forest_rinpan)

        forest_plan_note = QLabel(
            "※ 対応都道府県: 長崎県、熊本県、栃木県（渡良瀬川地域）\n"
            "※ 初回はダウンロードに時間がかかります（数十～数百MB）\n"
            "※ 小班: 元データのポリゴンをそのまま表示\n"
            "※ 林班: 林班番号で集約した大きな区画を表示"
        )
        forest_plan_note.setStyleSheet("color: gray; font-size: 10px;")
        data_layout.addWidget(forest_plan_note)

        layout.addWidget(data_group)

        # Download button
        self.btn_download_forest_plan = QPushButton("森林計画図を取得")
        self.btn_download_forest_plan.setMinimumHeight(40)
        self.btn_download_forest_plan.setStyleSheet("font-weight: bold;")
        layout.addWidget(self.btn_download_forest_plan)

        layout.addStretch()

    def setup_osm_tab(self):
        """Setup OSM forest roads tab."""
        layout = QVBoxLayout(self.tab_osm)

        # Current view info
        view_info = QLabel("※ 現在の表示範囲からOSMの林道・作業道を取得します")
        view_info.setStyleSheet("color: #006400; font-weight: bold;")
        layout.addWidget(view_info)

        # Road type selection
        type_group = QGroupBox("道路種別")
        type_layout = QVBoxLayout(type_group)

        self.cb_osm_track = QCheckBox("track（林道・農道）")
        self.cb_osm_track.setChecked(True)
        self.cb_osm_path = QCheckBox("path（作業道・歩道）")

        type_layout.addWidget(self.cb_osm_track)
        type_layout.addWidget(self.cb_osm_path)

        osm_note = QLabel("※ 表示範囲が広いとデータ量が多くなります")
        osm_note.setStyleSheet("color: gray; font-size: 10px;")
        type_layout.addWidget(osm_note)

        layout.addWidget(type_group)

        # Download button
        self.btn_download_osm = QPushButton("OSM林道データを取得")
        self.btn_download_osm.setMinimumHeight(40)
        self.btn_download_osm.setStyleSheet("font-weight: bold;")
        layout.addWidget(self.btn_download_osm)

        layout.addStretch()

    def setup_settings_tab(self):
        """Setup settings tab."""
        layout = QVBoxLayout(self.tab_settings)

        # Transparency controls
        opacity_group = QGroupBox("透過度設定")
        opacity_layout = QVBoxLayout(opacity_group)

        opacity_note_top = QLabel("読み込み後のレイヤにリアルタイム適用されます")
        opacity_note_top.setStyleSheet("color: #006400; font-size: 11px;")
        opacity_layout.addWidget(opacity_note_top)

        # Tree species polygon opacity
        tree_opacity_layout = QHBoxLayout()
        tree_opacity_layout.addWidget(QLabel("樹種ポリゴン:"))
        self.slider_tree_opacity = QSlider(Qt.Horizontal)
        self.slider_tree_opacity.setRange(0, 100)
        self.slider_tree_opacity.setValue(70)
        self.slider_tree_opacity.setFixedWidth(120)
        tree_opacity_layout.addWidget(self.slider_tree_opacity)
        self.spin_tree_opacity = QSpinBox()
        self.spin_tree_opacity.setRange(0, 100)
        self.spin_tree_opacity.setValue(70)
        self.spin_tree_opacity.setSuffix("%")
        tree_opacity_layout.addWidget(self.spin_tree_opacity)
        tree_opacity_layout.addStretch()
        opacity_layout.addLayout(tree_opacity_layout)

        # DEM opacity
        dem_opacity_layout = QHBoxLayout()
        dem_opacity_layout.addWidget(QLabel("DEM:"))
        self.slider_dem_opacity = QSlider(Qt.Horizontal)
        self.slider_dem_opacity.setRange(0, 100)
        self.slider_dem_opacity.setValue(100)
        self.slider_dem_opacity.setFixedWidth(120)
        dem_opacity_layout.addWidget(self.slider_dem_opacity)
        self.spin_dem_opacity = QSpinBox()
        self.spin_dem_opacity.setRange(0, 100)
        self.spin_dem_opacity.setValue(100)
        self.spin_dem_opacity.setSuffix("%")
        dem_opacity_layout.addWidget(self.spin_dem_opacity)
        dem_opacity_layout.addStretch()
        opacity_layout.addLayout(dem_opacity_layout)

        # CS map opacity
        cs_opacity_layout = QHBoxLayout()
        cs_opacity_layout.addWidget(QLabel("CS立体図:"))
        self.slider_cs_opacity = QSlider(Qt.Horizontal)
        self.slider_cs_opacity.setRange(0, 100)
        self.slider_cs_opacity.setValue(100)
        self.slider_cs_opacity.setFixedWidth(120)
        cs_opacity_layout.addWidget(self.slider_cs_opacity)
        self.spin_cs_opacity = QSpinBox()
        self.spin_cs_opacity.setRange(0, 100)
        self.spin_cs_opacity.setValue(100)
        self.spin_cs_opacity.setSuffix("%")
        cs_opacity_layout.addWidget(self.spin_cs_opacity)
        cs_opacity_layout.addStretch()
        opacity_layout.addLayout(cs_opacity_layout)

        # Slope map opacity
        slope_opacity_layout = QHBoxLayout()
        slope_opacity_layout.addWidget(QLabel("傾斜区分図:"))
        self.slider_slope_opacity = QSlider(Qt.Horizontal)
        self.slider_slope_opacity.setRange(0, 100)
        self.slider_slope_opacity.setValue(100)
        self.slider_slope_opacity.setFixedWidth(120)
        slope_opacity_layout.addWidget(self.slider_slope_opacity)
        self.spin_slope_opacity = QSpinBox()
        self.spin_slope_opacity.setRange(0, 100)
        self.spin_slope_opacity.setValue(100)
        self.spin_slope_opacity.setSuffix("%")
        slope_opacity_layout.addWidget(self.spin_slope_opacity)
        slope_opacity_layout.addStretch()
        opacity_layout.addLayout(slope_opacity_layout)

        layout.addWidget(opacity_group)

        # CS立体図の色設定
        cs_color_group = QGroupBox("CS立体図 色設定")
        cs_color_layout = QVBoxLayout(cs_color_group)

        cs_color_note = QLabel("CS立体図のレイヤに色を付けて表示します")
        cs_color_note.setStyleSheet("color: #006400; font-size: 11px;")
        cs_color_layout.addWidget(cs_color_note)

        cs_color_select_layout = QHBoxLayout()
        cs_color_select_layout.addWidget(QLabel("表示色:"))
        self.combo_cs_color = QComboBox()
        self.combo_cs_color.addItem("デフォルト（元の色）", "default")
        self.combo_cs_color.addItem("赤", "red")
        self.combo_cs_color.addItem("緑", "green")
        self.combo_cs_color.addItem("青", "blue")
        self.combo_cs_color.setCurrentIndex(0)
        cs_color_select_layout.addWidget(self.combo_cs_color)
        cs_color_select_layout.addStretch()
        cs_color_layout.addLayout(cs_color_select_layout)

        layout.addWidget(cs_color_group)

        # 国土数値情報の透過度設定
        kokudo_opacity_group = QGroupBox("国土数値情報")
        kokudo_opacity_layout = QVBoxLayout(kokudo_opacity_group)

        # 国有林野 opacity
        national_forest_layout = QHBoxLayout()
        national_forest_layout.addWidget(QLabel("国有林野:"))
        self.slider_national_forest_opacity = QSlider(Qt.Horizontal)
        self.slider_national_forest_opacity.setRange(0, 100)
        self.slider_national_forest_opacity.setValue(70)
        self.slider_national_forest_opacity.setFixedWidth(120)
        national_forest_layout.addWidget(self.slider_national_forest_opacity)
        self.spin_national_forest_opacity = QSpinBox()
        self.spin_national_forest_opacity.setRange(0, 100)
        self.spin_national_forest_opacity.setValue(70)
        self.spin_national_forest_opacity.setSuffix("%")
        national_forest_layout.addWidget(self.spin_national_forest_opacity)
        national_forest_layout.addStretch()
        kokudo_opacity_layout.addLayout(national_forest_layout)

        # 森林地域 opacity
        forest_area_layout = QHBoxLayout()
        forest_area_layout.addWidget(QLabel("森林地域:"))
        self.slider_forest_area_opacity = QSlider(Qt.Horizontal)
        self.slider_forest_area_opacity.setRange(0, 100)
        self.slider_forest_area_opacity.setValue(50)
        self.slider_forest_area_opacity.setFixedWidth(120)
        forest_area_layout.addWidget(self.slider_forest_area_opacity)
        self.spin_forest_area_opacity = QSpinBox()
        self.spin_forest_area_opacity.setRange(0, 100)
        self.spin_forest_area_opacity.setValue(50)
        self.spin_forest_area_opacity.setSuffix("%")
        forest_area_layout.addWidget(self.spin_forest_area_opacity)
        forest_area_layout.addStretch()
        kokudo_opacity_layout.addLayout(forest_area_layout)

        layout.addWidget(kokudo_opacity_group)

        # 森林計画図の透過度設定
        forest_plan_opacity_group = QGroupBox("森林計画図")
        forest_plan_opacity_layout = QVBoxLayout(forest_plan_opacity_group)

        # 林班 opacity
        rinpan_layout = QHBoxLayout()
        rinpan_layout.addWidget(QLabel("林班:"))
        self.slider_rinpan_opacity = QSlider(Qt.Horizontal)
        self.slider_rinpan_opacity.setRange(0, 100)
        self.slider_rinpan_opacity.setValue(70)
        self.slider_rinpan_opacity.setFixedWidth(120)
        rinpan_layout.addWidget(self.slider_rinpan_opacity)
        self.spin_rinpan_opacity = QSpinBox()
        self.spin_rinpan_opacity.setRange(0, 100)
        self.spin_rinpan_opacity.setValue(70)
        self.spin_rinpan_opacity.setSuffix("%")
        rinpan_layout.addWidget(self.spin_rinpan_opacity)
        rinpan_layout.addStretch()
        forest_plan_opacity_layout.addLayout(rinpan_layout)

        # 小班 opacity
        shohan_layout = QHBoxLayout()
        shohan_layout.addWidget(QLabel("小班:"))
        self.slider_shohan_opacity = QSlider(Qt.Horizontal)
        self.slider_shohan_opacity.setRange(0, 100)
        self.slider_shohan_opacity.setValue(70)
        self.slider_shohan_opacity.setFixedWidth(120)
        shohan_layout.addWidget(self.slider_shohan_opacity)
        self.spin_shohan_opacity = QSpinBox()
        self.spin_shohan_opacity.setRange(0, 100)
        self.spin_shohan_opacity.setValue(70)
        self.spin_shohan_opacity.setSuffix("%")
        shohan_layout.addWidget(self.spin_shohan_opacity)
        shohan_layout.addStretch()
        forest_plan_opacity_layout.addLayout(shohan_layout)

        layout.addWidget(forest_plan_opacity_group)

        layout.addStretch()

    def connect_signals(self):
        """Connect signals."""
        # Prefecture detection buttons
        self.btn_detect_rinnya.clicked.connect(lambda: self.detect_current_location('rinnya'))
        self.btn_detect_kokudo.clicked.connect(lambda: self.detect_current_location('kokudo'))
        self.btn_detect_forest_plan.clicked.connect(lambda: self.detect_current_location('forest_plan'))

        # Prefecture combo changes
        self.pref_combo_rinnya.currentIndexChanged.connect(self.on_rinnya_pref_changed)
        self.pref_combo_forest_plan.currentIndexChanged.connect(self.on_forest_plan_pref_changed)

        # Download buttons
        self.btn_download_rinnya.clicked.connect(self.download_rinnya_data)
        self.btn_download_kokudo.clicked.connect(self.download_kokudo_data)
        self.btn_download_forest_plan.clicked.connect(self.download_forest_plan_data)
        self.btn_download_osm.clicked.connect(self.download_osm_data)

        # Tree species format change (show/hide region selection)
        self.combo_tree_format.currentIndexChanged.connect(self.on_tree_format_changed)

        # Forest planning region selection
        self.btn_fetch_regions.clicked.connect(self.fetch_gpkg_regions)
        self.btn_select_all_regions.clicked.connect(self.select_all_regions)
        self.btn_select_none_regions.clicked.connect(self.select_none_regions)

        # Opacity slider/spinbox sync and apply
        self.slider_tree_opacity.valueChanged.connect(self.spin_tree_opacity.setValue)
        self.spin_tree_opacity.valueChanged.connect(self.slider_tree_opacity.setValue)
        self.spin_tree_opacity.valueChanged.connect(lambda v: self._apply_opacity('tree_species', v))

        self.slider_dem_opacity.valueChanged.connect(self.spin_dem_opacity.setValue)
        self.spin_dem_opacity.valueChanged.connect(self.slider_dem_opacity.setValue)
        self.spin_dem_opacity.valueChanged.connect(lambda v: self._apply_opacity('dem', v))

        self.slider_cs_opacity.valueChanged.connect(self.spin_cs_opacity.setValue)
        self.spin_cs_opacity.valueChanged.connect(self.slider_cs_opacity.setValue)
        self.spin_cs_opacity.valueChanged.connect(lambda v: self._apply_opacity('cs_map', v))

        self.slider_slope_opacity.valueChanged.connect(self.spin_slope_opacity.setValue)
        self.spin_slope_opacity.valueChanged.connect(self.slider_slope_opacity.setValue)
        self.spin_slope_opacity.valueChanged.connect(lambda v: self._apply_opacity('slope', v))

        # 国土数値情報の透過度
        self.slider_national_forest_opacity.valueChanged.connect(self.spin_national_forest_opacity.setValue)
        self.spin_national_forest_opacity.valueChanged.connect(self.slider_national_forest_opacity.setValue)
        self.spin_national_forest_opacity.valueChanged.connect(lambda v: self._apply_opacity('national_forest', v))

        self.slider_forest_area_opacity.valueChanged.connect(self.spin_forest_area_opacity.setValue)
        self.spin_forest_area_opacity.valueChanged.connect(self.slider_forest_area_opacity.setValue)
        self.spin_forest_area_opacity.valueChanged.connect(lambda v: self._apply_opacity('forest_area', v))

        # 森林計画図の透過度
        self.slider_rinpan_opacity.valueChanged.connect(self.spin_rinpan_opacity.setValue)
        self.spin_rinpan_opacity.valueChanged.connect(self.slider_rinpan_opacity.setValue)
        self.spin_rinpan_opacity.valueChanged.connect(lambda v: self._apply_opacity('forest_plan_rinpan', v))

        self.slider_shohan_opacity.valueChanged.connect(self.spin_shohan_opacity.setValue)
        self.spin_shohan_opacity.valueChanged.connect(self.slider_shohan_opacity.setValue)
        self.spin_shohan_opacity.valueChanged.connect(lambda v: self._apply_opacity('forest_plan_shohan', v))

        # CS立体図の色変更
        self.combo_cs_color.currentIndexChanged.connect(self._apply_cs_color)

    def load_prefectures(self):
        """Load prefectures into combo boxes."""
        for combo in [self.pref_combo_rinnya, self.pref_combo_kokudo]:
            combo.clear()
            combo.addItem("-- 選択してください --", "")
            for code, name in self.data_manager.get_prefectures():
                combo.addItem(f"{name} ({code})", code)

        # Forest plan combo - only show available prefectures
        self.pref_combo_forest_plan.clear()
        self.pref_combo_forest_plan.addItem("-- 選択してください --", "")
        available_prefs = self.loader.FOREST_PLAN_AVAILABLE
        for code, name in self.data_manager.get_prefectures():
            if code in available_prefs:
                plan_info = self.loader.FOREST_PLAN_DATA.get(code, {})
                size_mb = plan_info.get('size_mb', 0)
                self.pref_combo_forest_plan.addItem(f"{name} ({size_mb}MB)", code)

    def detect_current_location(self, tab_type: str):
        """Detect prefecture from current map view."""
        log_info(f"現在地検出開始: tab_type={tab_type}")

        canvas = self.iface.mapCanvas()
        center = canvas.center()
        crs = canvas.mapSettings().destinationCrs()

        # Convert to WGS84
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        if crs.authid() != "EPSG:4326":
            transform = QgsCoordinateTransform(crs, wgs84, QgsProject.instance())
            center = transform.transform(center)

        lat = center.y()
        lon = center.x()
        log_info(f"中心座標: lat={lat:.6f}, lon={lon:.6f}")

        # Reverse geocode
        result = self._reverse_geocode(lat, lon)

        if result:
            pref_code = result['pref_code']
            pref_name = result['pref_name']
            location_name = result.get('location_name', '')
            log_info(f"検出結果: {pref_name}({pref_code}) {location_name}")

            # Select in ALL combo boxes (sync across tabs)
            self._sync_all_pref_combos(pref_code)

            # Update location info for rinnya tab
            available = self.data_manager.get_available_data(pref_code)
            location_info = f"検出: {pref_name}"
            if location_name:
                location_info += f" {location_name}"

            if 'tree_species' in available:
                self.rinnya_pref_note.setText(f"※ {location_info}")
                self.rinnya_pref_note.setStyleSheet("color: #006400; font-size: 10px;")
            else:
                self.rinnya_pref_note.setText(f"※ {location_info}（樹種ポリゴン未対応）")
                self.rinnya_pref_note.setStyleSheet("color: #B22222; font-size: 10px;")

            # Store detected location for region matching
            self._detected_location = location_name

            # Auto-fetch regions if GeoPackage format is selected and tree species available
            if self.combo_tree_format.currentData() == 'gpkg' and 'tree_species' in available:
                self.gpkg_region_group.setVisible(True)
                self.fetch_gpkg_regions()
                if location_name and self.region_checkboxes:
                    self._auto_select_region(location_name)

            log_info("現在地検出完了")
        else:
            log_info("逆ジオコーディング失敗")
            QMessageBox.warning(self, "検出失敗", "都道府県を検出できませんでした。")

    def _sync_all_pref_combos(self, pref_code: str):
        """Sync all prefecture combo boxes to the same prefecture."""
        combos = [
            self.pref_combo_rinnya,
            self.pref_combo_kokudo,
            self.pref_combo_forest_plan,
        ]

        for combo in combos:
            # Block signals to avoid triggering change handlers multiple times
            combo.blockSignals(True)
            for i in range(combo.count()):
                if combo.itemData(i) == pref_code:
                    combo.setCurrentIndex(i)
                    break
            combo.blockSignals(False)

        # Manually trigger change handlers once
        self.on_rinnya_pref_changed(self.pref_combo_rinnya.currentIndex())
        self.on_forest_plan_pref_changed(self.pref_combo_forest_plan.currentIndex())
        log_info(f"全タブ連携完了: {pref_code}")

    def _auto_select_region(self, location_name: str):
        """Auto-select region checkbox that matches the detected location."""
        if not location_name or not self.region_checkboxes:
            return

        # First, deselect all
        for cb in self.region_checkboxes:
            cb.setChecked(False)

        # Try to find matching region
        # Location name might be like "長崎市" or "諫早市"
        # Region name might contain the area name
        matched = False
        for cb in self.region_checkboxes:
            region_name = cb.property('name') or cb.text()

            # Check if location name appears in region name (partial match)
            # Also check the other way around
            if location_name in region_name or any(part in location_name for part in region_name.split('_') if len(part) > 1):
                cb.setChecked(True)
                matched = True
                log_info(f"地域自動選択: {region_name} (検出地点: {location_name})")

        # If no match found, select all
        if not matched:
            log_info(f"地域マッチなし、全選択: {location_name}")
            for cb in self.region_checkboxes:
                cb.setChecked(True)

    def _reverse_geocode(self, lat, lon):
        """Reverse geocode using GSI API."""
        url = f"https://mreversegeocoder.gsi.go.jp/reverse-geocoder/LonLatToAddress?lat={lat}&lon={lon}"
        log_info(f"逆ジオコーディング: {url}")

        try:
            req = QNetworkRequest(QUrl(url))
            blocker = QgsBlockingNetworkRequest()
            err = blocker.get(req, forceRefresh=True)
            if err != QgsBlockingNetworkRequest.NoError:
                log_error(f"逆ジオコーディング ネットワークエラー: {blocker.errorMessage()}")
                return None

            content = bytes(blocker.reply().content())
            data = json.loads(content.decode('utf-8'))

            results = data.get('results')
            if not results:
                log_info("逆ジオコーディング: 結果なし")
                return None

            muni_cd = results.get('muniCd', '')
            if not muni_cd or len(muni_cd) < 2:
                log_info(f"逆ジオコーディング: 市町村コード不正 - {muni_cd}")
                return None

            pref_code = muni_cd[:2]
            pref_name = self.data_manager.get_prefecture_name(pref_code)

            lv02_nm = results.get('lv02Nm', '')
            lv03_nm = results.get('lv03Nm', '')

            location_name = ''
            if lv02_nm:
                location_name = lv02_nm
            if lv03_nm:
                location_name += lv03_nm

            log_info(f"逆ジオコーディング成功: {pref_name}({pref_code}) {location_name}")

            return {
                'pref_code': pref_code,
                'pref_name': pref_name,
                'location_name': location_name,
                'muni_code': muni_cd,
            }

        except Exception as e:
            log_error(f"逆ジオコーディングエラー: {str(e)}")
            return None

    def on_rinnya_pref_changed(self, index):
        """Prefecture changed for rinnya tab."""
        pref_code = self.pref_combo_rinnya.currentData()
        if not pref_code:
            self.rinnya_pref_note.setText("")
            self.cb_tree_species.setEnabled(True)
            self.cb_tree_species.setText("樹種ポリゴン")
            self._clear_region_list()
            return

        # Clear previous region list when prefecture changes
        self._clear_region_list()
        self._detected_location = ""

        # Check available data
        available = self.data_manager.get_available_data(pref_code)
        pref_name = self.data_manager.get_prefecture_name(pref_code)

        if 'tree_species' in available:
            self.rinnya_pref_note.setText(f"※ {pref_name}は樹種ポリゴン対応済み")
            self.rinnya_pref_note.setStyleSheet("color: #006400; font-size: 10px;")
            self.cb_tree_species.setEnabled(True)
            self.cb_tree_species.setText("樹種ポリゴン")
        else:
            self.rinnya_pref_note.setText(f"※ {pref_name}は樹種ポリゴン未対応")
            self.rinnya_pref_note.setStyleSheet("color: #B22222; font-size: 10px;")
            self.cb_tree_species.setEnabled(False)
            self.cb_tree_species.setChecked(False)
            self.cb_tree_species.setText("樹種ポリゴン（未対応）")

        # Update tile checkboxes
        self._update_tile_checkboxes(pref_code)

    def _update_tile_checkboxes(self, pref_code: str):
        """Update tile data checkboxes based on prefecture availability."""
        pref_tiles = self.loader.TILE_URLS.get(pref_code, {})

        if 'dem' in pref_tiles or 'dem_rgb' in pref_tiles:
            self.cb_dem.setText("DEM - 林野庁データ")
        else:
            self.cb_dem.setText("DEM - 国土地理院データ")

        if 'cs_map' in pref_tiles:
            self.cb_cs_map.setText("CS立体図 - 林野庁データ")
        else:
            self.cb_cs_map.setText("CS立体図（陰影起伏図）- 国土地理院")

        if 'slope' in pref_tiles:
            self.cb_slope.setText("傾斜区分図 - 林野庁データ")
        else:
            self.cb_slope.setText("傾斜区分図 - 国土地理院データ")

    def on_forest_plan_pref_changed(self, index):
        """Prefecture changed for forest plan tab."""
        pref_code = self.pref_combo_forest_plan.currentData()
        if not pref_code:
            self.forest_plan_pref_note.setText("")
            return

        plan_info = self.loader.FOREST_PLAN_DATA.get(pref_code, {})
        plan_name = plan_info.get('name', '')
        size_mb = plan_info.get('size_mb', 0)

        self.forest_plan_pref_note.setText(f"※ {plan_name} ({size_mb}MB)")
        self.forest_plan_pref_note.setStyleSheet("color: #006400; font-size: 10px;")

    def on_tree_format_changed(self, index):
        """Tree species format changed - show/hide region selection."""
        tree_format = self.combo_tree_format.currentData()
        if tree_format == 'gpkg':
            self.gpkg_region_group.setVisible(True)
            # Clear previous regions
            self._clear_region_list()
        else:
            self.gpkg_region_group.setVisible(False)

    def _clear_region_list(self):
        """Clear the region checkbox list."""
        for cb in self.region_checkboxes:
            self.region_list_layout.removeWidget(cb)
            cb.deleteLater()
        self.region_checkboxes = []
        self.available_gpkg_urls = []

    def fetch_gpkg_regions(self):
        """Fetch available GeoPackage regions for selected prefecture."""
        pref_code = self.pref_combo_rinnya.currentData()
        if not pref_code:
            QMessageBox.warning(self, "選択エラー", "都道府県を選択してください。")
            return

        # Check if prefecture has tree species data
        available = self.data_manager.get_available_data(pref_code)
        if 'tree_species' not in available:
            QMessageBox.warning(self, "未対応", "この都道府県は樹種ポリゴンに対応していません。")
            return

        self.progress_label.setText("森林計画区一覧を取得中...")
        self.btn_fetch_regions.setEnabled(False)
        self.repaint()

        from qgis.PyQt.QtWidgets import QApplication
        QApplication.processEvents()

        # Fetch region list from loader
        regions = self.loader.get_gpkg_regions(pref_code)

        self.btn_fetch_regions.setEnabled(True)
        self.progress_label.setText("")

        if not regions:
            QMessageBox.warning(self, "取得失敗",
                f"森林計画区一覧を取得できませんでした。\n{self.loader.last_error}")
            return

        # Clear previous list
        self._clear_region_list()

        # Add checkboxes for each region
        for region in regions:
            name = region.get('name', 'Unknown')
            url = region.get('url', '')
            size_mb = region.get('size_mb', 0)

            # Create checkbox with size info
            if size_mb > 0:
                label = f"{name} ({size_mb:.1f}MB)"
            else:
                label = name

            cb = QCheckBox(label)
            cb.setProperty('url', url)
            cb.setProperty('name', name)
            cb.setChecked(True)  # Default checked
            self.region_list_layout.addWidget(cb)
            self.region_checkboxes.append(cb)
            self.available_gpkg_urls.append(url)

        # Add stretch at the end
        self.region_list_layout.addStretch()

        log_info(f"森林計画区一覧取得: {len(regions)}件")

    def select_all_regions(self):
        """Select all region checkboxes."""
        for cb in self.region_checkboxes:
            cb.setChecked(True)

    def select_none_regions(self):
        """Deselect all region checkboxes."""
        for cb in self.region_checkboxes:
            cb.setChecked(False)

    def get_selected_gpkg_urls(self):
        """Get list of selected GeoPackage URLs."""
        selected = []
        for cb in self.region_checkboxes:
            if cb.isChecked():
                url = cb.property('url')
                if url:
                    selected.append(url)
        return selected

    def _get_current_extent_wgs84(self):
        """Get current map extent in WGS84."""
        canvas = self.iface.mapCanvas()
        extent = canvas.extent()
        crs = canvas.mapSettings().destinationCrs()

        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        if crs.authid() != "EPSG:4326":
            transform = QgsCoordinateTransform(crs, wgs84, QgsProject.instance())
            extent = transform.transformBoundingBox(extent)

        return extent

    def _update_progress(self, message: str):
        """Update progress label and process events."""
        from qgis.PyQt.QtWidgets import QApplication
        self.progress_label.setText(message)
        self.repaint()
        QApplication.processEvents()

    def download_rinnya_data(self):
        """Download Forest Agency data."""
        pref_code = self.pref_combo_rinnya.currentData()
        if not pref_code:
            QMessageBox.warning(self, "選択エラー", "都道府県を選択してください。")
            return

        selected = []
        if self.cb_tree_species.isChecked() and self.cb_tree_species.isEnabled():
            selected.append('tree_species')
        if self.cb_dem.isChecked():
            selected.append('dem')
        if self.cb_cs_map.isChecked():
            selected.append('cs_map')
        if self.cb_slope.isChecked():
            selected.append('slope')

        if not selected:
            QMessageBox.warning(self, "選択エラー", "取得するデータを選択してください。")
            return

        extent = self._get_current_extent_wgs84()
        log_info(f"林野庁データ取得: pref={pref_code}, types={selected}")
        log_info(f"範囲: {extent.xMinimum():.4f},{extent.yMinimum():.4f} - {extent.xMaximum():.4f},{extent.yMaximum():.4f}")

        self.btn_download_rinnya.setEnabled(False)
        loaded_layers = []
        errors = []

        # Set progress callback for loader
        self.loader.progress_callback = self._update_progress

        from qgis.PyQt.QtWidgets import QApplication

        for data_type in selected:
            data_name = self.loader.DATA_NAMES.get(data_type, data_type)
            self.progress_label.setText(f"取得中: {data_name}...")
            self.repaint()
            QApplication.processEvents()

            if data_type == 'tree_species':
                tree_format = self.combo_tree_format.currentData()
                # Pass selected region URLs if GeoPackage format and regions selected
                selected_urls = None
                if tree_format == 'gpkg' and self.region_checkboxes:
                    selected_urls = self.get_selected_gpkg_urls()
                    if not selected_urls:
                        errors.append("樹種ポリゴン: 森林計画区が選択されていません")
                        continue
                    log_info(f"選択された森林計画区: {len(selected_urls)}件")
                layer = self.loader.download_rinnya_data(data_type, pref_code, extent, tree_format, selected_urls)
            else:
                layer = self.loader.download_rinnya_data(data_type, pref_code, extent)

            if layer:
                loaded_layers.append(layer.name())
                self._apply_initial_opacity(layer, data_type)
            elif self.loader.last_error:
                errors.append(f"{data_name}: {self.loader.last_error}")

        self.progress_label.setText("")
        self.loader.progress_callback = None  # Clear callback
        self.btn_download_rinnya.setEnabled(True)
        self._show_result(loaded_layers, errors)

    def download_kokudo_data(self):
        """Download National Land Information data."""
        pref_code = self.pref_combo_kokudo.currentData()
        if not pref_code:
            QMessageBox.warning(self, "選択エラー", "都道府県を選択してください。")
            return

        selected = []
        if self.cb_national_forest.isChecked():
            selected.append('national_forest')
        if self.cb_forest_area.isChecked():
            selected.append('forest_area')

        if not selected:
            QMessageBox.warning(self, "選択エラー", "取得するデータを選択してください。")
            return

        extent = self._get_current_extent_wgs84()
        log_info(f"国土数値情報取得: pref={pref_code}, types={selected}")
        log_info(f"範囲: {extent.xMinimum():.4f},{extent.yMinimum():.4f} - {extent.xMaximum():.4f},{extent.yMaximum():.4f}")

        self.btn_download_kokudo.setEnabled(False)
        loaded_layers = []
        errors = []

        from qgis.PyQt.QtWidgets import QApplication

        for data_type in selected:
            data_name = self.loader.DATA_NAMES.get(data_type, data_type)
            self.progress_label.setText(f"取得中: {data_name}...")
            self.repaint()
            QApplication.processEvents()

            layer = self.loader.download_kokudo_data(data_type, pref_code, extent)
            if layer:
                loaded_layers.append(layer.name())
                self._apply_initial_opacity(layer, data_type)
            elif self.loader.last_error:
                errors.append(f"{data_name}: {self.loader.last_error}")

        self.progress_label.setText("")
        self.btn_download_kokudo.setEnabled(True)
        self._show_result(loaded_layers, errors)

    def download_forest_plan_data(self):
        """Download forest planning data (林班 or 小班)."""
        pref_code = self.pref_combo_forest_plan.currentData()
        if not pref_code:
            QMessageBox.warning(self, "選択エラー", "都道府県を選択してください。")
            return

        # Determine selected type from radio button
        if self.rb_forest_rinpan.isChecked():
            data_type = 'forest_plan_rinpan'
        else:
            data_type = 'forest_plan_shohan'

        # Warn about large download
        plan_info = self.loader.FOREST_PLAN_DATA.get(pref_code, {})
        size_mb = plan_info.get('size_mb', 0)
        if size_mb > 100:
            reply = QMessageBox.question(
                self, "ダウンロード確認",
                f"データサイズが約{size_mb}MBあります。\n"
                "初回はダウンロードに時間がかかります。続行しますか？",
                QMessageBox.Yes | QMessageBox.No
            )
            if reply == QMessageBox.No:
                return

        extent = self._get_current_extent_wgs84()
        data_name = self.loader.DATA_NAMES.get(data_type, data_type)
        log_info(f"森林計画図取得: pref={pref_code}, type={data_type}")
        log_info(f"範囲: {extent.xMinimum():.4f},{extent.yMinimum():.4f} - {extent.xMaximum():.4f},{extent.yMaximum():.4f}")

        self.btn_download_forest_plan.setEnabled(False)
        loaded_layers = []
        errors = []

        from qgis.PyQt.QtWidgets import QApplication

        self.progress_label.setText(f"取得中: {data_name}...")
        self.repaint()
        QApplication.processEvents()

        layer = self.loader.download_forest_plan(pref_code, data_type, extent)
        if layer:
            loaded_layers.append(layer.name())
            self._apply_initial_opacity(layer, data_type)
        elif self.loader.last_error:
            errors.append(f"{data_name}: {self.loader.last_error}")

        self.progress_label.setText("")
        self.btn_download_forest_plan.setEnabled(True)
        self._show_result(loaded_layers, errors)

    def download_osm_data(self):
        """Download OSM forest road data."""
        extent = self._get_current_extent_wgs84()

        # Check extent size
        area = extent.width() * extent.height()
        if area > 0.25:
            reply = QMessageBox.question(
                self, "範囲の確認",
                "表示範囲が広いため、データ量が多くなる可能性があります。\n続行しますか？",
                QMessageBox.Yes | QMessageBox.No
            )
            if reply == QMessageBox.No:
                return

        highway_types = []
        if self.cb_osm_track.isChecked():
            highway_types.append('track')
        if self.cb_osm_path.isChecked():
            highway_types.append('path')

        if not highway_types:
            QMessageBox.warning(self, "選択エラー", "取得する道路種別を選択してください。")
            return

        self.progress_label.setText("OSMデータを取得中...")
        self.btn_download_osm.setEnabled(False)
        self.repaint()

        bbox = f"{extent.yMinimum()},{extent.xMinimum()},{extent.yMaximum()},{extent.xMaximum()}"
        highway_filter = '|'.join(highway_types)
        query = f"""
        [out:json][timeout:60];
        (
          way["highway"~"^({highway_filter})$"]({bbox});
        );
        out body;
        >;
        out skel qt;
        """

        try:
            overpass_url = "https://overpass-api.de/api/interpreter"
            log_info(f"OSMデータ取得: bbox={bbox}, types={highway_types}")

            req = QNetworkRequest(QUrl(overpass_url))
            req.setRawHeader(b'User-Agent', b'QGIS Japan Forest Tools/1.0')
            req.setHeader(QNetworkRequest.ContentTypeHeader, 'application/x-www-form-urlencoded')

            blocker = QgsBlockingNetworkRequest()
            err = blocker.post(req, query.encode('utf-8'), forceRefresh=True)

            if err != QgsBlockingNetworkRequest.NoError:
                QMessageBox.warning(self, "取得失敗", f"ネットワークエラー:\n{blocker.errorMessage()}")
                return

            content = bytes(blocker.reply().content())
            data = json.loads(content.decode('utf-8'))

            layer = self._create_osm_layer(data, highway_types)

            if layer and layer.isValid():
                QgsProject.instance().addMapLayer(layer)
                feature_count = layer.featureCount()
                QMessageBox.information(
                    self, "取得完了",
                    f"OSM林道データを取得しました。\nフィーチャ数: {feature_count:,}"
                )
            else:
                QMessageBox.warning(self, "取得失敗", "OSMデータの取得に失敗しました。")

        except Exception as e:
            QMessageBox.warning(self, "取得失敗", f"エラー:\n{str(e)}")

        self.progress_label.setText("")
        self.btn_download_osm.setEnabled(True)

    def _create_osm_layer(self, osm_data, highway_types):
        """Create a memory layer from OSM data."""
        from qgis.core import QgsFeature, QgsGeometry, QgsPointXY, QgsField
        from qgis.PyQt.QtCore import QMetaType

        layer = QgsVectorLayer("LineString?crs=EPSG:4326", "OSM林道", "memory")
        provider = layer.dataProvider()

        provider.addAttributes([
            QgsField(name="osm_id", type=QMetaType.Type.LongLong),
            QgsField(name="highway", type=QMetaType.Type.QString),
            QgsField(name="name", type=QMetaType.Type.QString),
            QgsField(name="surface", type=QMetaType.Type.QString),
            QgsField(name="tracktype", type=QMetaType.Type.QString),
        ])
        layer.updateFields()

        nodes = {}
        for element in osm_data.get('elements', []):
            if element['type'] == 'node':
                nodes[element['id']] = (element['lon'], element['lat'])

        features = []
        for element in osm_data.get('elements', []):
            if element['type'] == 'way':
                tags = element.get('tags', {})
                highway = tags.get('highway', '')

                if highway in highway_types:
                    coords = []
                    for node_id in element.get('nodes', []):
                        if node_id in nodes:
                            coords.append(QgsPointXY(*nodes[node_id]))

                    if len(coords) >= 2:
                        feature = QgsFeature()
                        feature.setGeometry(QgsGeometry.fromPolylineXY(coords))
                        feature.setAttributes([
                            element['id'],
                            highway,
                            tags.get('name', ''),
                            tags.get('surface', ''),
                            tags.get('tracktype', ''),
                        ])
                        features.append(feature)

        provider.addFeatures(features)
        return layer

    def _show_result(self, loaded_layers, errors):
        """Show download result message."""
        if loaded_layers:
            msg = "以下のレイヤを読み込みました:\n" + "\n".join(f"・{name}" for name in loaded_layers)
            if errors:
                msg += "\n\n取得できなかったデータ:\n" + "\n".join(f"・{e}" for e in errors)
            QMessageBox.information(self, "取得完了", msg)
        elif errors:
            QMessageBox.warning(self, "取得失敗", "データの取得に失敗しました:\n" + "\n".join(f"・{e}" for e in errors))
        else:
            QMessageBox.warning(self, "取得失敗", "データの取得に失敗しました。")

    def _apply_opacity(self, data_type: str, opacity_percent: int):
        """Apply opacity to layers matching the data type."""
        opacity = opacity_percent / 100.0

        patterns = {
            'tree_species': ['樹種ポリゴン'],
            'dem': ['DEM_'],
            'cs_map': ['CS立体図'],
            'slope': ['傾斜区分図'],
            'national_forest': ['国有林野'],
            'forest_area': ['森林地域'],
            'forest_plan_rinpan': ['林班'],
            'forest_plan_shohan': ['小班'],
        }

        search_patterns = patterns.get(data_type, [])
        if not search_patterns:
            return

        for layer in QgsProject.instance().mapLayers().values():
            layer_name = layer.name()
            for pattern in search_patterns:
                if pattern in layer_name:
                    layer.setOpacity(opacity)
                    layer.triggerRepaint()
                    break

    def _apply_cs_color(self):
        """Apply color tint to CS立体図 layers."""
        from qgis.PyQt.QtGui import QColor

        color_key = self.combo_cs_color.currentData()

        for layer in QgsProject.instance().mapLayers().values():
            if 'CS立体図' in layer.name() and isinstance(layer, QgsRasterLayer):
                self._set_raster_colorize(layer, color_key)
                layer.triggerRepaint()

    def _set_raster_colorize(self, layer, color_key: str):
        """Set colorize filter on a raster layer."""
        from qgis.PyQt.QtGui import QColor

        hue_filter = layer.hueSaturationFilter()
        if not hue_filter:
            return

        if color_key == 'default':
            hue_filter.setColorizeOn(False)
        else:
            color_map = {
                'red': QColor(255, 60, 60),
                'green': QColor(60, 200, 60),
                'blue': QColor(60, 100, 255),
            }
            color = color_map.get(color_key, QColor(255, 255, 255))
            hue_filter.setColorizeOn(True)
            hue_filter.setColorizeColor(color)
            hue_filter.setColorizeStrength(80)

    def _apply_initial_opacity(self, layer, data_type: str):
        """Apply initial opacity from slider to newly loaded layer."""
        opacity_map = {
            'tree_species': self.spin_tree_opacity.value(),
            'dem': self.spin_dem_opacity.value(),
            'cs_map': self.spin_cs_opacity.value(),
            'slope': self.spin_slope_opacity.value(),
            'national_forest': self.spin_national_forest_opacity.value(),
            'forest_area': self.spin_forest_area_opacity.value(),
            'forest_plan_rinpan': self.spin_rinpan_opacity.value(),
            'forest_plan_shohan': self.spin_shohan_opacity.value(),
        }

        opacity_percent = opacity_map.get(data_type, 100)
        opacity = opacity_percent / 100.0
        layer.setOpacity(opacity)

        # Apply CS立体図 color if applicable
        if data_type == 'cs_map' and isinstance(layer, QgsRasterLayer):
            color_key = self.combo_cs_color.currentData()
            if color_key and color_key != 'default':
                self._set_raster_colorize(layer, color_key)
