# -*- coding: utf-8 -*-
"""
Japan Forest Tools - Forest Data Loader
"""

import os
import json
import shutil
import zipfile
import tempfile
from typing import Optional, List

from qgis.core import QgsMessageLog, Qgis, QgsBlockingNetworkRequest, QgsNetworkAccessManager
from qgis.PyQt.QtNetwork import QNetworkRequest
from qgis.PyQt.QtCore import QUrl, QEventLoop

DEBUG_LOG = False


def log_info(message: str):
    """Info level log."""
    if DEBUG_LOG:
        QgsMessageLog.logMessage(message, "JapanForestTools", Qgis.Info)


def log_error(message: str):
    """Error level log."""
    QgsMessageLog.logMessage(message, "JapanForestTools", Qgis.Warning)


class ForestDataLoader:
    """Load forest-related data from various sources."""

    USER_AGENT = "QGIS Japan Forest Tools/1.0"

    # Forest Agency open data (G-Spatial Information Center)
    # CKAN API URLs
    CKAN_SEARCH_URL = "https://www.geospatial.jp/ckan/api/3/action/package_search"
    CKAN_SHOW_URL = "https://www.geospatial.jp/ckan/api/3/action/package_show"

    # Known dataset IDs for tree species (verified on G-Spatial Information Center)
    # Format: 'pref_code': ('dataset_id', 'priority')
    # Priority: 'gpkg' = direct GeoPackage (best), 'xyz' = vector tiles, '7z'/'zip' = compressed
    KNOWN_DATASETS = {
        '09': ('tree_species_tochigi', 'xyz'),      # 栃木県 - 7z+xyz → use xyz (ライブラリ不要)
        '14': ('rinya-kanagawa-maptiles2', 'gpkg'), # 神奈川県 - gpkg available
        '16': ('tree_species', 'xyz'),              # 富山県 - zip+xyz → use xyz (ライブラリ不要)
        '25': ('rinya-shiga-maptiles', 'gpkg'),     # 滋賀県 - gpkg available
        '26': ('rinya-kyoto-maptiles', 'gpkg'),     # 京都府 - gpkg available (26 files)
        '27': ('rinya-osaka', 'gpkg'),              # 大阪府 - gpkg available
        '28': ('tree_species_hyogo', 'xyz'),        # 兵庫県 - 7z+xyz → use xyz (ライブラリ不要)
        '31': ('tree_species_tottori', 'gpkg'),     # 鳥取県 - gpkg available (17 files)
        '34': ('rinya-hiroshima-maptiles', 'gpkg'), # 広島県 - gpkg available
        '38': ('rinya-ehime-maptiles', 'gpkg'),     # 愛媛県 - gpkg available (also tree_species_ehime)
        '39': ('tree_species_kochi', 'xyz'),        # 高知県 - 7z+xyz → use xyz (ライブラリ不要)
        '42': ('rinya-nagasaki-maptiles', 'gpkg'),  # 長崎県 - gpkg available
    }

    # Prefectures with direct GeoPackage (no library needed, full legend support)
    GPKG_PREFECTURES = ['14', '25', '26', '27', '31', '34', '38', '42']

    # Prefectures using vector tiles (no library needed, limited legend)
    XYZ_PREFECTURES = ['09', '16', '28', '39']

    # Tile data URLs (DEM, CS立体図, 傾斜区分図)
    # Format: {pref_code: {data_type: tile_url}}
    # Note: DCHM tiles are NOT available (404) - only TIF files exist via CKAN
    TILE_URLS = {
        '14': {  # 神奈川県
            'dem': 'https://forestgeo.info/opendata/14_kanagawa/dem_2022/{z}/{x}/{y}.png',
            'dem_rgb': 'https://forestgeo.info/opendata/14_kanagawa/dem_terrainRGB_2022/{z}/{x}/{y}.png',
            'cs_map': 'https://forestgeo.info/opendata/14_kanagawa/csmap_2022/{z}/{x}/{y}.webp',
            'slope': 'https://forestgeo.info/opendata/14_kanagawa/ls_standtype_2020/{z}/{x}/{y}.webp',
        },
        '25': {  # 滋賀県
            'dem': 'https://forestgeo.info/opendata/25_shiga/dem_2023/{z}/{x}/{y}.png',
            'dem_rgb': 'https://forestgeo.info/opendata/25_shiga/dem_terrainRGB_2023/{z}/{x}/{y}.png',
            'cs_map': 'https://forestgeo.info/opendata/25_shiga/csmap_2023/{z}/{x}/{y}.webp',
            'slope': 'https://forestgeo.info/opendata/25_shiga/ls_standtype_2023/{z}/{x}/{y}.webp',
        },
        '26': {  # 京都府
            'dem': 'https://forestgeo.info/opendata/26_kyoto/dem_2024/{z}/{x}/{y}.png',
            'dem_rgb': 'https://forestgeo.info/opendata/26_kyoto/dem_terrainRGB_2024/{z}/{x}/{y}.png',
            'cs_map': 'https://forestgeo.info/opendata/26_kyoto/csmap_2024/{z}/{x}/{y}.webp',
        },
        '38': {  # 愛媛県
            'dem': 'https://forestgeo.info/opendata/38_ehime/dem_2019/{z}/{x}/{y}.png',
            'dem_rgb': 'https://forestgeo.info/opendata/38_ehime/dem_terrainRGB_2019/{z}/{x}/{y}.png',
        },
        '42': {  # 長崎県
            'dem': 'https://forestgeo.info/opendata/42_nagasaki/dem_2022/{z}/{x}/{y}.png',
            'dem_rgb': 'https://forestgeo.info/opendata/42_nagasaki/dem_terrainRGB_2022/{z}/{x}/{y}.png',
            'cs_map': 'https://forestgeo.info/opendata/42_nagasaki/rrim_2022/{z}/{x}/{y}.webp',  # 赤色立体地図
            'slope': 'https://forestgeo.info/opendata/42_nagasaki/ls_standtype_2022/{z}/{x}/{y}.webp',
        },
    }

    # 国土地理院のタイル (全国対応)
    # dem_png: zoom 1-14, slopemap: zoom 3-15, hillshademap: zoom 2-16
    GSI_DEM_URL = 'https://cyberjapandata.gsi.go.jp/xyz/dem_png/{z}/{x}/{y}.png'
    GSI_DEM_ZMAX = 14
    GSI_SLOPE_URL = 'https://cyberjapandata.gsi.go.jp/xyz/slopemap/{z}/{x}/{y}.png'
    GSI_SLOPE_ZMAX = 15
    # CS立体図の代替として陰影起伏図を使用
    GSI_HILLSHADE_URL = 'https://cyberjapandata.gsi.go.jp/xyz/hillshademap/{z}/{x}/{y}.png'
    GSI_HILLSHADE_ZMAX = 16
    # 全タイル共通のズームレベル（オーバーズーム許可）
    TILE_ZMAX = 18

    # Prefecture names for fallback search
    PREF_NAMES_JP = {
        '09': '栃木', '14': '神奈川', '16': '富山', '25': '滋賀',
        '26': '京都', '27': '大阪', '28': '兵庫', '31': '鳥取',
        '34': '広島', '38': '愛媛', '39': '高知', '42': '長崎',
    }

    PREF_NAMES_EN = {
        '09': 'tochigi', '14': 'kanagawa', '16': 'toyama', '25': 'shiga',
        '26': 'kyoto', '27': 'osaka', '28': 'hyogo', '31': 'tottori',
        '34': 'hiroshima', '38': 'ehime', '39': 'kochi', '42': 'nagasaki',
    }

    # 森林計画図（林班・小班）データ - BODIK等から取得
    # Format: {pref_code: {'url': download_url, 'name': display_name, 'size_mb': approx_size}}
    FOREST_PLAN_DATA = {
        '42': {  # 長崎県
            'url': 'https://data.bodik.jp/dataset/ed19e048-3635-4fec-957d-2414f069edfb/resource/7370c4fb-20bd-4e04-bb51-18b654577114/download/shape.zip',
            'name': '長崎県森林計画図',
            'size_mb': 147,
        },
        '43': {  # 熊本県
            'url': 'https://data.bodik.jp/dataset/303306a4-d871-4a49-ab7a-409c0180cd65/resource/647296e5-e422-453a-a229-94d698e1256b/download/339__shp.zip',
            'name': '熊本県森林計画図',
            'size_mb': 511,
        },
        '09': {  # 栃木県（渡良瀬川地域のみ）
            'url': 'https://data.bodik.jp/dataset/c1b82f93-911c-4c21-b1e1-02f349c4e873/resource/c07cd08f-e841-4c35-a3b9-5140a4ba42af/download/33288_2021_keikakuzu-watarase_01015.zip',
            'name': '栃木県森林計画図（渡良瀬川地域）',
            'size_mb': 56,
        },
    }

    # 森林計画図で利用可能な都道府県
    FOREST_PLAN_AVAILABLE = list(FOREST_PLAN_DATA.keys())

    # National Land Information download URLs
    # A45: 国有林野（林班・小班区画含む） (2019年版が最新)
    # A13: 森林地域 (年度コード: 06=2006, 11=2011, 15=2015)
    # N01: 道路 (旧形式gmlold、年度コード07=1995年) - サブディレクトリなし
    KOKUDO_URLS = {
        'national_forest': 'https://nlftp.mlit.go.jp/ksj/gml/data/A45/A45-19/A45-19_{pref_code}_GML.zip',
        'forest_area': 'https://nlftp.mlit.go.jp/ksj/gml/data/A13/A13-15/A13-15_{pref_code}_GML.zip',
        'road_network': 'https://nlftp.mlit.go.jp/ksj/gmlold/data/N01/N01-07L-{pref_code}-01.0a_GML.zip',
    }

    KOKUDO_FALLBACK_URLS = {
        'national_forest': [
            'https://nlftp.mlit.go.jp/ksj/gml/data/A45/A45-18/A45-18_{pref_code}_GML.zip',
        ],
        'forest_area': [
            'https://nlftp.mlit.go.jp/ksj/gml/data/A13/A13-11/A13-11_{pref_code}_GML.zip',
            'https://nlftp.mlit.go.jp/ksj/gml/data/A13/A13-06/A13-06_{pref_code}_GML.zip',
        ],
        'road_network': [
            # 世界測地系版がない場合は日本測地系版を試行
            'https://nlftp.mlit.go.jp/ksj/gmlold/data/N01/N01-07L-{pref_code}-01.0_GML.zip',
        ],
    }

    DATA_NAMES = {
        'national_forest': '国有林野（林班・小班）',
        'forest_area': '森林地域',
        'road_network': '道路',
        'tree_species': '樹種ポリゴン',
        'dem': 'DEM',
        'cs_map': 'CS立体図',
        'slope': '傾斜区分図',
        'forest_plan_rinpan': '林班',
        'forest_plan_shohan': '小班',
    }

    def __init__(self):
        self.cache_dir = self._get_cache_dir()
        self.last_error = ""
        self.progress_callback = None  # Callback for progress updates: func(message: str)

    def _get_cache_dir(self) -> str:
        """Get cache directory path."""
        from qgis.core import QgsApplication
        cache_base = QgsApplication.qgisSettingsDirPath()
        cache_dir = os.path.join(cache_base, "cache", "japan_forest_tools")
        os.makedirs(cache_dir, exist_ok=True)
        return cache_dir

    def _report_progress(self, message: str):
        """Report progress to callback if set."""
        if self.progress_callback:
            self.progress_callback(message)
        # Also process events to keep UI responsive
        self._process_events()

    def _process_events(self):
        """Process Qt events to keep UI responsive during long operations."""
        try:
            from qgis.PyQt.QtWidgets import QApplication
            QApplication.processEvents()
        except:
            pass

    def _make_request(self, url: str) -> QNetworkRequest:
        """Create a QNetworkRequest with User-Agent header."""
        req = QNetworkRequest(QUrl(url))
        req.setRawHeader(b'User-Agent', self.USER_AGENT.encode())
        return req

    def _fetch_bytes(self, url: str) -> Optional[bytes]:
        """Fetch URL content as bytes using QgsBlockingNetworkRequest."""
        req = self._make_request(url)
        blocker = QgsBlockingNetworkRequest()
        err = blocker.get(req, forceRefresh=True)
        if err != QgsBlockingNetworkRequest.NoError:
            self.last_error = blocker.errorMessage()
            log_error(f"ネットワークエラー: {self.last_error} ({url})")
            return None
        return bytes(blocker.reply().content())

    def _fetch_json(self, url: str) -> Optional[dict]:
        """Fetch JSON from URL using QgsBlockingNetworkRequest."""
        data = self._fetch_bytes(url)
        if data is None:
            return None
        try:
            return json.loads(data.decode('utf-8'))
        except (json.JSONDecodeError, UnicodeDecodeError) as e:
            self.last_error = f"JSONパースエラー: {e}"
            log_error(self.last_error)
            return None

    def _download_to_path(self, url: str, file_path: str,
                          progress_prefix: str = "") -> int:
        """Download URL to file using QgsNetworkAccessManager with progress.

        Returns bytes downloaded, or 0 on failure.
        """
        nam = QgsNetworkAccessManager.instance()
        req = self._make_request(url)
        reply = nam.get(req)

        loop = QEventLoop()
        reply.finished.connect(loop.quit)

        if self.progress_callback and progress_prefix:
            def on_progress(received, total):
                mb = received / (1024 * 1024)
                if total > 0:
                    pct = received * 100 // total
                    self._report_progress(f"{progress_prefix} {mb:.1f}MB ({pct}%)")
                else:
                    self._report_progress(f"{progress_prefix} {mb:.1f}MB")
            reply.downloadProgress.connect(on_progress)

        loop.exec_()

        if reply.error():
            self.last_error = reply.errorString()
            log_error(f"ダウンロードエラー: {self.last_error} ({url})")
            reply.deleteLater()
            return 0

        data = reply.readAll()
        reply.deleteLater()

        with open(file_path, 'wb') as f:
            f.write(bytes(data))

        return len(data)

    def get_gpkg_regions(self, pref_code: str) -> list:
        """Get list of available GeoPackage regions for a prefecture.

        Args:
            pref_code: Prefecture code (2 digits)

        Returns:
            List of dicts: [{'name': str, 'url': str, 'size_mb': float}, ...]
        """
        import json

        self.last_error = ""

        if pref_code not in self.KNOWN_DATASETS:
            self.last_error = "この都道府県はG空間情報センターにデータセットがありません"
            return []

        dataset_info = self.KNOWN_DATASETS[pref_code]
        dataset_id = dataset_info[0] if isinstance(dataset_info, tuple) else dataset_info

        api_url = f"{self.CKAN_SHOW_URL}?id={dataset_id}"
        log_info(f"CKAN API呼び出し (森林計画区一覧): {api_url}")

        data = self._fetch_json(api_url)
        if data is None:
            return []

        if not data.get('success'):
            self.last_error = "CKAN APIエラー"
            return []

        resources = data.get('result', {}).get('resources', [])
        regions = []

        for resource in resources:
            url = resource.get('url', '')
            name = resource.get('name', '')

            # Only include direct GeoPackage files
            if url and url.endswith('.gpkg'):
                size_bytes = resource.get('size', 0)
                size_mb = size_bytes / (1024 * 1024) if size_bytes else 0

                display_name = name
                if not display_name:
                    display_name = url.split('/')[-1].replace('.gpkg', '')

                regions.append({
                    'name': display_name,
                    'url': url,
                    'size_mb': size_mb
                })

        log_info(f"GeoPackage一覧取得: {len(regions)}件")
        return regions

    def download_kokudo_data(self, data_type: str, pref_code: str, extent=None):
        """
        Download and load national land information data.

        Args:
            data_type: Type of data ('national_forest', 'forest_area', 'road_network')
            pref_code: Prefecture code (2 digits)
            extent: Optional QgsRectangle to filter data (in WGS84)

        Returns:
            QgsVectorLayer or None
        """
        from qgis.core import QgsVectorLayer, QgsProject

        log_info(f"国土数値情報取得開始: type={data_type}, pref={pref_code}")

        # Check cache first
        cache_path = self._get_cached_kokudo(data_type, pref_code)
        if cache_path:
            log_info(f"キャッシュ使用: {cache_path}")
            layer_name = f"{self.DATA_NAMES.get(data_type, data_type)}_{pref_code}"
            layer = QgsVectorLayer(cache_path, layer_name, "ogr")
            if layer.isValid():
                # Apply extent filter if provided
                if extent:
                    layer = self._apply_extent_filter(layer, extent, layer_name)
                    if layer:
                        QgsProject.instance().addMapLayer(layer)
                        return layer
                    else:
                        # Cache is valid but no data in current extent - don't re-download
                        log_info(f"キャッシュは有効ですが、表示範囲にデータがありません")
                        return None
                else:
                    # No extent filter, just add the layer
                    QgsProject.instance().addMapLayer(layer)
                    return layer

        # Try to download (only if no cache exists)
        urls_to_try = [self.KOKUDO_URLS.get(data_type, '').format(pref_code=pref_code)]
        urls_to_try.extend([
            url.format(pref_code=pref_code)
            for url in self.KOKUDO_FALLBACK_URLS.get(data_type, [])
        ])

        for url in urls_to_try:
            if not url:
                continue

            log_info(f"URL試行: {url}")
            try:
                zip_path = self._download_file(url)
                if zip_path:
                    log_info(f"ダウンロード成功、展開中...")
                    extracted = self._extract_kokudo_zip(zip_path, data_type, pref_code)
                    os.remove(zip_path)

                    if extracted:
                        log_info(f"展開完了: {extracted}")
                        layer_name = f"{self.DATA_NAMES.get(data_type, data_type)}_{pref_code}"
                        layer = QgsVectorLayer(extracted, layer_name, "ogr")
                        if layer.isValid():
                            # Apply extent filter if provided
                            if extent:
                                layer = self._apply_extent_filter(layer, extent, layer_name)
                            if layer:
                                QgsProject.instance().addMapLayer(layer)
                                return layer
            except Exception as e:
                log_error(f"エラー: {str(e)} - {url}")
                self.last_error = str(e)
                continue

        log_error(f"国土数値情報取得失敗: {self.last_error}")
        return None

    def download_forest_plan(self, pref_code: str, data_type: str, extent=None):
        """
        Download forest planning data (林班・小班) from BODIK etc.

        Args:
            pref_code: Prefecture code (2 digits)
            data_type: 'forest_plan_rinpan' (林班) or 'forest_plan_shohan' (小班)
            extent: Optional QgsRectangle to filter data (in WGS84)

        Returns:
            QgsVectorLayer or None
        """
        from qgis.core import (
            QgsVectorLayer, QgsProject, QgsFeatureRequest,
            QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsRectangle
        )

        log_info(f"森林計画図取得開始: pref={pref_code}, type={data_type}")

        if pref_code not in self.FOREST_PLAN_DATA:
            self.last_error = "この都道府県の森林計画図データは未対応です"
            log_error(self.last_error)
            return None

        plan_info = self.FOREST_PLAN_DATA[pref_code]
        url = plan_info['url']
        plan_name = plan_info['name']
        size_mb = plan_info['size_mb']

        log_info(f"データ: {plan_name} ({size_mb}MB)")

        # Check cache
        cache_dir = os.path.join(self.cache_dir, 'forest_plan', pref_code)
        os.makedirs(cache_dir, exist_ok=True)

        # Find all cached shapefiles
        shp_files = self._find_shapefiles(cache_dir)

        # Check if cache is corrupted (contains garbled filenames)
        if shp_files:
            cache_valid = self._validate_cache(cache_dir)
            if not cache_valid:
                log_info("キャッシュが破損しています。再ダウンロードします...")
                shutil.rmtree(cache_dir)
                os.makedirs(cache_dir, exist_ok=True)
                shp_files = []

        if not shp_files:
            # Download and extract
            log_info(f"ダウンロード開始: {url}")
            try:
                zip_path = self._download_file(url)
                if zip_path:
                    log_info(f"ダウンロード完了、展開中...")
                    # Extract ZIP with proper encoding
                    self._extract_zip_with_encoding(zip_path, cache_dir)
                    os.remove(zip_path)

                    # Find shapefiles again
                    shp_files = self._find_shapefiles(cache_dir)

                    if shp_files:
                        log_info(f"展開完了: {len(shp_files)}個のShapefile")
                    else:
                        self.last_error = "ZIPファイルにShapefileが含まれていません"
                        return None
            except Exception as e:
                self.last_error = str(e)
                log_error(f"エラー: {self.last_error}")
                return None

        if not shp_files:
            self.last_error = "Shapefileが見つかりません"
            return None

        log_info(f"全Shapefile数: {len(shp_files)}")

        # Filter shapefiles by type (小班 or 林班)
        # BODIK data has separate files: 市町村名_小班.shp / 市町村名_林班.shp
        layer_suffix = '林班' if data_type == 'forest_plan_rinpan' else '小班'
        layer_name = f"{plan_name}_{layer_suffix}"

        filtered_files = self._filter_shapefiles_by_type(shp_files, data_type)
        log_info(f"フィルター後Shapefile数: {len(filtered_files)} ({layer_suffix})")

        if not filtered_files:
            # Fallback: if no matching files, use all files
            log_info("ファイル名フィルターに一致するファイルがないため、全ファイルを使用します")
            filtered_files = shp_files

        # Log data structure for debugging
        if filtered_files:
            self._log_shapefile_info(filtered_files[0])

        log_info(f"データタイプ: {data_type}, レイヤ名: {layer_name}")

        # For 林班 with separate files, no dissolve needed
        # For 林班 without separate files, dissolve by 林班番号
        dissolve_field = None
        if data_type == 'forest_plan_rinpan' and filtered_files == shp_files:
            # No separate 林班 files found, try dissolving
            dissolve_field = self._find_rinpan_field(filtered_files)
            if dissolve_field:
                log_info(f"林班番号フィールド発見: {dissolve_field} - 林班単位で集約します")
            else:
                log_info("林班番号フィールドが見つかりません")
        else:
            log_info(f"{layer_suffix}モード: 対応ファイルのみ読み込み")

        return self._merge_shapefiles_with_extent(filtered_files, layer_name, extent, dissolve_field)

    def download_forest_plan_both(self, pref_code: str, extent=None):
        """
        Download forest planning data and create both 林班 and 小班 layers efficiently.

        Args:
            pref_code: Prefecture code (2 digits)
            extent: Optional QgsRectangle to filter data (in WGS84)

        Returns:
            Dict with 'forest_plan_rinpan' and 'forest_plan_shohan' layers
        """
        from qgis.core import (
            QgsVectorLayer, QgsProject, QgsFeatureRequest, QgsFeature,
            QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsRectangle
        )

        log_info(f"森林計画図取得（林班・小班両方）: pref={pref_code}")

        if pref_code not in self.FOREST_PLAN_DATA:
            self.last_error = "この都道府県の森林計画図データは未対応です"
            return {}

        plan_info = self.FOREST_PLAN_DATA[pref_code]
        url = plan_info['url']
        plan_name = plan_info['name']

        # Check/download cache (same as download_forest_plan)
        cache_dir = os.path.join(self.cache_dir, 'forest_plan', pref_code)
        os.makedirs(cache_dir, exist_ok=True)

        shp_files = self._find_shapefiles(cache_dir)

        if shp_files:
            cache_valid = self._validate_cache(cache_dir)
            if not cache_valid:
                log_info("キャッシュが破損しています。再ダウンロードします...")
                shutil.rmtree(cache_dir)
                os.makedirs(cache_dir, exist_ok=True)
                shp_files = []

        if not shp_files:
            log_info(f"ダウンロード開始: {url}")
            try:
                zip_path = self._download_file(url)
                if zip_path:
                    log_info(f"ダウンロード完了、展開中...")
                    self._extract_zip_with_encoding(zip_path, cache_dir)
                    os.remove(zip_path)
                    shp_files = self._find_shapefiles(cache_dir)
            except Exception as e:
                self.last_error = str(e)
                log_error(f"エラー: {self.last_error}")
                return {}

        if not shp_files:
            self.last_error = "Shapefileが見つかりません"
            return {}

        log_info(f"全Shapefile数: {len(shp_files)}")

        # Filter shapefiles by type
        shohan_files = self._filter_shapefiles_by_type(shp_files, 'forest_plan_shohan')
        rinpan_files = self._filter_shapefiles_by_type(shp_files, 'forest_plan_rinpan')

        log_info(f"小班ファイル数: {len(shohan_files)}, 林班ファイル数: {len(rinpan_files)}")

        result = {}

        # Create 小班 layer
        log_info("小班レイヤ作成中...")
        shohan_features, shohan_fields, shohan_crs = self._read_features_with_extent(shohan_files, extent)
        if shohan_features:
            log_info(f"小班フィーチャ数: {len(shohan_features)}")
            shohan_layer = self._create_layer_from_features(
                shohan_features, shohan_fields, shohan_crs, f"{plan_name}_小班"
            )
            if shohan_layer:
                self._apply_forest_plan_style(shohan_layer, False)
                QgsProject.instance().addMapLayer(shohan_layer)
                result['forest_plan_shohan'] = shohan_layer

        # Create 林班 layer
        log_info("林班レイヤ作成中...")
        rinpan_features, rinpan_fields, rinpan_crs = self._read_features_with_extent(rinpan_files, extent)
        if rinpan_features:
            log_info(f"林班フィーチャ数: {len(rinpan_features)}")
            rinpan_layer = self._create_layer_from_features(
                rinpan_features, rinpan_fields, rinpan_crs, f"{plan_name}_林班"
            )
            if rinpan_layer:
                self._apply_forest_plan_style(rinpan_layer, True)
                QgsProject.instance().addMapLayer(rinpan_layer)
                result['forest_plan_rinpan'] = rinpan_layer

        if not result:
            self.last_error = "表示範囲にデータがありません"

        return result

    def _read_features_with_extent(self, shp_files: list, extent):
        """Read features from shapefiles within extent.

        Returns:
            Tuple of (features, fields, crs)
        """
        from qgis.core import (
            QgsVectorLayer, QgsProject, QgsFeatureRequest, QgsFeature,
            QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsRectangle
        )

        all_features = []
        fields = None
        layer_crs = None
        max_features = 10000

        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        filter_extent = None

        if extent:
            log_info(f"範囲フィルター: {extent.xMinimum():.4f},{extent.yMinimum():.4f} - {extent.xMaximum():.4f},{extent.yMaximum():.4f}")

        for shp_path in shp_files:
            try:
                layer = QgsVectorLayer(shp_path, "temp", "ogr")
                if not layer.isValid():
                    continue

                if layer_crs is None:
                    layer_crs = layer.crs()
                    fields = layer.fields()

                    if extent and layer_crs.authid() != "EPSG:4326":
                        transform = QgsCoordinateTransform(wgs84, layer_crs, QgsProject.instance())
                        filter_extent = transform.transformBoundingBox(extent)
                    else:
                        filter_extent = extent

                    if filter_extent:
                        buffer = (filter_extent.width() + filter_extent.height()) * 0.05
                        filter_extent = QgsRectangle(
                            filter_extent.xMinimum() - buffer,
                            filter_extent.yMinimum() - buffer,
                            filter_extent.xMaximum() + buffer,
                            filter_extent.yMaximum() + buffer
                        )

                if filter_extent:
                    request = QgsFeatureRequest().setFilterRect(filter_extent)
                    features = [QgsFeature(f) for f in layer.getFeatures(request)]
                else:
                    features = [QgsFeature(f) for f in layer.getFeatures()]

                all_features.extend(features)
                self._process_events()

                if len(all_features) > max_features:
                    log_info(f"フィーチャ数制限 ({max_features}) に達しました")
                    break

            except Exception as e:
                log_error(f"読み込みエラー: {str(e)}")
                continue

        return all_features, fields, layer_crs

    def _create_layer_from_features(self, features, fields, crs, layer_name):
        """Create a memory layer from features."""
        from qgis.core import QgsVectorLayer

        if not features or not crs:
            return None

        uri = f"Polygon?crs={crs.authid()}"
        mem_layer = QgsVectorLayer(uri, layer_name, "memory")
        mem_provider = mem_layer.dataProvider()

        if fields:
            mem_provider.addAttributes(fields.toList())
            mem_layer.updateFields()

        mem_provider.addFeatures(features)
        mem_layer.updateExtents()

        return mem_layer

    def _log_shapefile_info(self, shp_path: str):
        """Log shapefile structure for debugging."""
        from qgis.core import QgsVectorLayer

        try:
            layer = QgsVectorLayer(shp_path, "temp", "ogr")
            if not layer.isValid():
                return

            feature_count = layer.featureCount()
            field_names = [f.name() for f in layer.fields()]

            log_info(f"=== Shapefile情報 ===")
            log_info(f"ファイル: {os.path.basename(shp_path)}")
            log_info(f"総フィーチャ数: {feature_count}")
            log_info(f"CRS: {layer.crs().authid()}")
            log_info(f"全フィールド: {field_names}")

            # Check for 林班/小班 related fields
            rinpan_fields = [f for f in field_names if '林班' in f]
            shohan_fields = [f for f in field_names if '小班' in f]
            log_info(f"林班関連フィールド: {rinpan_fields if rinpan_fields else 'なし'}")
            log_info(f"小班関連フィールド: {shohan_fields if shohan_fields else 'なし'}")

            # Check sample features
            sample_count = min(5, feature_count)
            for i, feat in enumerate(layer.getFeatures()):
                if i >= sample_count:
                    break
                geom = feat.geometry()
                attrs = feat.attributes()

                # Get 林班/小班 values
                rinpan_val = ''
                shohan_val = ''
                for fn in rinpan_fields:
                    idx = field_names.index(fn)
                    if idx < len(attrs):
                        rinpan_val = f"{fn}={attrs[idx]}"
                for fn in shohan_fields:
                    idx = field_names.index(fn)
                    if idx < len(attrs):
                        shohan_val = f"{fn}={attrs[idx]}"

                area_info = f"面積={geom.area():.2f}" if geom and not geom.isEmpty() else "ジオメトリなし"
                log_info(f"  サンプル{i+1}: {rinpan_val} {shohan_val} {area_info}")

            log_info(f"=== Shapefile情報終了 ===")

        except Exception as e:
            log_error(f"Shapefile情報取得エラー: {str(e)}")

    def _find_shapefiles(self, directory: str) -> list:
        """Find all shapefiles in directory recursively."""
        shp_files = []
        if not os.path.exists(directory):
            return shp_files

        for root, dirs, files in os.walk(directory):
            for f in files:
                if f.lower().endswith('.shp'):
                    shp_files.append(os.path.join(root, f))
        return shp_files

    def _filter_shapefiles_by_type(self, shp_files: list, data_type: str) -> list:
        """Filter shapefiles by data type based on filename.

        BODIK data has separate files for 小班 and 林班:
            市町村名_小班.shp
            市町村名_林班.shp

        Args:
            shp_files: List of all shapefile paths
            data_type: 'forest_plan_rinpan' or 'forest_plan_shohan'

        Returns:
            Filtered list of shapefile paths
        """
        if data_type == 'forest_plan_rinpan':
            keyword = '林班'
            exclude = '小班'
        else:
            keyword = '小班'
            exclude = '林班'

        # Check if files have 林班/小班 in their names
        has_separate_files = any(keyword in os.path.basename(f) for f in shp_files)

        if not has_separate_files:
            log_info(f"ファイル名に'{keyword}'を含むファイルが見つかりません")
            return shp_files

        # Filter: include files with keyword, exclude files with the other type
        filtered = []
        for f in shp_files:
            basename = os.path.basename(f)
            if keyword in basename:
                filtered.append(f)
                log_info(f"  採用: {basename}")
            elif exclude in basename:
                log_info(f"  除外: {basename}")
            else:
                # File without 林班/小班 in name - include it
                filtered.append(f)
                log_info(f"  採用(その他): {basename}")

        return filtered

    def _find_rinpan_field(self, shp_files: list) -> str:
        """Find the 林班番号 field name from shapefile."""
        from qgis.core import QgsVectorLayer

        # Common field names for 林班番号
        possible_names = [
            '林班番号', '林班', 'RINPAN', '林班NO', '林班No',
            '林班ＮＯ', 'RINBAN', '林班コード', '林班CD',
            '林班名', 'COMPARTMENT', '林小班', '林班番'
        ]

        for shp_path in shp_files[:3]:  # Check first 3 files
            try:
                layer = QgsVectorLayer(shp_path, "temp", "ogr")
                if not layer.isValid():
                    continue

                field_names = [f.name() for f in layer.fields()]
                log_info(f"利用可能なフィールド: {field_names[:15]}")

                # Check for exact match
                for name in possible_names:
                    if name in field_names:
                        return name

                # Check for partial match (case insensitive)
                for field in field_names:
                    field_lower = field.lower()
                    if '林班' in field and '小班' not in field:
                        return field
                    if 'rinpan' in field_lower or 'rinban' in field_lower:
                        return field

            except Exception as e:
                log_error(f"フィールド検索エラー: {str(e)}")
                continue

        return None

    def _validate_cache(self, cache_dir: str) -> bool:
        """Check if cache directory contains valid (non-corrupted) files."""
        # Look for signs of corrupted Japanese filenames
        # Corrupted names often contain characters like Æ, ╖, ì, Φ, etc.
        corrupted_chars = ['Æ', '╖', 'ì', 'Φ', 'Ä', 'ö', '╟']

        for root, dirs, files in os.walk(cache_dir):
            # Check directory names
            for d in dirs:
                for char in corrupted_chars:
                    if char in d:
                        log_info(f"破損したディレクトリ名を検出: {d}")
                        return False
            # Check file names
            for f in files:
                for char in corrupted_chars:
                    if char in f:
                        log_info(f"破損したファイル名を検出: {f}")
                        return False

        return True

    def _extract_zip_with_encoding(self, zip_path: str, extract_dir: str):
        """Extract ZIP file with proper Japanese encoding handling."""
        with zipfile.ZipFile(zip_path, 'r') as zf:
            for member in zf.namelist():
                # Try to fix Japanese encoding (CP932/Shift-JIS)
                try:
                    # Try to decode as CP437 (default) then encode as CP932
                    fixed_name = member.encode('cp437').decode('cp932')
                except (UnicodeDecodeError, UnicodeEncodeError):
                    try:
                        fixed_name = member.encode('utf-8').decode('utf-8')
                    except:
                        fixed_name = member

                # Create safe path
                target_path = os.path.join(extract_dir, fixed_name)

                # Create directory if needed
                if member.endswith('/'):
                    os.makedirs(target_path, exist_ok=True)
                else:
                    os.makedirs(os.path.dirname(target_path), exist_ok=True)
                    # Extract file
                    with zf.open(member) as source:
                        with open(target_path, 'wb') as target:
                            shutil.copyfileobj(source, target)

    def _merge_shapefiles_with_extent(self, shp_files: list, layer_name: str, extent, dissolve_field: str = None):
        """Merge multiple shapefiles and filter by extent.

        Args:
            shp_files: List of shapefile paths
            layer_name: Name for the output layer
            extent: QgsRectangle to filter features (in WGS84)
            dissolve_field: If provided, dissolve polygons by this field
        """
        from qgis.core import (
            QgsVectorLayer, QgsProject, QgsFeatureRequest, QgsFeature,
            QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsRectangle,
            QgsField, QgsGeometry
        )

        all_features = []
        fields = None
        layer_crs = None
        max_features = 10000  # Limit to prevent freeze

        # Transform extent to common CRS if provided
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        filter_extent = None

        # Log extent info
        if extent:
            log_info(f"範囲フィルター適用: {extent.xMinimum():.4f},{extent.yMinimum():.4f} - {extent.xMaximum():.4f},{extent.yMaximum():.4f}")
        else:
            log_info("警告: 範囲フィルターなし - 全データを読み込みます")

        for shp_idx, shp_path in enumerate(shp_files):
            try:
                layer = QgsVectorLayer(shp_path, "temp", "ogr")
                if not layer.isValid():
                    continue

                total_in_file = layer.featureCount()
                log_info(f"Shapefile {shp_idx+1}/{len(shp_files)}: {os.path.basename(shp_path)} (総数: {total_in_file})")

                # Get CRS from first valid layer
                if layer_crs is None:
                    layer_crs = layer.crs()
                    fields = layer.fields()
                    log_info(f"CRS: {layer_crs.authid()}")

                    # Transform extent to layer CRS
                    if extent and layer_crs.authid() != "EPSG:4326":
                        transform = QgsCoordinateTransform(wgs84, layer_crs, QgsProject.instance())
                        filter_extent = transform.transformBoundingBox(extent)
                        log_info(f"変換後範囲: {filter_extent.xMinimum():.2f},{filter_extent.yMinimum():.2f} - {filter_extent.xMaximum():.2f},{filter_extent.yMaximum():.2f}")
                    else:
                        filter_extent = extent

                    # Add small buffer (5%)
                    if filter_extent:
                        buffer = (filter_extent.width() + filter_extent.height()) * 0.05
                        filter_extent = QgsRectangle(
                            filter_extent.xMinimum() - buffer,
                            filter_extent.yMinimum() - buffer,
                            filter_extent.xMaximum() + buffer,
                            filter_extent.yMaximum() + buffer
                        )

                # Get features in extent
                if filter_extent:
                    request = QgsFeatureRequest().setFilterRect(filter_extent)
                    features = list(layer.getFeatures(request))
                    log_info(f"  範囲内フィーチャ: {len(features)}/{total_in_file}")
                else:
                    features = list(layer.getFeatures())
                    log_info(f"  フィーチャ: {len(features)}")

                all_features.extend(features)

                # Process events to prevent freeze
                self._process_events()

                # Limit total features to prevent freeze
                if len(all_features) > max_features:
                    log_info(f"フィーチャ数制限 ({max_features}) に達しました。表示範囲を小さくしてください。")
                    break

            except Exception as e:
                log_error(f"Shapefile読み込みエラー: {shp_path} - {str(e)}")
                continue

        if not all_features:
            self.last_error = "表示範囲にデータがありません"
            return None

        log_info(f"合計フィーチャ数（集約前）: {len(all_features)}")

        # Dissolve by field if specified
        if dissolve_field and fields:
            all_features, fields = self._dissolve_features(all_features, fields, dissolve_field)
            log_info(f"集約後フィーチャ数: {len(all_features)}")

        # Create memory layer
        geom_type = 'Polygon'
        uri = f"{geom_type}?crs={layer_crs.authid()}"
        mem_layer = QgsVectorLayer(uri, layer_name, "memory")
        mem_provider = mem_layer.dataProvider()

        if fields:
            mem_provider.addAttributes(fields.toList())
            mem_layer.updateFields()

        mem_provider.addFeatures(all_features)
        mem_layer.updateExtents()

        # Apply styling with visible borders
        self._apply_forest_plan_style(mem_layer, dissolve_field is not None)

        QgsProject.instance().addMapLayer(mem_layer)
        return mem_layer

    def _dissolve_features(self, features: list, fields, dissolve_field: str):
        """Dissolve features by a field value.

        Args:
            features: List of QgsFeature
            fields: QgsFields object
            dissolve_field: Field name to dissolve by

        Returns:
            Tuple of (dissolved_features, new_fields)
        """
        from qgis.core import QgsFeature, QgsGeometry, QgsFields, QgsField
        from qgis.PyQt.QtCore import QMetaType

        log_info(f"林班番号で集約中: {dissolve_field}")

        # Group features by dissolve field value
        groups = {}
        dissolve_idx = None

        # Find field index
        for i, field in enumerate(fields):
            if field.name() == dissolve_field:
                dissolve_idx = i
                break

        if dissolve_idx is None:
            log_error(f"集約フィールドが見つかりません: {dissolve_field}")
            return features, fields

        # Group features
        for feat in features:
            attrs = feat.attributes()
            if dissolve_idx < len(attrs):
                key = str(attrs[dissolve_idx]) if attrs[dissolve_idx] is not None else ''
            else:
                key = ''

            if key not in groups:
                groups[key] = []
            groups[key].append(feat)

        log_info(f"グループ数: {len(groups)}")

        # Create new fields (just the dissolve field)
        new_fields = QgsFields()
        new_fields.append(QgsField(dissolve_field, QMetaType.Type.QString))

        # Dissolve each group
        dissolved_features = []
        for key, group_features in groups.items():
            if not group_features:
                continue

            # Combine geometries
            combined_geom = None
            for feat in group_features:
                geom = feat.geometry()
                if geom.isNull() or geom.isEmpty():
                    continue

                if combined_geom is None:
                    combined_geom = QgsGeometry(geom)
                else:
                    combined_geom = combined_geom.combine(geom)

            if combined_geom and not combined_geom.isNull():
                new_feat = QgsFeature()
                new_feat.setGeometry(combined_geom)
                new_feat.setAttributes([key])
                dissolved_features.append(new_feat)

        return dissolved_features, new_fields

    def _apply_forest_plan_style(self, layer, is_rinpan: bool):
        """Apply styling to forest plan layer with visible borders.

        Args:
            layer: QgsVectorLayer to style
            is_rinpan: True if this is 林班 (use thicker border), False for 小班
        """
        try:
            from qgis.core import (
                QgsFillSymbol, QgsSimpleFillSymbolLayer,
                QgsSingleSymbolRenderer
            )
            from qgis.PyQt.QtGui import QColor

            if is_rinpan:
                # 林班: Thicker border, semi-transparent fill
                symbol = QgsFillSymbol.createSimple({
                    'color': '50,150,50,100',  # Semi-transparent green
                    'outline_color': '#1B5E20',  # Dark green border
                    'outline_width': '1.5',  # Thick border
                    'outline_style': 'solid'
                })
                log_info("林班スタイル作成: 緑塗りつぶし、太い緑枠")
            else:
                # 小班: No fill, only visible borders to show detailed boundaries
                symbol = QgsFillSymbol.createSimple({
                    'color': '0,0,0,0',  # Fully transparent (no fill)
                    'outline_color': '#D32F2F',  # Red border for visibility
                    'outline_width': '0.5',  # Medium border
                    'outline_style': 'solid'
                })
                log_info("小班スタイル作成: 塗りつぶしなし、赤枠のみ")

            renderer = QgsSingleSymbolRenderer(symbol)
            layer.setRenderer(renderer)
            layer.triggerRepaint()
            log_info(f"森林計画図スタイル適用完了: {'林班' if is_rinpan else '小班'}")

        except Exception as e:
            log_error(f"スタイル適用エラー: {str(e)}")
            import traceback
            log_error(traceback.format_exc())

    def _apply_extent_filter(self, layer, extent, layer_name: str):
        """Apply spatial filter to layer and create memory layer with filtered features."""
        from qgis.core import (
            QgsVectorLayer, QgsFeatureRequest, QgsCoordinateReferenceSystem,
            QgsCoordinateTransform, QgsProject, QgsRectangle, QgsFeature
        )

        log_info(f"範囲フィルター適用: {layer_name}")

        # Transform extent to layer CRS if needed
        layer_crs = layer.crs()
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")

        filter_extent = extent
        if layer_crs.authid() != "EPSG:4326":
            transform = QgsCoordinateTransform(wgs84, layer_crs, QgsProject.instance())
            filter_extent = transform.transformBoundingBox(extent)

        # Add small buffer to extent
        buffer = (filter_extent.width() + filter_extent.height()) * 0.05
        extent_with_buffer = QgsRectangle(
            filter_extent.xMinimum() - buffer,
            filter_extent.yMinimum() - buffer,
            filter_extent.xMaximum() + buffer,
            filter_extent.yMaximum() + buffer
        )

        # Get features in extent
        request = QgsFeatureRequest().setFilterRect(extent_with_buffer)
        features = list(layer.getFeatures(request))
        feature_count = len(features)

        log_info(f"範囲内フィーチャ数: {feature_count}")

        if feature_count == 0:
            self.last_error = "表示範囲にデータがありません"
            return None

        # Create memory layer with filtered features (prevents freeze with large datasets)
        geom_type = layer.geometryType()
        geom_type_str = ['Point', 'LineString', 'Polygon'][geom_type] if geom_type < 3 else 'Polygon'

        uri = f"{geom_type_str}?crs={layer_crs.authid()}"
        mem_layer = QgsVectorLayer(uri, layer_name, "memory")
        mem_provider = mem_layer.dataProvider()

        # Copy fields
        mem_provider.addAttributes(layer.fields().toList())
        mem_layer.updateFields()

        # Add filtered features
        mem_provider.addFeatures(features)
        mem_layer.updateExtents()

        log_info(f"メモリレイヤ作成: {feature_count}フィーチャ")

        return mem_layer

    def _filter_layer_by_extent(self, layer, extent, layer_name: str):
        """Filter layer by extent and return a memory layer with filtered features."""
        from qgis.core import (
            QgsVectorLayer, QgsFeatureRequest, QgsCoordinateReferenceSystem,
            QgsCoordinateTransform, QgsProject, QgsRectangle,
            QgsFeature, QgsField, QgsFields
        )

        log_info(f"範囲フィルター適用中...")

        # Transform extent to layer CRS if needed
        layer_crs = layer.crs()
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        filter_extent = extent

        if layer_crs.authid() != "EPSG:4326":
            transform = QgsCoordinateTransform(wgs84, layer_crs, QgsProject.instance())
            filter_extent = transform.transformBoundingBox(extent)

        # Add small buffer
        buffer = (filter_extent.width() + filter_extent.height()) * 0.05
        filter_extent = QgsRectangle(
            filter_extent.xMinimum() - buffer,
            filter_extent.yMinimum() - buffer,
            filter_extent.xMaximum() + buffer,
            filter_extent.yMaximum() + buffer
        )

        # Create request with spatial filter
        request = QgsFeatureRequest().setFilterRect(filter_extent)

        # Get filtered features
        features = list(layer.getFeatures(request))
        feature_count = len(features)
        log_info(f"フィルター後のフィーチャ数: {feature_count}")

        if feature_count == 0:
            self.last_error = "表示範囲にデータがありません"
            log_error(self.last_error)
            return None

        # Create memory layer with filtered features
        geom_type = layer.geometryType()
        geom_type_str = ['Point', 'LineString', 'Polygon'][geom_type] if geom_type < 3 else 'Polygon'

        # Build memory layer URI with fields
        uri = f"{geom_type_str}?crs={layer_crs.authid()}"
        mem_layer = QgsVectorLayer(uri, layer_name, "memory")
        mem_provider = mem_layer.dataProvider()

        # Copy fields
        mem_provider.addAttributes(layer.fields().toList())
        mem_layer.updateFields()

        # Add filtered features
        mem_provider.addFeatures(features)
        mem_layer.updateExtents()

        log_info(f"メモリレイヤ作成完了: {feature_count}フィーチャ")
        return mem_layer

    def _apply_categorized_style(self, layer):
        """Apply categorized style based on tree species field."""
        from qgis.core import (
            QgsCategorizedSymbolRenderer, QgsRendererCategory,
            QgsFillSymbol, QgsSymbol
        )

        log_info("カテゴリスタイル適用中...")

        # Find tree species field
        field_names = [f.name() for f in layer.fields()]
        species_field = None

        # Try common field names
        possible_names = ['解析樹種', '樹種', '樹種名', 'tree_type', 'species', '種別', '樹種cd']
        for name in possible_names:
            if name in field_names:
                species_field = name
                break

        if not species_field:
            log_info(f"樹種フィールドが見つかりません。フィールド一覧: {field_names[:10]}")
            return

        log_info(f"樹種フィールド: {species_field}")

        # Get unique values (with limit and progress)
        unique_values = set()
        feature_count = 0
        max_scan = 10000  # Limit to prevent long scan

        for feature in layer.getFeatures():
            val = feature[species_field]
            if val:
                unique_values.add(str(val))
            feature_count += 1

            # Process events periodically
            if feature_count % 2000 == 0:
                self._process_events()

            # Stop after scanning enough features
            if feature_count >= max_scan:
                break

        log_info(f"樹種カテゴリ数: {len(unique_values)}")

        # Color mapping for tree species
        color_map = {
            'スギ': '#00cc66',
            'ヒノキ': '#99ff66',
            'ヒノキ類': '#99ff66',
            'マツ': '#cc0000',
            'マツ類': '#cc0000',
            'アカマツ': '#ff3300',
            'クロマツ': '#cc3300',
            'カラマツ': '#ff6600',
            'モミ・ツガ': '#ff9900',
            'エゾマツ・トドマツ': '#66cc00',
            'その他針葉樹': '#99cc00',
            '広葉樹': '#ffff00',
            'クヌギ・コナラ': '#cc9900',
            'ブナ': '#996600',
            'その他広葉樹': '#ffcc00',
            '竹林': '#9933ff',
            '針広混交林': '#66ccff',
            '新植地': '#99ff99',
            '伐採跡地': '#ff00ff',
            'その他': '#999999',
        }

        categories = []
        for value in sorted(unique_values):
            color = color_map.get(value, '#888888')
            symbol = QgsFillSymbol.createSimple({
                'color': color,
                'outline_color': '#333333',
                'outline_width': '0.2'
            })
            category = QgsRendererCategory(value, symbol, value)
            categories.append(category)

        if categories:
            renderer = QgsCategorizedSymbolRenderer(species_field, categories)
            layer.setRenderer(renderer)
            layer.triggerRepaint()
            log_info(f"カテゴリスタイル適用完了: {len(categories)}カテゴリ")

    def _download_and_merge_gpkg(self, gpkg_urls: list, pref_code: str, extent=None):
        """Download multiple GeoPackage files and merge into a single layer."""
        from qgis.core import (
            QgsVectorLayer, QgsProject, QgsFeatureRequest, QgsFeature,
            QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsRectangle
        )
        import tempfile

        total_files = len(gpkg_urls)
        log_info(f"GeoPackageダウンロード開始: {total_files}ファイル")
        self._report_progress(f"樹種ポリゴン: {total_files}ファイルを確認中...")

        all_features = []
        layer_crs = None
        fields = None
        files_with_data = 0
        max_files_with_data = 3  # Stop after finding 3 files with data in view
        reached_limit = False  # Flag to stop completely when limit reached

        for i, url in enumerate(gpkg_urls):
            # Stop if we have enough data
            if files_with_data >= max_files_with_data:
                log_info(f"十分なデータを取得したため終了 ({files_with_data}ファイル)")
                self._report_progress(f"樹種ポリゴン: 十分なデータを取得しました")
                break

            # Stop if total limit reached
            if reached_limit:
                break

            # Extract filename from URL for display
            filename = url.split('/')[-1] if '/' in url else f"ファイル{i+1}"
            if len(filename) > 30:
                filename = filename[:27] + "..."

            progress_msg = f"樹種ポリゴン: ダウンロード中 ({i+1}/{total_files}) {filename}"
            log_info(f"ダウンロード中 ({i+1}/{total_files}): {url[-50:]}...")
            self._report_progress(progress_msg)

            try:
                # Download to temp file with progress
                fd, temp_path = tempfile.mkstemp(suffix='.gpkg')
                os.close(fd)
                try:
                    downloaded = self._download_to_path(
                        url, temp_path,
                        progress_prefix=f"樹種ポリゴン: ダウンロード中 ({i+1}/{total_files})"
                    )
                    if downloaded == 0:
                        log_error(f"ダウンロード失敗: {url}")
                        continue

                    self._report_progress(f"樹種ポリゴン: 読み込み中 ({i+1}/{total_files})...")

                    # Load as vector layer
                    temp_layer = QgsVectorLayer(temp_path, "temp", "ogr")
                    if not temp_layer.isValid():
                        log_error(f"無効なGeoPackage: {url}")
                        continue

                    # Get CRS and fields from first valid layer
                    if layer_crs is None:
                        layer_crs = temp_layer.crs()
                        fields = temp_layer.fields()

                    # Apply STRICT extent filter (no buffer for accuracy)
                    if extent:
                        # Transform extent to layer CRS
                        filter_extent = extent
                        if layer_crs.authid() != "EPSG:4326":
                            wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
                            transform = QgsCoordinateTransform(wgs84, layer_crs, QgsProject.instance())
                            filter_extent = transform.transformBoundingBox(extent)

                        # Use exact extent (no buffer) for strict filtering
                        feat_request = QgsFeatureRequest().setFilterRect(filter_extent)
                    else:
                        feat_request = QgsFeatureRequest()

                    # Iterate features with progress and limits
                    features = []
                    max_features_per_file = 20000  # Limit per file to prevent memory issues
                    feature_count = 0

                    for feat in temp_layer.getFeatures(feat_request):
                        features.append(QgsFeature(feat))  # Copy feature
                        feature_count += 1

                        # Process events periodically
                        if feature_count % 1000 == 0:
                            self._report_progress(
                                f"樹種ポリゴン: フィーチャ読み込み中 ({feature_count:,}件)..."
                            )

                        # Stop if too many features
                        if feature_count >= max_features_per_file:
                            log_info(f"  フィーチャ数制限に達しました: {max_features_per_file}")
                            break

                        # Also stop if total features too high
                        if len(all_features) + len(features) >= 50000:
                            log_info(f"  合計フィーチャ数制限に達しました")
                            break

                    if features:
                        all_features.extend(features)
                        files_with_data += 1
                        log_info(f"  {len(features)}フィーチャ取得 (累計: {len(all_features)})")
                        self._report_progress(
                            f"樹種ポリゴン: {len(features):,}件取得 (累計: {len(all_features):,}件)"
                        )
                    else:
                        log_info(f"  表示範囲外のためスキップ")

                    # Check total limit
                    if len(all_features) >= 50000:
                        log_info(f"合計フィーチャ数制限に達したため終了")
                        self._report_progress("樹種ポリゴン: フィーチャ数上限に達しました")
                        reached_limit = True
                        break

                finally:
                    # Clean up temp file
                    try:
                        os.remove(temp_path)
                    except:
                        pass

            except Exception as e:
                log_error(f"ダウンロードエラー: {str(e)}")
                continue

        if not all_features:
            self.last_error = "表示範囲にデータがありません"
            log_error(self.last_error)
            self._report_progress("")
            return None

        log_info(f"合計{len(all_features)}フィーチャをマージ")
        self._report_progress(f"樹種ポリゴン: レイヤ作成中 ({len(all_features):,}件)...")

        # Create memory layer
        geom_type = 'Polygon'  # Tree species data is polygon
        uri = f"{geom_type}?crs={layer_crs.authid()}"
        layer_name = f"樹種ポリゴン_{pref_code}"
        mem_layer = QgsVectorLayer(uri, layer_name, "memory")
        mem_provider = mem_layer.dataProvider()

        # Copy fields
        if fields:
            mem_provider.addAttributes(fields.toList())
            mem_layer.updateFields()

        # Add features in batches to prevent freeze
        self._report_progress(f"樹種ポリゴン: データ追加中...")
        batch_size = 5000
        for start in range(0, len(all_features), batch_size):
            end = min(start + batch_size, len(all_features))
            mem_provider.addFeatures(all_features[start:end])
            self._report_progress(f"樹種ポリゴン: データ追加中 ({end:,}/{len(all_features):,}件)")

        mem_layer.updateExtents()

        log_info(f"メモリレイヤ作成完了: {len(all_features)}フィーチャ")
        self._report_progress(f"樹種ポリゴン: スタイル適用中...")

        # Apply categorized style
        self._apply_categorized_style(mem_layer)

        # Add to project
        QgsProject.instance().addMapLayer(mem_layer)

        # Expand legend
        try:
            root = QgsProject.instance().layerTreeRoot()
            layer_node = root.findLayer(mem_layer.id())
            if layer_node:
                layer_node.setExpanded(True)
        except Exception:
            pass

        self._report_progress("")  # Clear progress
        return mem_layer

    def download_rinnya_data(self, data_type: str, pref_code: str, extent=None, tree_format: str = None, selected_urls: list = None):
        """
        Download and load Forest Agency open data.

        Args:
            data_type: Type of data ('tree_species', 'dem', 'cs_map', 'slope')
            pref_code: Prefecture code (2 digits)
            extent: Optional QgsRectangle to filter data (in WGS84)
            tree_format: Format preference for tree_species ('xyz' for vector tiles, 'gpkg' for GeoPackage)
            selected_urls: Optional list of GeoPackage URLs to download (for tree_species with gpkg format)

        Returns:
            QgsMapLayer or None
        """
        from qgis.core import QgsVectorLayer, QgsRasterLayer, QgsProject

        self.last_error = ""

        if data_type == 'tree_species':
            return self._download_tree_species(pref_code, extent, tree_format, selected_urls)
        elif data_type in ['dem', 'cs_map', 'slope']:
            return self._add_rinnya_tile_layer(data_type, pref_code)
        else:
            self.last_error = f"不明なデータタイプ: {data_type}"
            return None

    def _download_tree_species(self, pref_code: str, extent=None, user_format: str = None, selected_urls: list = None):
        """Download tree species polygon data using CKAN API.

        Args:
            pref_code: Prefecture code (2 digits)
            extent: Optional QgsRectangle to filter data (in WGS84)
            user_format: User-selected format preference ('xyz' for vector tiles, 'gpkg' for GeoPackage)
                         If None, uses the default format from KNOWN_DATASETS.
            selected_urls: Optional list of specific GeoPackage URLs to download.
                          If provided, only these URLs will be downloaded.
        """
        from qgis.core import QgsVectorLayer, QgsProject

        log_info(f"樹種ポリゴン取得開始: pref={pref_code}")
        if extent:
            log_info(f"範囲フィルター: {extent.xMinimum():.4f},{extent.yMinimum():.4f} - {extent.xMaximum():.4f},{extent.yMaximum():.4f}")

        # If selected URLs are provided for GeoPackage format, use them directly
        if selected_urls and user_format == 'gpkg':
            log_info(f"選択された森林計画区を使用: {len(selected_urls)}件")
            return self._download_and_merge_gpkg(selected_urls, pref_code, extent)

        if pref_code not in self.KNOWN_DATASETS:
            self.last_error = "この都道府県はG空間情報センターにデータセットがありません"
            log_error(f"未対応都道府県: {pref_code} (G空間情報センターにデータセットなし)")
            return None

        # Get dataset info (now tuple: (dataset_id, default_format))
        dataset_info = self.KNOWN_DATASETS[pref_code]
        dataset_id = dataset_info[0] if isinstance(dataset_info, tuple) else dataset_info
        default_format = dataset_info[1] if isinstance(dataset_info, tuple) else 'gpkg'

        # Use user-selected format if provided, otherwise fall back to default
        preferred_format = user_format if user_format else default_format
        log_info(f"データセットID: {dataset_id}, 選択フォーマット: {preferred_format} (デフォルト: {default_format})")

        # Get dataset details
        api_url = f"{self.CKAN_SHOW_URL}?id={dataset_id}"
        log_info(f"CKAN API呼び出し: {api_url}")

        try:
            data = self._fetch_json(api_url)
            if data is None:
                return None

            if not data.get('success'):
                self.last_error = "CKAN APIエラー"
                log_error(f"CKAN API失敗: {data}")
                return None

            # Find resources - scan all first
            resources = data.get('result', {}).get('resources', [])
            vector_tile_url = None
            style_json_url = None
            gpkg_urls = []  # Collect all direct .gpkg URLs
            compressed_url = None
            compressed_format = None

            # First pass: find all relevant URLs
            for resource in resources:
                url = resource.get('url', '')
                name = resource.get('name', '')
                fmt = resource.get('format', '').lower()
                description = resource.get('description', '')

                log_info(f"リソース発見: {name} ({fmt}) - {url[:80] if url else '(empty)'}...")

                # Vector tiles (XYZ) - exclude PDFs
                if fmt == 'xyz' and 'pdf' not in name.lower():
                    tile_url = url
                    if not tile_url and description:
                        import re
                        url_match = re.search(r'https?://[^\s"\'<>]+\.pbf', description)
                        if not url_match:
                            url_match = re.search(r'https?://[^\s"\'<>]+/\{z\}/\{x\}/\{y\}', description)
                        if url_match:
                            tile_url = url_match.group(0)
                            log_info(f"descriptionからURL抽出: {tile_url}")
                    if tile_url and '.pbf' in tile_url or '{z}' in tile_url:
                        vector_tile_url = tile_url
                        log_info(f"ベクトルタイルURL発見: {tile_url}")

                # Style JSON
                elif fmt == 'json' and 'style' in name.lower():
                    if url:
                        style_json_url = url
                        log_info(f"スタイルJSON発見: {url}")

                # Direct GeoPackage (collect ALL)
                elif url and url.endswith('.gpkg'):
                    gpkg_urls.append(url)
                    log_info(f"直接GeoPackage発見: {url}")

                # 7z compressed (fallback)
                elif url and (url.endswith('.gpkg.7z') or (fmt == '7z' and 'gpkg' in name.lower())):
                    if not compressed_url:
                        compressed_url = url
                        compressed_format = '7z'

                # ZIP (fallback)
                elif url and (url.endswith('.zip') or fmt == 'zip'):
                    if not compressed_url:
                        compressed_url = url
                        compressed_format = 'zip'

            # Choose format based on preferred_format from KNOWN_DATASETS
            # Priority 1: Direct GeoPackage (if preferred and available)
            if preferred_format == 'gpkg' and gpkg_urls:
                log_info(f"直接GeoPackageを使用します（{len(gpkg_urls)}ファイル、ライブラリ不要）")
                return self._download_and_merge_gpkg(gpkg_urls, pref_code, extent)

            # Priority 2: Vector tiles (for xyz-preferred or as fallback)
            if preferred_format == 'xyz' or (preferred_format != 'gpkg' and vector_tile_url):
                if vector_tile_url:
                    log_info("ベクトルタイルを使用します（ライブラリ不要、凡例制限あり）")
                    return self._add_vector_tile_layer(vector_tile_url, pref_code, style_json_url)

            # Fallback to gpkg if available but not preferred
            if gpkg_urls:
                log_info(f"直接GeoPackageを使用します（{len(gpkg_urls)}ファイル）")
                return self._download_and_merge_gpkg(gpkg_urls, pref_code, extent)

            # Fallback to vector tiles
            if vector_tile_url:
                log_info("ベクトルタイルを使用します（凡例制限あり）")
                return self._add_vector_tile_layer(vector_tile_url, pref_code, style_json_url)

            # Last resort: Compressed files (requires library)
            if not compressed_url:
                self.last_error = "ダウンロードURLが見つかりません"
                log_error("リソースにGeoPackage/ベクトルタイルが見つかりません")
                return None

            # Warn user about library requirement
            log_info(f"圧縮ファイルを使用します（{compressed_format}、ライブラリが必要な場合があります）")
            download_url = compressed_url
            file_format = compressed_format

            log_info(f"ダウンロードURL: {download_url}")
            log_info(f"ファイル形式: {file_format}")

            # Download the file
            if file_format == '7z':
                ext = '.7z'
            elif file_format == 'zip':
                ext = '.zip'
            else:
                ext = '.gpkg'

            file_path = self._download_to_cache(download_url, 'tree_species', pref_code, ext)

            if file_path:
                # Extract if compressed
                if file_format == '7z':
                    extracted = self._extract_7z(file_path, pref_code)
                    if extracted:
                        file_path = extracted
                    else:
                        self.last_error = "7z展開に失敗しました（7-Zipがインストールされていない可能性があります）"
                        return None
                elif file_format == 'zip':
                    extracted = self._extract_rinnya_zip(file_path, pref_code)
                    if extracted:
                        file_path = extracted
                    else:
                        self.last_error = "ZIP展開に失敗しました"
                        return None

                log_info(f"ダウンロード成功: {file_path}")
                layer_name = f"樹種ポリゴン_{pref_code}"
                layer = QgsVectorLayer(file_path, layer_name, "ogr")

                if layer.isValid():
                    # Apply extent filter if provided
                    if extent:
                        layer = self._filter_layer_by_extent(layer, extent, layer_name)
                        if not layer:
                            return None

                    # Apply categorized styling
                    self._apply_categorized_style(layer)

                    QgsProject.instance().addMapLayer(layer)

                    # Expand legend
                    try:
                        root = QgsProject.instance().layerTreeRoot()
                        layer_node = root.findLayer(layer.id())
                        if layer_node:
                            layer_node.setExpanded(True)
                    except Exception:
                        pass

                    return layer
                else:
                    self.last_error = "レイヤの読み込みに失敗しました"
                    log_error(f"無効なレイヤ: {file_path}")

        except Exception as e:
            log_error(f"エラー: {str(e)}")
            self.last_error = str(e)

        log_error(f"樹種ポリゴン取得失敗: {self.last_error}")
        return None

    def _ensure_py7zr(self, ask_permission: bool = True) -> bool:
        """Ensure py7zr is installed.

        Args:
            ask_permission: If True, ask user for permission before installing.

        Returns:
            True if py7zr is available, False otherwise.
        """
        try:
            import py7zr
            return True
        except ImportError:
            pass

        # Ask for user permission before installing
        if ask_permission:
            from qgis.PyQt.QtWidgets import QMessageBox
            reply = QMessageBox.question(
                None,
                "ライブラリのインストール確認",
                "この都道府県のデータは7z形式で圧縮されています。\n"
                "展開するために「py7zr」ライブラリのインストールが必要です。\n\n"
                "インストールには時間がかかる場合があります（1〜3分）。\n\n"
                "インストールしてもよろしいですか？",
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No
            )
            if reply != QMessageBox.Yes:
                log_info("ユーザーがライブラリインストールをキャンセルしました")
                self.last_error = "ライブラリのインストールがキャンセルされました"
                return False

        # Try to install py7zr
        log_info("py7zrをインストール中...（しばらくお待ちください）")
        import subprocess

        # Find Python executable for QGIS
        python_path = self._find_qgis_python()
        if not python_path:
            log_error("QGIS Python実行ファイルが見つかりません")
            return False

        log_info(f"Python: {python_path}")

        try:
            result = subprocess.run(
                [python_path, '-m', 'pip', 'install', 'py7zr', '--quiet', '--user'],
                capture_output=True,
                text=True,
                timeout=180  # 3 minutes
            )
            if result.returncode == 0:
                log_info("py7zrインストール成功")
                # Reload to make it available
                import importlib
                import sys
                if 'py7zr' in sys.modules:
                    importlib.reload(sys.modules['py7zr'])
                return True
            else:
                log_error(f"py7zrインストール失敗: {result.stderr}")
                return False
        except subprocess.TimeoutExpired:
            log_error("py7zrインストールタイムアウト（3分）")
            return False
        except Exception as e:
            log_error(f"py7zrインストールエラー: {str(e)}")
            return False

    def _find_qgis_python(self) -> Optional[str]:
        """Find QGIS Python executable."""
        import sys
        import glob

        # Method 1: Check if sys.executable is actually Python
        if 'python' in sys.executable.lower() and os.path.exists(sys.executable):
            return sys.executable

        # Method 2: Find from QGIS prefix path
        try:
            from qgis.core import QgsApplication
            prefix = QgsApplication.prefixPath()
            # Windows: C:\Program Files\QGIS 3.x.x\apps\qgis-ltr
            # Python is at: C:\Program Files\QGIS 3.x.x\apps\Python3x\python.exe

            qgis_root = os.path.dirname(os.path.dirname(prefix))
            python_patterns = [
                os.path.join(qgis_root, 'apps', 'Python*', 'python.exe'),
                os.path.join(qgis_root, 'bin', 'python*.exe'),
            ]

            for pattern in python_patterns:
                matches = glob.glob(pattern)
                if matches:
                    log_info(f"Python発見: {matches[0]}")
                    return matches[0]
        except Exception as e:
            log_error(f"Pythonパス検索エラー: {str(e)}")

        # Method 3: Common QGIS installation paths on Windows
        common_paths = [
            r'C:\Program Files\QGIS 3.42.2\apps\Python312\python.exe',
            r'C:\Program Files\QGIS 3.42.2\apps\Python311\python.exe',
            r'C:\Program Files\QGIS 3.42.2\apps\Python39\python.exe',
            r'C:\Program Files\QGIS 3.40\apps\Python312\python.exe',
            r'C:\Program Files\QGIS 3.38\apps\Python311\python.exe',
            r'C:\Program Files\QGIS 3.34\apps\Python39\python.exe',
            r'C:\OSGeo4W\apps\Python39\python.exe',
            r'C:\OSGeo4W64\apps\Python39\python.exe',
        ]

        for path in common_paths:
            if os.path.exists(path):
                log_info(f"Python発見（固定パス）: {path}")
                return path

        return None

    def _extract_7z(self, archive_path: str, pref_code: str) -> Optional[str]:
        """Extract 7z archive file."""
        import subprocess

        cache_subdir = os.path.join(self.cache_dir, 'rinnya', 'tree_species', pref_code)
        os.makedirs(cache_subdir, exist_ok=True)

        log_info(f"7z展開中: {archive_path}")

        # First, try py7zr (auto-install if needed)
        if self._ensure_py7zr():
            try:
                import py7zr
                log_info("py7zrで展開中...")
                with py7zr.SevenZipFile(archive_path, mode='r') as z:
                    z.extractall(path=cache_subdir)
                log_info("7z展開成功")

                # Find extracted gpkg file
                for root, dirs, files in os.walk(cache_subdir):
                    for f in files:
                        if f.endswith('.gpkg'):
                            log_info(f"GeoPackage発見: {f}")
                            return os.path.join(root, f)
                        elif f.endswith('.shp'):
                            log_info(f"Shapefile発見: {f}")
                            return os.path.join(root, f)
                log_error("展開後のファイルが見つかりません")
                return None
            except Exception as e:
                log_error(f"py7zr展開エラー: {str(e)}")

        # Fallback: Try using 7z command line tool
        seven_zip_paths = [
            '7z',
            r'C:\Program Files\7-Zip\7z.exe',
            r'C:\Program Files (x86)\7-Zip\7z.exe',
        ]

        for sz_path in seven_zip_paths:
            try:
                result = subprocess.run(
                    [sz_path, 'x', '-y', f'-o{cache_subdir}', archive_path],
                    capture_output=True,
                    text=True,
                    timeout=120
                )
                if result.returncode == 0:
                    log_info("7z展開成功（7-Zipコマンド）")
                    for root, dirs, files in os.walk(cache_subdir):
                        for f in files:
                            if f.endswith('.gpkg'):
                                return os.path.join(root, f)
                            elif f.endswith('.shp'):
                                return os.path.join(root, f)
                    return None
            except FileNotFoundError:
                continue
            except Exception:
                continue

        log_error("7z展開に失敗しました")
        return None

    def _extract_rinnya_zip(self, zip_path: str, pref_code: str) -> Optional[str]:
        """Extract Forest Agency ZIP file."""
        cache_subdir = os.path.join(self.cache_dir, 'rinnya', 'tree_species', pref_code)

        try:
            log_info(f"ZIP展開中: {zip_path}")
            with zipfile.ZipFile(zip_path, 'r') as zf:
                zf.extractall(cache_subdir)

            # Find GeoPackage or Shapefile
            for root, dirs, files in os.walk(cache_subdir):
                for f in files:
                    if f.endswith('.gpkg'):
                        log_info(f"GeoPackage発見: {f}")
                        return os.path.join(root, f)
                    elif f.endswith('.shp'):
                        log_info(f"Shapefile発見: {f}")
                        return os.path.join(root, f)

            log_error("ZIP内にGeoPackage/Shapefileが見つかりません")
            return None
        except Exception as e:
            log_error(f"ZIP展開エラー: {str(e)}")
            return None

    def _search_ckan_dataset(self, query: str, pref_name_jp: str,
                              pref_name_en: str) -> Optional[str]:
        """Search CKAN for a dataset matching the query."""
        from urllib.parse import quote

        search_url = f"{self.CKAN_SEARCH_URL}?q={quote(query)}&rows=20"
        log_info(f"CKAN検索: {search_url}")

        try:
            data = self._fetch_json(search_url)
            if data is None:
                return None

            if not data.get('success'):
                log_error("CKAN検索失敗")
                return None

            results = data.get('result', {}).get('results', [])
            log_info(f"検索結果: {len(results)}件")

            # Look for dataset matching prefecture
            for dataset in results:
                name = dataset.get('name', '').lower()
                title = dataset.get('title', '')
                notes = dataset.get('notes', '')

                # Check if this is a tree species dataset for our prefecture
                is_tree = any(kw in title.lower() or kw in name for kw in
                             ['樹種', 'jushu', 'tree', '森林資源'])
                is_pref = (pref_name_jp in title or pref_name_en in name or
                          pref_name_jp in notes)
                is_rinnya = '林野庁' in title or 'rinya' in name

                if is_tree and is_pref:
                    log_info(f"データセット発見: {dataset.get('name')} - {title}")
                    return dataset.get('name')

                # Also check for general forest data from Forest Agency
                if is_rinnya and is_pref:
                    log_info(f"林野庁データセット発見: {dataset.get('name')} - {title}")
                    return dataset.get('name')

            return None

        except Exception as e:
            log_error(f"CKAN検索エラー: {str(e)}")
            return None

    def _add_vector_tile_layer(self, tile_url: str, pref_code: str, style_json_url: str = None):
        """Add vector tile layer (no download required)."""
        from qgis.core import QgsVectorTileLayer, QgsProject

        log_info(f"ベクトルタイルレイヤ追加: {tile_url}")

        # Create vector tile layer URI
        if '{z}' not in tile_url:
            if not tile_url.endswith('/'):
                tile_url += '/'
            tile_url += '{z}/{x}/{y}.pbf'

        uri = f"type=xyz&url={tile_url}&zmin=0&zmax=18"
        layer_name = f"樹種ポリゴン_{pref_code}"

        layer = QgsVectorTileLayer(uri, layer_name)

        if layer.isValid():
            log_info("ベクトルタイルレイヤ作成成功")

            # Try to apply style from JSON URL first
            style_applied = False
            if style_json_url:
                style_applied = self._apply_style_from_json(layer, style_json_url)

            # Fall back to default styling if no style JSON
            if not style_applied:
                self._apply_tree_species_style(layer)

            QgsProject.instance().addMapLayer(layer)

            # Force refresh and expand legend
            try:
                layer.triggerRepaint()

                # Expand legend to show all categories
                root = QgsProject.instance().layerTreeRoot()
                layer_tree_node = root.findLayer(layer.id())
                if layer_tree_node:
                    layer_tree_node.setExpanded(True)
                    # Force legend refresh
                    layer_tree_node.setCustomProperty("showFeatureCount", False)

                log_info("凡例展開完了")
            except Exception as e:
                log_error(f"凡例展開エラー: {str(e)}")

            return layer
        else:
            log_error("ベクトルタイルレイヤ作成失敗")
            self.last_error = "ベクトルタイルレイヤの作成に失敗しました"
            return None

    def _apply_style_from_json(self, layer, style_url: str) -> bool:
        """Apply Mapbox GL style from JSON URL."""
        log_info(f"スタイルJSON読み込み: {style_url}")

        try:
            style_data = self._fetch_json(style_url)
            if style_data is None:
                return False

            # Parse Mapbox GL style and convert to QGIS style
            layers = style_data.get('layers', [])
            log_info(f"スタイルレイヤ数: {len(layers)}")
            if not layers:
                log_error("スタイルJSONにレイヤ情報がありません")
                return False

            from qgis.core import (
                QgsVectorTileBasicRenderer, QgsVectorTileBasicRendererStyle,
                QgsFillSymbol, QgsLineSymbol, QgsWkbTypes
            )

            styles = []

            for layer_def in layers:
                layer_type = layer_def.get('type', '')
                layer_id = layer_def.get('id', '')
                source_layer = layer_def.get('source-layer', '')
                paint = layer_def.get('paint', {})
                filter_expr = layer_def.get('filter', [])

                if layer_type == 'fill':
                    style = QgsVectorTileBasicRendererStyle()
                    style.setStyleName(layer_id)
                    style.setLayerName(source_layer)
                    style.setGeometryType(QgsWkbTypes.PolygonGeometry)
                    style.setMinZoomLevel(int(layer_def.get('minzoom', 0)))
                    style.setMaxZoomLevel(int(layer_def.get('maxzoom', 18)))
                    style.setEnabled(True)

                    # Convert Mapbox filter to QGIS expression
                    if filter_expr:
                        qgis_filter = self._convert_mapbox_filter(filter_expr)
                        if qgis_filter:
                            style.setFilterExpression(qgis_filter)

                    # Get colors
                    fill_color = paint.get('fill-color', '#888888')
                    outline_color = paint.get('fill-outline-color', '')

                    # Handle color expressions (simplified)
                    if isinstance(fill_color, dict) or isinstance(fill_color, list):
                        fill_color = '#888888'
                    if isinstance(outline_color, dict) or isinstance(outline_color, list):
                        outline_color = ''

                    props = {
                        'color': fill_color,
                        'outline_width': '0.1'
                    }
                    if outline_color:
                        props['outline_color'] = outline_color
                    else:
                        props['outline_style'] = 'no'

                    symbol = QgsFillSymbol.createSimple(props)
                    style.setSymbol(symbol)
                    styles.append(style)
                    log_info(f"  {layer_id}: {fill_color}")

                elif layer_type == 'line':
                    style = QgsVectorTileBasicRendererStyle()
                    style.setStyleName(layer_id)
                    style.setLayerName(source_layer)
                    style.setGeometryType(QgsWkbTypes.LineGeometry)
                    style.setMinZoomLevel(layer_def.get('minzoom', 0))
                    style.setMaxZoomLevel(layer_def.get('maxzoom', 18))

                    line_color = paint.get('line-color', '#333333')
                    line_width = paint.get('line-width', 1)

                    if isinstance(line_color, dict) or isinstance(line_color, list):
                        line_color = '#333333'
                    if isinstance(line_width, dict) or isinstance(line_width, list):
                        line_width = 1

                    symbol = QgsLineSymbol.createSimple({
                        'color': line_color,
                        'width': str(line_width)
                    })
                    style.setSymbol(symbol)
                    styles.append(style)

            if styles:
                renderer = QgsVectorTileBasicRenderer()
                renderer.setStyles(styles)
                layer.setRenderer(renderer)

                # Verify styles were applied
                applied_styles = renderer.styles()
                log_info(f"スタイルJSON適用成功: {len(applied_styles)}スタイル適用済み")

                # List all style names for debugging
                for s in applied_styles[:5]:  # First 5 only
                    log_info(f"  凡例項目: {s.styleName()}")

                return True

            return False

        except Exception as e:
            log_error(f"スタイルJSON読み込みエラー: {str(e)}")
            return False

    def _convert_mapbox_filter(self, filter_expr) -> str:
        """Convert Mapbox GL filter to QGIS expression (simplified)."""
        if not filter_expr or not isinstance(filter_expr, list):
            return ''

        try:
            op = filter_expr[0]

            if op == '==':
                field = filter_expr[1]
                value = filter_expr[2]
                # Handle ["get", "field"] format
                if isinstance(field, list) and field[0] == 'get':
                    field = field[1]
                # Field is already a string
                return f'"{field}" = \'{value}\''

            elif op == '!=':
                field = filter_expr[1]
                value = filter_expr[2]
                if isinstance(field, list) and field[0] == 'get':
                    field = field[1]
                return f'"{field}" != \'{value}\''

            elif op == 'in':
                field = filter_expr[1]
                values = filter_expr[2:]
                if isinstance(field, list) and field[0] == 'get':
                    field = field[1]
                values_str = ', '.join([f"'{v}'" for v in values])
                return f'"{field}" IN ({values_str})'

            elif op == 'all':
                parts = [self._convert_mapbox_filter(f) for f in filter_expr[1:]]
                parts = [p for p in parts if p]
                return ' AND '.join(parts) if parts else ''

            elif op == 'any':
                parts = [self._convert_mapbox_filter(f) for f in filter_expr[1:]]
                parts = [p for p in parts if p]
                return ' OR '.join(parts) if parts else ''

            elif op == 'has':
                field = filter_expr[1]
                return f'"{field}" IS NOT NULL'

            elif op == '!has':
                field = filter_expr[1]
                return f'"{field}" IS NULL'

        except Exception as e:
            log_error(f"フィルター変換エラー: {filter_expr} - {str(e)}")

        return ''

    def _apply_tree_species_style(self, layer):
        """Apply categorized style for tree species."""
        from qgis.core import (
            QgsVectorTileBasicRenderer, QgsVectorTileBasicRendererStyle,
            QgsFillSymbol, QgsWkbTypes
        )

        # Tree species color mapping (based on common classifications)
        # Code-based colors (林野庁の樹種コード)
        tree_styles = [
            # 針葉樹 (Conifers)
            ('スギ', '1', '#1B5E20'),           # Dark green
            ('ヒノキ', '2', '#2E7D32'),          # Green
            ('アカマツ', '3', '#388E3C'),        # Light green
            ('クロマツ', '4', '#43A047'),
            ('カラマツ', '5', '#4CAF50'),
            ('エゾマツ', '6', '#66BB6A'),
            ('トドマツ', '7', '#81C784'),
            ('その他針葉樹', '8', '#A5D6A7'),
            # 広葉樹 (Broadleaf)
            ('クヌギ', '11', '#FF8F00'),
            ('コナラ', '12', '#FFA000'),
            ('ブナ', '13', '#FFB300'),
            ('その他広葉樹', '19', '#FFCA28'),
            ('広葉樹', '10', '#FFD54F'),
            # その他
            ('竹林', '20', '#7CB342'),
            ('無立木地', '30', '#D7CCC8'),
            ('非森林', '90', '#EFEBE9'),
        ]

        styles = []

        for name, code, color in tree_styles:
            style = QgsVectorTileBasicRendererStyle()
            style.setStyleName(name)
            style.setLayerName('')  # Match all layers in tile
            style.setGeometryType(QgsWkbTypes.PolygonGeometry)
            style.setEnabled(True)
            # Try multiple field names that might contain tree species
            style.setFilterExpression(
                f'"樹種コード" = \'{code}\' OR "樹種" = \'{name}\' OR '
                f'"解析樹種" = \'{name}\' OR "code" = \'{code}\' OR '
                f'"type" = \'{code}\' OR "種別" = \'{name}\''
            )
            style.setMinZoomLevel(0)
            style.setMaxZoomLevel(18)

            symbol = QgsFillSymbol.createSimple({
                'color': color,
                'outline_color': '#555555',
                'outline_width': '0.1'
            })
            style.setSymbol(symbol)
            styles.append(style)

        # Default style (catch-all for any unmatched polygons)
        default_style = QgsVectorTileBasicRendererStyle()
        default_style.setStyleName('その他')
        default_style.setLayerName('')
        default_style.setGeometryType(QgsWkbTypes.PolygonGeometry)
        default_style.setEnabled(True)
        default_style.setFilterExpression('')  # No filter = match all
        default_style.setMinZoomLevel(0)
        default_style.setMaxZoomLevel(18)

        default_symbol = QgsFillSymbol.createSimple({
            'color': '#B0BEC5',
            'outline_color': '#555555',
            'outline_width': '0.1'
        })
        default_style.setSymbol(default_symbol)
        styles.append(default_style)

        # Apply renderer
        renderer = QgsVectorTileBasicRenderer()
        renderer.setStyles(styles)
        layer.setRenderer(renderer)

        log_info(f"樹種スタイル適用完了: {len(tree_styles)}種類 + デフォルト")

    def _add_rinnya_tile_layer(self, data_type: str, pref_code: str):
        """Add Forest Agency raster tile layer (DEM, CS立体図, 傾斜区分図)."""
        from qgis.core import QgsRasterLayer, QgsProject
        import tempfile

        log_info(f"タイルレイヤ追加: type={data_type}, pref={pref_code}")

        # Get tile URL and max zoom for this prefecture and data type
        tile_url = None
        is_gsi = False
        actual_zmax = 18  # Default for 林野庁 tiles

        # Check if prefecture has this tile type
        if pref_code in self.TILE_URLS:
            pref_tiles = self.TILE_URLS[pref_code]
            if data_type in pref_tiles:
                tile_url = pref_tiles[data_type]
            elif data_type == 'dem' and 'dem_rgb' in pref_tiles:
                tile_url = pref_tiles['dem_rgb']

        # Fallback to GSI tiles (available nationwide)
        if not tile_url:
            if data_type == 'dem':
                tile_url = self.GSI_DEM_URL
                actual_zmax = self.GSI_DEM_ZMAX
                is_gsi = True
                log_info("国土地理院DEMタイルを使用します")
            elif data_type == 'slope':
                tile_url = self.GSI_SLOPE_URL
                actual_zmax = self.GSI_SLOPE_ZMAX
                is_gsi = True
                log_info("国土地理院傾斜量図を使用します")
            elif data_type == 'cs_map':
                # CS立体図の代替として国土地理院の陰影起伏図を使用
                tile_url = self.GSI_HILLSHADE_URL
                actual_zmax = self.GSI_HILLSHADE_ZMAX
                is_gsi = True
                log_info("国土地理院陰影起伏図を使用します（CS立体図の代替）")
            else:
                self.last_error = f"この都道府県では{self.DATA_NAMES.get(data_type, data_type)}は利用できません"
                log_error(self.last_error)
                return None

        # Set layer name
        data_name = self.DATA_NAMES.get(data_type, data_type)
        source_name = "GSI" if is_gsi else "林野庁"
        layer_name = f"{data_name}_{pref_code}（{source_name}）"

        log_info(f"タイルURL: {tile_url}")
        log_info(f"実際のズームレベル: 2-{actual_zmax}（オーバーズーム対応）")

        # Use GDAL WMS XML for better overzoom support
        # This allows QGIS to show lower zoom tiles when higher ones don't exist
        gdal_xml = self._create_gdal_wms_xml(tile_url, actual_zmax, data_type)

        # Write XML to temp file
        fd, xml_path = tempfile.mkstemp(suffix='.xml')
        try:
            with os.fdopen(fd, 'w', encoding='utf-8') as f:
                f.write(gdal_xml)

            layer = QgsRasterLayer(xml_path, layer_name)

            if layer.isValid():
                log_info(f"タイルレイヤ作成成功: {layer_name}")
                QgsProject.instance().addMapLayer(layer)

                # Set layer extent to Japan bounds (prevents "zoom to layer" showing entire globe)
                self._set_japan_extent(layer, pref_code)

                return layer
            else:
                # Fallback to standard XYZ method
                log_info("GDAL WMS失敗、標準XYZ方式にフォールバック")
                uri = f"type=xyz&url={tile_url}&zmin=2&zmax={actual_zmax}"
                layer = QgsRasterLayer(uri, layer_name, "wms")
                if layer.isValid():
                    QgsProject.instance().addMapLayer(layer)
                    return layer
                else:
                    self.last_error = "タイルレイヤの作成に失敗しました"
                    log_error(self.last_error)
                    return None
        finally:
            # Clean up temp file after a delay (layer might still need it)
            pass

    def _set_japan_extent(self, layer, pref_code: str):
        """Set layer extent to prefecture bounds (prevents zoom-to-layer showing entire globe)."""
        from qgis.core import QgsRectangle, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject

        # Prefecture bounding boxes (approximate, in WGS84)
        # Format: (min_lon, min_lat, max_lon, max_lat)
        PREF_BOUNDS = {
            '09': (139.3, 36.2, 140.3, 37.2),   # 栃木県
            '14': (138.9, 35.1, 139.8, 35.7),   # 神奈川県
            '16': (136.7, 36.3, 137.8, 37.0),   # 富山県
            '25': (135.8, 34.8, 136.5, 35.5),   # 滋賀県
            '26': (134.8, 34.7, 136.1, 35.8),   # 京都府
            '27': (135.1, 34.3, 135.7, 35.1),   # 大阪府
            '28': (134.2, 34.2, 135.5, 35.7),   # 兵庫県
            '31': (133.2, 35.1, 134.5, 35.6),   # 鳥取県
            '34': (132.0, 34.0, 133.4, 35.0),   # 広島県
            '38': (132.0, 33.0, 133.7, 34.3),   # 愛媛県
            '39': (132.5, 32.7, 134.3, 33.9),   # 高知県
            '42': (128.6, 32.5, 130.4, 34.7),   # 長崎県
        }

        # Default to Japan bounds if prefecture not found
        bounds = PREF_BOUNDS.get(pref_code, (122.0, 24.0, 154.0, 46.0))

        try:
            # Create extent in WGS84
            extent_wgs84 = QgsRectangle(bounds[0], bounds[1], bounds[2], bounds[3])

            # Transform to layer CRS (likely EPSG:3857)
            layer_crs = layer.crs()
            if layer_crs.authid() != "EPSG:4326":
                wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
                transform = QgsCoordinateTransform(wgs84, layer_crs, QgsProject.instance())
                extent = transform.transformBoundingBox(extent_wgs84)
            else:
                extent = extent_wgs84

            # Set the extent
            layer.setExtent(extent)
            log_info(f"レイヤ範囲を{pref_code}に設定")

        except Exception as e:
            log_error(f"レイヤ範囲設定エラー: {str(e)}")

    def _create_gdal_wms_xml(self, tile_url: str, max_zoom: int, data_type: str) -> str:
        """Create GDAL WMS XML configuration for tile layer with overzoom support."""
        # Convert {z}/{x}/{y} format to ${z}/${x}/${y} for GDAL
        gdal_url = tile_url.replace('{z}', '${z}').replace('{x}', '${x}').replace('{y}', '${y}')

        # Determine bands count based on data type
        bands = 4 if data_type in ['cs_map', 'slope'] else 3

        xml = f'''<GDAL_WMS>
    <Service name="TMS">
        <ServerUrl>{gdal_url}</ServerUrl>
    </Service>
    <DataWindow>
        <UpperLeftX>-20037508.34</UpperLeftX>
        <UpperLeftY>20037508.34</UpperLeftY>
        <LowerRightX>20037508.34</LowerRightX>
        <LowerRightY>-20037508.34</LowerRightY>
        <TileLevel>{max_zoom}</TileLevel>
        <TileCountX>1</TileCountX>
        <TileCountY>1</TileCountY>
        <YOrigin>top</YOrigin>
    </DataWindow>
    <Projection>EPSG:3857</Projection>
    <BlockSizeX>256</BlockSizeX>
    <BlockSizeY>256</BlockSizeY>
    <BandsCount>{bands}</BandsCount>
    <MaxConnections>5</MaxConnections>
    <Cache/>
    <ZeroBlockHttpCodes>204,404,500,502,503</ZeroBlockHttpCodes>
    <ZeroBlockOnServerException>true</ZeroBlockOnServerException>
</GDAL_WMS>'''
        return xml

    def _get_cached_rinnya(self, data_type: str, pref_code: str) -> Optional[str]:
        """Check if Forest Agency data is cached."""
        cache_subdir = os.path.join(self.cache_dir, 'rinnya', data_type, pref_code)
        if os.path.exists(cache_subdir):
            for f in os.listdir(cache_subdir):
                if f.endswith('.gpkg') or f.endswith('.shp'):
                    return os.path.join(cache_subdir, f)
        return None

    def _download_to_cache(self, url: str, data_type: str, pref_code: str,
                           ext: str = '.gpkg') -> Optional[str]:
        """Download file to cache directory."""
        log_info(f"キャッシュにダウンロード中: {url}")
        cache_subdir = os.path.join(self.cache_dir, 'rinnya', data_type, pref_code)
        os.makedirs(cache_subdir, exist_ok=True)

        file_path = os.path.join(cache_subdir, f"{data_type}_{pref_code}{ext}")

        total = self._download_to_path(url, file_path, progress_prefix="ダウンロード中:")
        if total == 0:
            return None

        if total > 1024 * 1024:
            size_str = f"{total / (1024 * 1024):.1f} MB"
        elif total > 1024:
            size_str = f"{total / 1024:.1f} KB"
        else:
            size_str = f"{total} bytes"
        log_info(f"キャッシュ保存完了: {file_path} ({size_str})")
        return file_path

    def _get_cached_kokudo(self, data_type: str, pref_code: str) -> Optional[str]:
        """Check if data is cached."""
        cache_subdir = os.path.join(self.cache_dir, 'kokudo', data_type, pref_code)
        if os.path.exists(cache_subdir):
            for f in os.listdir(cache_subdir):
                if f.endswith('.shp') or f.endswith('.geojson'):
                    return os.path.join(cache_subdir, f)
        return None

    def _download_file(self, url: str) -> Optional[str]:
        """Download file from URL."""
        log_info(f"ファイルダウンロード中: {url}")
        fd, temp_path = tempfile.mkstemp(suffix='.zip')
        os.close(fd)
        try:
            total = self._download_to_path(url, temp_path, progress_prefix="ダウンロード中:")
            if total == 0:
                if os.path.exists(temp_path):
                    os.remove(temp_path)
                return None
            log_info(f"ダウンロード完了: {total} bytes")
            return temp_path
        except Exception as e:
            log_error(f"ダウンロード中にエラー: {str(e)}")
            if os.path.exists(temp_path):
                os.remove(temp_path)
            raise

    def _extract_kokudo_zip(self, zip_path: str, data_type: str,
                            pref_code: str) -> Optional[str]:
        """Extract national land information ZIP file."""
        cache_subdir = os.path.join(self.cache_dir, 'kokudo', data_type, pref_code)
        os.makedirs(cache_subdir, exist_ok=True)

        try:
            log_info(f"ZIP展開中: {zip_path} -> {cache_subdir}")
            with zipfile.ZipFile(zip_path, 'r') as zf:
                zf.extractall(cache_subdir)

            # Find shapefile or geojson
            for root, dirs, files in os.walk(cache_subdir):
                for f in files:
                    if f.endswith('.shp'):
                        log_info(f"Shapefile発見: {f}")
                        return os.path.join(root, f)
                    elif f.endswith('.geojson'):
                        log_info(f"GeoJSON発見: {f}")
                        return os.path.join(root, f)

            log_error("ZIP内にShapefileまたはGeoJSONが見つかりません")
            return None
        except Exception as e:
            log_error(f"ZIP展開エラー: {str(e)}")
            return None

    def load_local_shapefile(self, file_path: str,
                              layer_name: str = None) -> Optional[str]:
        """Load a local shapefile."""
        from qgis.core import QgsVectorLayer, QgsProject

        if not os.path.exists(file_path):
            return None

        if not layer_name:
            layer_name = os.path.splitext(os.path.basename(file_path))[0]

        layer = QgsVectorLayer(file_path, layer_name, "ogr")
        if layer.isValid():
            QgsProject.instance().addMapLayer(layer)
            return layer_name

        return None
