# -*- coding: utf-8 -*-
"""
Japan CRS Selector - Zone Master Data
"""

from dataclasses import dataclass
from typing import Dict, List, Optional


@dataclass
class ZoneInfo:
    zone_number: int
    origin_lat: float
    origin_lon: float
    epsg_jgd2000: int
    epsg_jgd2011: int
    regions: List[str]
    prefectures: List[str]

    @property
    def zone_name(self) -> str:
        roman = ["I", "II", "III", "IV", "V", "VI", "VII", "VIII", "IX",
                 "X", "XI", "XII", "XIII", "XIV", "XV", "XVI", "XVII", "XVIII", "XIX"]
        return f"第{roman[self.zone_number - 1]}系"

    @property
    def origin_lat_dms(self) -> str:
        return self._decimal_to_dms(self.origin_lat, "lat")

    @property
    def origin_lon_dms(self) -> str:
        return self._decimal_to_dms(self.origin_lon, "lon")

    def _decimal_to_dms(self, decimal: float, coord_type: str) -> str:
        d = int(decimal)
        m = int((decimal - d) * 60)
        s = ((decimal - d) * 60 - m) * 60
        if coord_type == "lat":
            direction = "北緯" if decimal >= 0 else "南緯"
        else:
            direction = "東経" if decimal >= 0 else "西経"
        return f"{direction}{abs(d):02d}°{m:02d}′{s:05.2f}″"

    def get_epsg_code(self, datum: str) -> int:
        if datum in ("JGD2024", "JGD2011"):
            return self.epsg_jgd2011
        elif datum == "JGD2000":
            return self.epsg_jgd2000
        else:
            raise ValueError(f"Unknown datum: {datum}")


class ZoneMaster:
    """座標系マスターデータ管理クラス"""

    DATUMS = {
        'JGD2024': {
            'name': '日本測地系2024',
            'name_en': 'JGD2024',
            'start_date': '2025-04-01',
            'epsg_base': 6669,
            'note': '水平座標はJGD2011と同一です。標高のみ変更されています。'
        },
        'JGD2011': {
            'name': '日本測地系2011',
            'name_en': 'JGD2011',
            'start_date': '2011-10-21',
            'epsg_base': 6669,
            'note': None
        },
        'JGD2000': {
            'name': '日本測地系2000',
            'name_en': 'JGD2000',
            'start_date': '2002-04-01',
            'epsg_base': 2443,
            'note': None
        }
    }

    ZONES: Dict[int, ZoneInfo] = {
        1: ZoneInfo(1, 33.0, 129.5, 2443, 6669, ['長崎県', '鹿児島県西部'], ['長崎県', '鹿児島県']),
        2: ZoneInfo(2, 33.0, 131.0, 2444, 6670, ['福岡県', '佐賀県', '熊本県', '大分県', '宮崎県', '鹿児島県（I系除く）'], ['福岡県', '佐賀県', '熊本県', '大分県', '宮崎県', '鹿児島県']),
        3: ZoneInfo(3, 36.0, 132.166666667, 2445, 6671, ['山口県', '島根県', '広島県'], ['山口県', '島根県', '広島県']),
        4: ZoneInfo(4, 33.0, 133.5, 2446, 6672, ['香川県', '愛媛県', '徳島県', '高知県'], ['香川県', '愛媛県', '徳島県', '高知県']),
        5: ZoneInfo(5, 36.0, 134.333333333, 2447, 6673, ['兵庫県', '鳥取県', '岡山県'], ['兵庫県', '鳥取県', '岡山県']),
        6: ZoneInfo(6, 36.0, 136.0, 2448, 6674, ['京都府', '大阪府', '福井県', '滋賀県', '三重県', '奈良県', '和歌山県'], ['京都府', '大阪府', '福井県', '滋賀県', '三重県', '奈良県', '和歌山県']),
        7: ZoneInfo(7, 36.0, 137.166666667, 2449, 6675, ['石川県', '富山県', '岐阜県', '愛知県'], ['石川県', '富山県', '岐阜県', '愛知県']),
        8: ZoneInfo(8, 36.0, 138.5, 2450, 6676, ['新潟県', '長野県', '山梨県', '静岡県'], ['新潟県', '長野県', '山梨県', '静岡県']),
        9: ZoneInfo(9, 36.0, 139.833333333, 2451, 6677, ['東京都（島嶼除く）', '福島県', '栃木県', '茨城県', '埼玉県', '千葉県', '群馬県', '神奈川県'], ['東京都', '福島県', '栃木県', '茨城県', '埼玉県', '千葉県', '群馬県', '神奈川県']),
        10: ZoneInfo(10, 40.0, 140.833333333, 2452, 6678, ['青森県', '秋田県', '山形県', '岩手県', '宮城県'], ['青森県', '秋田県', '山形県', '岩手県', '宮城県']),
        11: ZoneInfo(11, 44.0, 140.25, 2453, 6679, ['北海道（西部：小樽、札幌、函館）'], ['北海道']),
        12: ZoneInfo(12, 44.0, 142.25, 2454, 6680, ['北海道（中央部：旭川、留萌）'], ['北海道']),
        13: ZoneInfo(13, 44.0, 144.25, 2455, 6681, ['北海道（東部：北見、網走、根室、釧路）'], ['北海道']),
        14: ZoneInfo(14, 26.0, 142.0, 2456, 6682, ['東京都（南鳥島）'], ['東京都']),
        15: ZoneInfo(15, 26.0, 127.5, 2457, 6683, ['沖縄県（那覇、久米島）'], ['沖縄県']),
        16: ZoneInfo(16, 26.0, 124.0, 2458, 6684, ['沖縄県（石垣島、西表島）'], ['沖縄県']),
        17: ZoneInfo(17, 26.0, 131.0, 2459, 6685, ['沖縄県（大東諸島）'], ['沖縄県']),
        18: ZoneInfo(18, 20.0, 136.0, 2460, 6686, ['東京都（沖ノ鳥島）'], ['東京都']),
        19: ZoneInfo(19, 26.0, 154.0, 2461, 6687, ['東京都（小笠原諸島、南硫黄島）'], ['東京都']),
    }

    PREFECTURE_TO_ZONE: Dict[str, List[int]] = {
        '北海道': [11, 12, 13], '青森県': [10], '岩手県': [10], '宮城県': [10],
        '秋田県': [10], '山形県': [10], '福島県': [9], '茨城県': [9],
        '栃木県': [9], '群馬県': [9], '埼玉県': [9], '千葉県': [9],
        '東京都': [9, 14, 18, 19], '神奈川県': [9], '新潟県': [8], '富山県': [7],
        '石川県': [7], '福井県': [6], '山梨県': [8], '長野県': [8],
        '岐阜県': [7], '静岡県': [8], '愛知県': [7], '三重県': [6],
        '滋賀県': [6], '京都府': [6], '大阪府': [6], '兵庫県': [5],
        '奈良県': [6], '和歌山県': [6], '鳥取県': [5], '島根県': [3],
        '岡山県': [5], '広島県': [3], '山口県': [3], '徳島県': [4],
        '香川県': [4], '愛媛県': [4], '高知県': [4], '福岡県': [2],
        '佐賀県': [2], '長崎県': [1], '熊本県': [2], '大分県': [2],
        '宮崎県': [2], '鹿児島県': [1, 2], '沖縄県': [15, 16, 17],
    }

    def __init__(self):
        pass

    def get_zone(self, zone_number: int) -> Optional[ZoneInfo]:
        return self.ZONES.get(zone_number)

    def get_all_zones(self) -> Dict[int, ZoneInfo]:
        return self.ZONES

    def get_epsg_code(self, zone: int, datum: str) -> int:
        zone_info = self.get_zone(zone)
        if zone_info is None:
            raise ValueError(f"Invalid zone number: {zone}")
        return zone_info.get_epsg_code(datum)

    def get_zones_for_prefecture(self, prefecture: str) -> List[int]:
        return self.PREFECTURE_TO_ZONE.get(prefecture, [])

    def get_datum_info(self, datum: str) -> Dict:
        return self.DATUMS.get(datum, {})

    def get_all_datums(self) -> List[str]:
        return list(self.DATUMS.keys())
