# -*- coding: utf-8 -*-
"""
Japan CRS Selector - Zone Detector
座標から平面直角座標系を判別
"""

import math
from typing import List, Optional, Tuple

try:
    from qgis.core import QgsPointXY, QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject
    HAS_QGIS = True
except ImportError:
    HAS_QGIS = False

from .zone_master import ZoneMaster, ZoneInfo


class ZoneDetector:
    PREFECTURE_CENTERS = {
        "北海道": (43.0, 141.3), "青森県": (40.8, 140.7), "岩手県": (39.5, 141.1),
        "宮城県": (38.4, 140.9), "秋田県": (39.7, 140.1), "山形県": (38.2, 140.0),
        "福島県": (37.4, 140.0), "茨城県": (36.3, 140.4), "栃木県": (36.6, 139.9),
        "群馬県": (36.4, 139.0), "埼玉県": (35.9, 139.6), "千葉県": (35.5, 140.1),
        "東京都": (35.7, 139.7), "神奈川県": (35.4, 139.4), "新潟県": (37.9, 139.0),
        "富山県": (36.7, 137.2), "石川県": (36.6, 136.6), "福井県": (35.8, 136.2),
        "山梨県": (35.6, 138.6), "長野県": (36.2, 138.0), "岐阜県": (35.7, 137.0),
        "静岡県": (34.9, 138.3), "愛知県": (35.0, 137.0), "三重県": (34.5, 136.5),
        "滋賀県": (35.2, 136.1), "京都府": (35.0, 135.8), "大阪府": (34.7, 135.5),
        "兵庫県": (35.0, 134.8), "奈良県": (34.3, 135.9), "和歌山県": (33.8, 135.5),
        "鳥取県": (35.5, 133.9), "島根県": (35.1, 132.5), "岡山県": (34.7, 133.8),
        "広島県": (34.4, 132.5), "山口県": (34.2, 131.5), "徳島県": (33.9, 134.4),
        "香川県": (34.2, 134.0), "愛媛県": (33.6, 132.8), "高知県": (33.4, 133.5),
        "福岡県": (33.6, 130.4), "佐賀県": (33.2, 130.1), "長崎県": (32.9, 129.9),
        "熊本県": (32.8, 130.7), "大分県": (33.2, 131.6), "宮崎県": (32.0, 131.4),
        "鹿児島県": (31.6, 130.5), "沖縄県": (26.2, 127.7),
    }

    JAPAN_BOUNDS = {"lat_min": 20.0, "lat_max": 46.0, "lon_min": 122.0, "lon_max": 154.0}

    def __init__(self):
        self.zone_master = ZoneMaster()

    def detect_from_latlon(self, lat: float, lon: float) -> List[ZoneInfo]:
        if not self._is_in_japan(lat, lon):
            return []
        prefecture = self._find_nearest_prefecture(lat, lon)
        zones_from_pref = self.zone_master.get_zones_for_prefecture(prefecture)
        zones_by_distance = self._find_zones_by_origin_distance(lat, lon)
        result_zones = []
        seen = set()
        for zone_num in zones_from_pref:
            if zone_num not in seen:
                zone_info = self.zone_master.get_zone(zone_num)
                if zone_info:
                    result_zones.append(zone_info)
                    seen.add(zone_num)
        for zone_num, _ in zones_by_distance[:5]:
            if zone_num not in seen:
                zone_info = self.zone_master.get_zone(zone_num)
                if zone_info:
                    result_zones.append(zone_info)
                    seen.add(zone_num)
        return result_zones

    def detect_from_point(self, point, source_crs=None) -> List[ZoneInfo]:
        if not HAS_QGIS:
            raise RuntimeError("QGIS is not available")
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        if source_crs is None or source_crs == wgs84:
            lat, lon = point.y(), point.x()
        else:
            transform = QgsCoordinateTransform(source_crs, wgs84, QgsProject.instance())
            transformed = transform.transform(point)
            lat, lon = transformed.y(), transformed.x()
        return self.detect_from_latlon(lat, lon)

    def get_primary_zone(self, lat: float, lon: float) -> Optional[ZoneInfo]:
        zones = self.detect_from_latlon(lat, lon)
        return zones[0] if zones else None

    def _is_in_japan(self, lat: float, lon: float) -> bool:
        b = self.JAPAN_BOUNDS
        return b["lat_min"] <= lat <= b["lat_max"] and b["lon_min"] <= lon <= b["lon_max"]

    def _find_nearest_prefecture(self, lat: float, lon: float) -> str:
        min_distance = float("inf")
        nearest_pref = ""
        for pref, (pref_lat, pref_lon) in self.PREFECTURE_CENTERS.items():
            distance = self._haversine_distance(lat, lon, pref_lat, pref_lon)
            if distance < min_distance:
                min_distance = distance
                nearest_pref = pref
        return nearest_pref

    def _find_zones_by_origin_distance(self, lat: float, lon: float) -> List[Tuple[int, float]]:
        distances = []
        for zone_num, zone_info in self.zone_master.get_all_zones().items():
            distance = self._haversine_distance(lat, lon, zone_info.origin_lat, zone_info.origin_lon)
            distances.append((zone_num, distance))
        distances.sort(key=lambda x: x[1])
        return distances

    def _haversine_distance(self, lat1: float, lon1: float, lat2: float, lon2: float) -> float:
        R = 6371.0
        lat1_rad = math.radians(lat1)
        lat2_rad = math.radians(lat2)
        dlat = math.radians(lat2 - lat1)
        dlon = math.radians(lon2 - lon1)
        a = math.sin(dlat / 2) ** 2 + math.cos(lat1_rad) * math.cos(lat2_rad) * math.sin(dlon / 2) ** 2
        c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))
        return R * c

    def get_zone_candidates_for_region(self, region: str) -> List[ZoneInfo]:
        zones = []
        zone_nums = self.zone_master.get_zones_for_prefecture(region)
        for zone_num in zone_nums:
            zone_info = self.zone_master.get_zone(zone_num)
            if zone_info:
                zones.append(zone_info)
        return zones
