# -*- coding: utf-8 -*-
"""
Japan CRS Selector - Main Plugin Module

This module provides the main plugin class for QGIS integration.
It handles toolbar creation, menu registration, and dialog management.

Copyright (C) 2026 Link Field
License: GPL-3.0
"""

import os
from qgis.PyQt.QtWidgets import QAction
from qgis.PyQt.QtCore import Qt, QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon


class JapanCrsSelector:
    """
    Main plugin class for Japan CRS Selector.

    This plugin helps users to identify and set the appropriate
    Japan Plane Rectangular Coordinate System (zones I-XIX) based on
    geographic coordinates.

    Attributes:
        iface: QGIS interface instance
        plugin_dir: Plugin directory path
        actions: List of QAction objects
        menu: Menu name string
        toolbar: Plugin toolbar
        main_dialog: Main dialog instance
        convert_dialog: Coordinate conversion dialog instance
    """

    def __init__(self, iface):
        """
        Initialize the plugin.

        :param iface: A QGIS interface instance.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = "&Japan CRS Selector"
        self.toolbar = self.iface.addToolBar("Japan CRS Selector")
        self.toolbar.setObjectName("JapanCrsSelector")
        self.toolbar.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
        self.main_dialog = None
        self.convert_dialog = None

        # Initialize locale
        locale_setting = QSettings().value('locale/userLocale')
        if locale_setting:
            locale = locale_setting[0:2]
        else:
            locale = 'en'
        locale_path = os.path.join(self.plugin_dir, 'i18n', f'{locale}.qm')

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        return QCoreApplication.translate('JapanCrsSelector', message)

    def _get_icon_path(self):
        """Get the path to the plugin icon."""
        return os.path.join(self.plugin_dir, "icon.png")

    def initGui(self):
        """
        Initialize the GUI elements.

        Creates toolbar buttons and menu entries for the plugin.
        """
        icon = QIcon(self._get_icon_path())

        # Main action - coordinate system detection
        self.action_main = QAction(icon, "座標系取得", self.iface.mainWindow())
        self.action_main.triggered.connect(self.run)
        self.action_main.setStatusTip("平面直角座標系を自動判別・設定")
        self.toolbar.addAction(self.action_main)
        self.iface.addPluginToMenu(self.menu, self.action_main)
        self.actions.append(self.action_main)

        # Coordinate conversion tool action
        self.action_convert = QAction("座標変換ツール", self.iface.mainWindow())
        self.action_convert.triggered.connect(self.run_convert)
        self.action_convert.setStatusTip("緯度経度と平面直角座標の相互変換")
        self.iface.addPluginToMenu(self.menu, self.action_convert)
        self.actions.append(self.action_convert)

        # Zone list display action
        self.action_zones = QAction("座標系一覧", self.iface.mainWindow())
        self.action_zones.triggered.connect(self.show_zone_list)
        self.action_zones.setStatusTip("平面直角座標系（第I〜XIX系）の一覧を表示")
        self.iface.addPluginToMenu(self.menu, self.action_zones)
        self.actions.append(self.action_zones)

    def unload(self):
        """
        Remove the plugin menu items and toolbar.

        Called when the plugin is unloaded from QGIS.
        """
        for action in self.actions:
            self.iface.removePluginMenu(self.menu, action)
            self.iface.removeToolBarIcon(action)

        del self.toolbar

        # Clean up dialogs
        if self.main_dialog:
            self.main_dialog.close()
            self.main_dialog = None
        if self.convert_dialog:
            self.convert_dialog.close()
            self.convert_dialog = None

    def run(self):
        """
        Show the main coordinate system selector dialog.

        Creates the dialog if it doesn't exist, then shows it.
        """
        from .dialogs import MainDialog
        if self.main_dialog is None:
            self.main_dialog = MainDialog(self.iface, self.iface.mainWindow())
        self.main_dialog.show()
        self.main_dialog.raise_()
        self.main_dialog.activateWindow()

    def run_convert(self):
        """
        Show the coordinate conversion tool dialog.

        Creates the dialog if it doesn't exist, then shows it.
        """
        from .dialogs import ConvertDialog
        if self.convert_dialog is None:
            self.convert_dialog = ConvertDialog(self.iface, self.iface.mainWindow())
        self.convert_dialog.show()
        self.convert_dialog.raise_()
        self.convert_dialog.activateWindow()

    def show_zone_list(self):
        """
        Show a dialog with a table of all 19 coordinate system zones.

        Displays zone information including origin coordinates,
        EPSG codes, and applicable regions.
        """
        from qgis.PyQt.QtWidgets import (
            QDialog, QVBoxLayout, QTableWidget, QTableWidgetItem, QHeaderView
        )
        from .core import ZoneMaster

        dlg = QDialog(self.iface.mainWindow())
        dlg.setWindowTitle("平面直角座標系一覧")
        dlg.setMinimumSize(750, 500)
        layout = QVBoxLayout(dlg)

        table = QTableWidget()
        table.setColumnCount(6)
        table.setHorizontalHeaderLabels([
            "系", "原点緯度", "原点経度", "JGD2011 EPSG", "JGD2000 EPSG", "適用地域"
        ])
        table.horizontalHeader().setSectionResizeMode(QHeaderView.ResizeToContents)
        table.horizontalHeader().setStretchLastSection(True)
        table.setAlternatingRowColors(True)

        master = ZoneMaster()
        zones = master.get_all_zones()
        table.setRowCount(len(zones))

        for row, (num, zone) in enumerate(zones.items()):
            table.setItem(row, 0, QTableWidgetItem(zone.zone_name))
            table.setItem(row, 1, QTableWidgetItem(zone.origin_lat_dms))
            table.setItem(row, 2, QTableWidgetItem(zone.origin_lon_dms))
            table.setItem(row, 3, QTableWidgetItem(str(zone.epsg_jgd2011)))
            table.setItem(row, 4, QTableWidgetItem(str(zone.epsg_jgd2000)))
            table.setItem(row, 5, QTableWidgetItem(", ".join(zone.regions)))

        layout.addWidget(table)
        dlg.exec_()
