# -*- coding: utf-8 -*-
"""
Japan CRS Selector - Convert Dialog
座標変換ダイアログ
"""

from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QGroupBox, QLabel, QLineEdit,
    QPushButton, QRadioButton, QButtonGroup, QComboBox, QCheckBox,
    QMessageBox, QFileDialog, QApplication
)
from qgis.PyQt.QtCore import Qt, QTimer
from qgis.PyQt.QtGui import QDoubleValidator, QColor
from qgis.core import QgsProject, QgsCoordinateReferenceSystem, QgsCoordinateTransform
from qgis.gui import QgsMapToolEmitPoint, QgsVertexMarker

from ..core import ZoneMaster, CoordinateConverter


class ConvertDialog(QDialog):
    def __init__(self, iface, parent=None):
        super().__init__(parent)
        self.iface = iface
        self.zone_master = ZoneMaster()
        self.converter = CoordinateConverter()
        self.map_tool = None
        self.previous_map_tool = None
        self.marker = None
        # モードレスダイアログにする
        self.setModal(False)
        # ダイアログを閉じた時に自動削除しない
        self.setAttribute(Qt.WA_DeleteOnClose, False)
        self.setup_ui()
        self.connect_signals()

    def setup_ui(self):
        self.setWindowTitle("座標変換ツール")
        self.setMinimumWidth(450)
        layout = QVBoxLayout(self)

        # 変換方向グループ
        dir_group = QGroupBox("変換方向")
        dir_layout = QVBoxLayout(dir_group)
        self.direction_group = QButtonGroup(self)
        self.rb_latlon_to_plane = QRadioButton("緯度経度 → 平面直角座標")
        self.rb_plane_to_latlon = QRadioButton("平面直角座標 → 緯度経度")
        self.direction_group.addButton(self.rb_latlon_to_plane, 0)
        self.direction_group.addButton(self.rb_plane_to_latlon, 1)
        self.rb_latlon_to_plane.setChecked(True)
        dir_layout.addWidget(self.rb_latlon_to_plane)
        dir_layout.addWidget(self.rb_plane_to_latlon)
        layout.addWidget(dir_group)

        # 座標系選択
        zone_layout = QHBoxLayout()
        zone_layout.addWidget(QLabel("座標系:"))
        self.zone_combo = QComboBox()
        for i in range(1, 20):
            zone = self.zone_master.get_zone(i)
            if zone:
                self.zone_combo.addItem(f"{zone.zone_name}（{i}系）", i)
        zone_layout.addWidget(self.zone_combo)
        self.datum_combo = QComboBox()
        self.datum_combo.addItems(["JGD2024", "JGD2011", "JGD2000"])
        zone_layout.addWidget(self.datum_combo)
        layout.addLayout(zone_layout)

        # 入力グループ
        input_group = QGroupBox("入力")
        input_layout = QVBoxLayout(input_group)
        coord_layout = QHBoxLayout()
        self.label_input1 = QLabel("緯度:")
        self.input1 = QLineEdit()
        self.input1.setValidator(QDoubleValidator())
        self.input1.setPlaceholderText("35.6812")
        self.label_input2 = QLabel("経度:")
        self.input2 = QLineEdit()
        self.input2.setValidator(QDoubleValidator())
        self.input2.setPlaceholderText("139.7671")
        coord_layout.addWidget(self.label_input1)
        coord_layout.addWidget(self.input1)
        coord_layout.addWidget(self.label_input2)
        coord_layout.addWidget(self.input2)
        input_layout.addLayout(coord_layout)

        # 座標取得ボタン
        get_coord_layout = QHBoxLayout()
        self.btn_get_center = QPushButton("マップ中心を取得")
        self.btn_click_map = QPushButton("地図をクリック")
        self.btn_click_map.setCheckable(True)
        get_coord_layout.addWidget(self.btn_get_center)
        get_coord_layout.addWidget(self.btn_click_map)
        input_layout.addLayout(get_coord_layout)

        self.cb_swap_xy = QCheckBox("X/Y軸反転（公共測量座標形式）")
        input_layout.addWidget(self.cb_swap_xy)
        layout.addWidget(input_group)

        # 出力グループ
        output_group = QGroupBox("出力")
        output_layout = QVBoxLayout(output_group)
        result_layout = QHBoxLayout()
        self.label_output1 = QLabel("X(E):")
        self.output1 = QLineEdit()
        self.output1.setReadOnly(True)
        self.label_output2 = QLabel("Y(N):")
        self.output2 = QLineEdit()
        self.output2.setReadOnly(True)
        self.btn_copy = QPushButton("コピー")
        result_layout.addWidget(self.label_output1)
        result_layout.addWidget(self.output1)
        result_layout.addWidget(self.label_output2)
        result_layout.addWidget(self.output2)
        result_layout.addWidget(self.btn_copy)
        output_layout.addLayout(result_layout)
        layout.addWidget(output_group)

        # 変換ボタン
        btn_layout = QHBoxLayout()
        self.btn_convert = QPushButton("変換")
        self.btn_close = QPushButton("閉じる")
        btn_layout.addWidget(self.btn_convert)
        btn_layout.addWidget(self.btn_close)
        layout.addLayout(btn_layout)

    def connect_signals(self):
        self.direction_group.buttonClicked.connect(self.update_labels)
        self.btn_convert.clicked.connect(self.do_convert)
        self.btn_copy.clicked.connect(self.copy_result)
        self.btn_close.clicked.connect(self.close)
        self.btn_get_center.clicked.connect(self.get_map_center)
        self.btn_click_map.clicked.connect(self.toggle_map_click)

    def set_zone(self, zone_num: int, datum: str):
        index = self.zone_combo.findData(zone_num)
        if index >= 0:
            self.zone_combo.setCurrentIndex(index)
        index = self.datum_combo.findText(datum)
        if index >= 0:
            self.datum_combo.setCurrentIndex(index)

    def update_labels(self):
        if self.rb_latlon_to_plane.isChecked():
            self.label_input1.setText("緯度:")
            self.label_input2.setText("経度:")
            self.label_output1.setText("X(E):")
            self.label_output2.setText("Y(N):")
            self.input1.setPlaceholderText("35.6812")
            self.input2.setPlaceholderText("139.7671")
        else:
            self.label_input1.setText("X(E):")
            self.label_input2.setText("Y(N):")
            self.label_output1.setText("緯度:")
            self.label_output2.setText("経度:")
            self.input1.setPlaceholderText("-2189.412")
            self.input2.setPlaceholderText("-32183.916")

    def do_convert(self):
        try:
            val1 = float(self.input1.text())
            val2 = float(self.input2.text())
        except ValueError:
            QMessageBox.warning(self, "入力エラー", "有効な座標値を入力してください。")
            return
        zone = self.zone_combo.currentData()
        datum = self.datum_combo.currentText()
        try:
            if self.rb_latlon_to_plane.isChecked():
                x, y = self.converter.latlon_to_plane(val1, val2, zone, datum)
                if self.cb_swap_xy.isChecked():
                    x, y = self.converter.swap_xy(x, y)
                self.output1.setText(f"{x:.3f}")
                self.output2.setText(f"{y:.3f}")
            else:
                if self.cb_swap_xy.isChecked():
                    val1, val2 = self.converter.swap_xy(val1, val2)
                lat, lon = self.converter.plane_to_latlon(val1, val2, zone, datum)
                self.output1.setText(f"{lat:.8f}")
                self.output2.setText(f"{lon:.8f}")
        except Exception as e:
            QMessageBox.critical(self, "変換エラー", str(e))

    def copy_result(self):
        text = f"{self.output1.text()}, {self.output2.text()}"
        QApplication.clipboard().setText(text)
        QMessageBox.information(self, "コピー", "クリップボードにコピーしました。")

    def get_map_center(self):
        """マップ中心座標を取得して入力欄に設定"""
        canvas = self.iface.mapCanvas()
        center = canvas.center()
        crs = canvas.mapSettings().destinationCrs()

        # マーカーを表示
        self.show_marker(center)

        if self.rb_latlon_to_plane.isChecked():
            # 緯度経度に変換
            if crs.authid() != "EPSG:4326":
                wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
                transform = QgsCoordinateTransform(crs, wgs84, QgsProject.instance())
                center_wgs = transform.transform(center)
            else:
                center_wgs = center
            self.input1.setText(f"{center_wgs.y():.8f}")
            self.input2.setText(f"{center_wgs.x():.8f}")
        else:
            # 平面直角座標に変換
            zone = self.zone_combo.currentData()
            datum = self.datum_combo.currentText()
            epsg = self.zone_master.get_epsg_code(zone, datum)
            target_crs = QgsCoordinateReferenceSystem(f"EPSG:{epsg}")
            if crs != target_crs:
                transform = QgsCoordinateTransform(crs, target_crs, QgsProject.instance())
                center_plane = transform.transform(center)
            else:
                center_plane = center
            if self.cb_swap_xy.isChecked():
                self.input1.setText(f"{center_plane.y():.3f}")
                self.input2.setText(f"{center_plane.x():.3f}")
            else:
                self.input1.setText(f"{center_plane.x():.3f}")
                self.input2.setText(f"{center_plane.y():.3f}")

    def toggle_map_click(self, checked):
        """地図クリックモードの切り替え"""
        if checked:
            self.previous_map_tool = self.iface.mapCanvas().mapTool()
            self.map_tool = QgsMapToolEmitPoint(self.iface.mapCanvas())
            self.map_tool.canvasClicked.connect(self.on_map_clicked)
            self.iface.mapCanvas().setMapTool(self.map_tool)
            self.btn_click_map.setText("クリック待機中...")
            # ダイアログを最小化して地図をクリックできるようにする
            self.showMinimized()
        else:
            self.restore_map_tool()
            self.btn_click_map.setText("地図をクリック")

    def on_map_clicked(self, point, button):
        """地図クリック時の処理"""
        canvas = self.iface.mapCanvas()
        crs = canvas.mapSettings().destinationCrs()

        # クリック位置にマーカーを表示
        self.show_marker(point)

        if self.rb_latlon_to_plane.isChecked():
            # 緯度経度に変換
            if crs.authid() != "EPSG:4326":
                wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
                transform = QgsCoordinateTransform(crs, wgs84, QgsProject.instance())
                point_wgs = transform.transform(point)
            else:
                point_wgs = point
            self.input1.setText(f"{point_wgs.y():.8f}")
            self.input2.setText(f"{point_wgs.x():.8f}")
        else:
            # 平面直角座標に変換
            zone = self.zone_combo.currentData()
            datum = self.datum_combo.currentText()
            epsg = self.zone_master.get_epsg_code(zone, datum)
            target_crs = QgsCoordinateReferenceSystem(f"EPSG:{epsg}")
            if crs != target_crs:
                transform = QgsCoordinateTransform(crs, target_crs, QgsProject.instance())
                point_plane = transform.transform(point)
            else:
                point_plane = point
            if self.cb_swap_xy.isChecked():
                self.input1.setText(f"{point_plane.y():.3f}")
                self.input2.setText(f"{point_plane.x():.3f}")
            else:
                self.input1.setText(f"{point_plane.x():.3f}")
                self.input2.setText(f"{point_plane.y():.3f}")

        # クリックモードを解除
        self.btn_click_map.setChecked(False)
        self.restore_map_tool()
        self.btn_click_map.setText("地図をクリック")
        # ダイアログを再表示
        self.showNormal()
        self.raise_()
        self.activateWindow()

    def restore_map_tool(self):
        """以前のマップツールを復元"""
        if self.previous_map_tool:
            self.iface.mapCanvas().setMapTool(self.previous_map_tool)
        self.map_tool = None

    def show_marker(self, point):
        """取得位置にマーカーを表示"""
        self.remove_marker()
        canvas = self.iface.mapCanvas()
        self.marker = QgsVertexMarker(canvas)
        self.marker.setCenter(point)
        self.marker.setColor(QColor(255, 0, 0))
        self.marker.setIconType(QgsVertexMarker.ICON_CROSS)
        self.marker.setIconSize(15)
        self.marker.setPenWidth(3)
        # 3秒後にマーカーを削除
        QTimer.singleShot(3000, self.remove_marker)

    def remove_marker(self):
        """マーカーを削除"""
        if self.marker:
            canvas = self.iface.mapCanvas()
            canvas.scene().removeItem(self.marker)
            self.marker = None

    def closeEvent(self, event):
        """ダイアログを閉じる時の処理"""
        if self.btn_click_map.isChecked():
            self.restore_map_tool()
        self.remove_marker()
        super().closeEvent(event)
