# -*- coding: utf-8 -*-
"""
Japan CRS Selector - Coordinate Converter
座標変換処理
"""

from typing import Tuple, Optional

try:
    from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsProject, QgsPointXY
    HAS_QGIS = True
except ImportError:
    HAS_QGIS = False

from .zone_master import ZoneMaster


class CoordinateConverter:
    def __init__(self):
        self.zone_master = ZoneMaster()

    def latlon_to_plane(self, lat: float, lon: float, zone: int, datum: str = "JGD2011") -> Tuple[float, float]:
        if not HAS_QGIS:
            raise RuntimeError("QGIS is not available")
        epsg_code = self.zone_master.get_epsg_code(zone, datum)
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        plane_crs = QgsCoordinateReferenceSystem(f"EPSG:{epsg_code}")
        transform = QgsCoordinateTransform(wgs84, plane_crs, QgsProject.instance())
        point = QgsPointXY(lon, lat)
        transformed = transform.transform(point)
        return (transformed.x(), transformed.y())

    def plane_to_latlon(self, x: float, y: float, zone: int, datum: str = "JGD2011") -> Tuple[float, float]:
        if not HAS_QGIS:
            raise RuntimeError("QGIS is not available")
        epsg_code = self.zone_master.get_epsg_code(zone, datum)
        plane_crs = QgsCoordinateReferenceSystem(f"EPSG:{epsg_code}")
        wgs84 = QgsCoordinateReferenceSystem("EPSG:4326")
        transform = QgsCoordinateTransform(plane_crs, wgs84, QgsProject.instance())
        point = QgsPointXY(x, y)
        transformed = transform.transform(point)
        return (transformed.y(), transformed.x())

    def swap_xy(self, x: float, y: float) -> Tuple[float, float]:
        return (y, x)

    def survey_to_gis(self, n: float, e: float) -> Tuple[float, float]:
        return (e, n)

    def gis_to_survey(self, x: float, y: float) -> Tuple[float, float]:
        return (y, x)

    def latlon_to_survey(self, lat: float, lon: float, zone: int, datum: str = "JGD2011") -> Tuple[float, float]:
        x, y = self.latlon_to_plane(lat, lon, zone, datum)
        return self.gis_to_survey(x, y)

    def survey_to_latlon(self, n: float, e: float, zone: int, datum: str = "JGD2011") -> Tuple[float, float]:
        x, y = self.survey_to_gis(n, e)
        return self.plane_to_latlon(x, y, zone, datum)

    def get_epsg_crs(self, zone: int, datum: str = "JGD2011") -> Optional["QgsCoordinateReferenceSystem"]:
        if not HAS_QGIS:
            return None
        epsg_code = self.zone_master.get_epsg_code(zone, datum)
        return QgsCoordinateReferenceSystem(f"EPSG:{epsg_code}")

    def format_coordinate(self, value: float, precision: int = 3) -> str:
        return f"{value:.{precision}f}"

    def format_latlon_dms(self, lat: float, lon: float) -> Tuple[str, str]:
        def to_dms(value: float, is_lat: bool) -> str:
            direction = "N" if value >= 0 else "S" if is_lat else "E" if value >= 0 else "W"
            value = abs(value)
            d = int(value)
            m = int((value - d) * 60)
            s = ((value - d) * 60 - m) * 60
            return f"{d}°{m:02d}′{s:05.2f}″{direction}"
        return (to_dms(lat, True), to_dms(lon, False))
