# -*- coding: utf-8 -*-
"""
Japan CRS Selector - Main Dialog
"""

from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QGroupBox, QLabel, QLineEdit,
    QPushButton, QRadioButton, QButtonGroup, QTextEdit, QComboBox,
    QMessageBox, QFrame
)
from qgis.PyQt.QtCore import Qt, QTimer
from qgis.PyQt.QtGui import QDoubleValidator, QColor
from qgis.core import QgsProject, QgsCoordinateReferenceSystem
from qgis.gui import QgsVertexMarker

from ..core import ZoneDetector, ZoneMaster, CoordinateConverter


class MainDialog(QDialog):
    def __init__(self, iface, parent=None):
        super().__init__(parent)
        self.iface = iface
        self.zone_detector = ZoneDetector()
        self.zone_master = ZoneMaster()
        self.converter = CoordinateConverter()
        self.current_zone = None
        self.marker = None
        self.convert_dialog = None
        self.setup_ui()
        self.connect_signals()

    def setup_ui(self):
        self.setWindowTitle("Japan CRS Selector - 平面直角座標系選択")
        self.setMinimumWidth(500)
        layout = QVBoxLayout(self)

        coord_group = QGroupBox("座標入力")
        coord_layout = QVBoxLayout(coord_group)

        input_layout = QHBoxLayout()
        input_layout.addWidget(QLabel("緯度:"))
        self.lat_edit = QLineEdit()
        self.lat_edit.setValidator(QDoubleValidator(-90, 90, 8))
        self.lat_edit.setPlaceholderText("35.6812")
        input_layout.addWidget(self.lat_edit)
        input_layout.addWidget(QLabel("経度:"))
        self.lon_edit = QLineEdit()
        self.lon_edit.setValidator(QDoubleValidator(-180, 180, 8))
        self.lon_edit.setPlaceholderText("139.7671")
        input_layout.addWidget(self.lon_edit)
        coord_layout.addLayout(input_layout)

        btn_layout = QHBoxLayout()
        self.btn_get_center = QPushButton("マップ中心を取得")
        self.btn_detect = QPushButton("座標系を判別")
        btn_layout.addWidget(self.btn_get_center)
        btn_layout.addWidget(self.btn_detect)
        coord_layout.addLayout(btn_layout)
        layout.addWidget(coord_group)

        datum_group = QGroupBox("測地系")
        datum_layout = QHBoxLayout(datum_group)
        self.datum_group = QButtonGroup(self)
        self.rb_jgd2024 = QRadioButton("JGD2024（2025年4月〜、推奨）")
        self.rb_jgd2011 = QRadioButton("JGD2011（2011年〜2025年3月）")
        self.rb_jgd2000 = QRadioButton("JGD2000（2002年〜2011年）")
        self.datum_group.addButton(self.rb_jgd2024, 0)
        self.datum_group.addButton(self.rb_jgd2011, 1)
        self.datum_group.addButton(self.rb_jgd2000, 2)
        self.rb_jgd2024.setChecked(True)
        datum_layout.addWidget(self.rb_jgd2024)
        datum_layout.addWidget(self.rb_jgd2011)
        datum_layout.addWidget(self.rb_jgd2000)
        layout.addWidget(datum_group)

        self.jgd2024_note = QLabel("※JGD2024: 水平座標はJGD2011と同一（現在JGD2011のEPSGを使用）\n　QGIS 4.0（2026年2月予定）で正式対応見込み")
        self.jgd2024_note.setStyleSheet("color: gray; font-size: 10px;")
        layout.addWidget(self.jgd2024_note)

        result_group = QGroupBox("判別結果")
        result_layout = QVBoxLayout(result_group)
        self.zone_combo = QComboBox()
        result_layout.addWidget(self.zone_combo)
        self.result_text = QTextEdit()
        self.result_text.setReadOnly(True)
        self.result_text.setMaximumHeight(120)
        result_layout.addWidget(self.result_text)
        layout.addWidget(result_group)

        button_layout = QHBoxLayout()
        self.btn_set_crs = QPushButton("プロジェクトCRSに設定")
        self.btn_set_crs.setEnabled(False)
        self.btn_convert = QPushButton("座標変換ツール")
        self.btn_close = QPushButton("閉じる")
        button_layout.addWidget(self.btn_set_crs)
        button_layout.addWidget(self.btn_convert)
        button_layout.addWidget(self.btn_close)
        layout.addLayout(button_layout)

    def connect_signals(self):
        self.btn_get_center.clicked.connect(self.get_map_center)
        self.btn_detect.clicked.connect(self.detect_zone)
        self.btn_set_crs.clicked.connect(self.set_project_crs)
        self.btn_convert.clicked.connect(self.open_convert_dialog)
        self.btn_close.clicked.connect(self.close)
        self.zone_combo.currentIndexChanged.connect(self.on_zone_changed)
        self.datum_group.buttonClicked.connect(self.update_result_display)

    def get_map_center(self):
        canvas = self.iface.mapCanvas()
        center_map = canvas.center()
        crs = canvas.mapSettings().destinationCrs()

        # マーカーを表示（マップ座標系で）
        self.show_marker(center_map)

        # WGS84に変換
        if crs.authid() != "EPSG:4326":
            from qgis.core import QgsCoordinateTransform
            transform = QgsCoordinateTransform(crs, QgsCoordinateReferenceSystem("EPSG:4326"), QgsProject.instance())
            center = transform.transform(center_map)
        else:
            center = center_map
        self.lat_edit.setText(f"{center.y():.6f}")
        self.lon_edit.setText(f"{center.x():.6f}")

    def show_marker(self, point):
        """取得位置にマーカーを表示"""
        # 既存のマーカーを削除
        self.remove_marker()

        canvas = self.iface.mapCanvas()
        self.marker = QgsVertexMarker(canvas)
        self.marker.setCenter(point)
        self.marker.setColor(QColor(255, 0, 0))
        self.marker.setIconType(QgsVertexMarker.ICON_CROSS)
        self.marker.setIconSize(15)
        self.marker.setPenWidth(3)

        # 3秒後にマーカーを削除
        QTimer.singleShot(3000, self.remove_marker)

    def remove_marker(self):
        """マーカーを削除"""
        if self.marker:
            canvas = self.iface.mapCanvas()
            canvas.scene().removeItem(self.marker)
            self.marker = None

    def detect_zone(self):
        try:
            lat = float(self.lat_edit.text())
            lon = float(self.lon_edit.text())
        except ValueError:
            QMessageBox.warning(self, "入力エラー", "有効な緯度・経度を入力してください。")
            return
        zones = self.zone_detector.detect_from_latlon(lat, lon)
        self.zone_combo.clear()
        if not zones:
            self.result_text.setText("該当する座標系が見つかりませんでした。\n日本国内の座標を入力してください。")
            self.btn_set_crs.setEnabled(False)
            return
        for zone in zones:
            self.zone_combo.addItem(f"{zone.zone_name}（{zone.zone_number}系）", zone.zone_number)
        self.btn_set_crs.setEnabled(True)
        self.on_zone_changed(0)

    def on_zone_changed(self, index):
        if index < 0:
            return
        zone_num = self.zone_combo.currentData()
        if zone_num is None:
            return
        self.current_zone = self.zone_master.get_zone(zone_num)
        self.update_result_display()

    def get_selected_datum(self) -> str:
        if self.rb_jgd2024.isChecked():
            return "JGD2024"
        elif self.rb_jgd2011.isChecked():
            return "JGD2011"
        else:
            return "JGD2000"

    def update_result_display(self):
        if self.current_zone is None:
            return
        zone = self.current_zone
        datum = self.get_selected_datum()
        epsg = zone.get_epsg_code(datum)
        text = f"座標系: {zone.zone_name}（{zone.zone_number}系）\n"
        text += f"EPSG: {epsg}（{datum}）\n"
        text += f"原点: {zone.origin_lat_dms} {zone.origin_lon_dms}\n"
        text += f"適用地域: {', '.join(zone.regions)}"
        self.result_text.setText(text)

    def set_project_crs(self):
        if self.current_zone is None:
            return
        datum = self.get_selected_datum()
        epsg = self.current_zone.get_epsg_code(datum)
        crs = QgsCoordinateReferenceSystem(f"EPSG:{epsg}")
        if not crs.isValid():
            QMessageBox.critical(self, "エラー", f"CRS EPSG:{epsg} を設定できませんでした。")
            return
        QgsProject.instance().setCrs(crs)
        QMessageBox.information(self, "完了", f"プロジェクトCRSを {self.current_zone.zone_name}（EPSG:{epsg}）に設定しました。")

    def open_convert_dialog(self):
        from .convert_dialog import ConvertDialog
        if self.convert_dialog is None:
            self.convert_dialog = ConvertDialog(self.iface, self.iface.mainWindow())
        if self.current_zone:
            self.convert_dialog.set_zone(self.current_zone.zone_number, self.get_selected_datum())
        self.convert_dialog.show()
        self.convert_dialog.raise_()
        self.convert_dialog.activateWindow()
