# -*- coding: utf-8 -*-
"""
/***************************************************************************
                                 A QGIS plugin
 This plugin gives access to Israeli open spatial data sources
                             -------------------
        begin                : 2020-04-01
        copyright            : (C) 2020-2034 by Kaplan Open Source Consulting Inc.
        email                : lior@kaplanopensource.co.il
        git sha              : $Format:%H$
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QUrl, QByteArray
from PyQt5.QtGui import QIcon, QPixmap, QDesktopServices
from PyQt5.QtWidgets import QAction, QTreeWidgetItem, QAbstractItemView, QTreeWidget
from PyQt5.QtNetwork import QNetworkAccessManager, QNetworkRequest
from qgis.PyQt.QtNetwork import QNetworkRequest, QNetworkProxy
import requests as rq
import json
import zlib
import codecs
from urllib.parse import urlparse #test of URL is valid

import os # This is is needed in the pyqgis console also
from qgis.core import QgsVectorLayer, QgsRasterLayer, Qgis, QgsProject, QgsLayerTreeLayer, QgsLayerTreeGroup, QgsSettings,QgsNetworkAccessManager,QgsError
# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .opendata_loader_dialog import opendata_loaderDialog #, UserServiceDialog
#from .data_sources_list import *
import os.path
import io
import zipfile
import tempfile
from qgis.gui import *

class OpenDataLoader:
    """QGIS Plugin Implementation."""
    
    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        self.mb = self.iface.messageBar()
        self.dlg = opendata_loaderDialog()
        # added form for adding user services
        #self.formdlg = UserServiceDialog()
        #self.populateExistingUserServices()
        self.headers = {"Agent":"QGIS"}
        self.prefix = 'https://qgis-plugin.kaplanopensource.co.il'
        self.proxyHandler = {"url":"","proxy":""}
        #self.dlg.treeView.headerItem().setText(0, "Gov Orgs")
        
        
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        f = open(os.path.join(self.plugin_dir,"metadata.txt"), "r")
        lines = f.readlines()
        self.version = 'v'+ [line for line in lines if line.startswith("version")][0][:-1].split("=")[1]
        # initialize locale
        """
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'opendata_loader_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)
        """
        # Declare instance attributes
        self.actions = []
        
        self.menu = self.tr(u'&Israeli Open Data Sources')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('opendata_loader', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToWebMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/israeli_opendata_loader/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Add Israeli open data'),
            callback=self.run,
            parent=self.iface.mainWindow())
        
        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginWebMenu(
                self.tr(u'&Israeli Open Data Sources'),
                action)
            self.iface.removeToolBarIcon(action)
            
    def openRegisterButton(self):
        QDesktopServices.openUrl(QUrl('https://kaplanopensource.co.il/services/qgis'))

    def clearSelection(self):
        
        selectedAll = []
        selectedAll.append(self.dlg.govTree.selectedItems())
        selectedAll.append(self.dlg.muniTree.selectedItems())
        selectedAll.append(self.dlg.orgTree.selectedItems())
        selectedAll.append(self.dlg.opendataTree.selectedItems())
        # selectedAll.append(self.dlg.userServicesTree.selectedItems())
        selected = []
        for i in selectedAll:
            selected += i
        
        for item in selected:
            item.setSelected(False)

        self.dlg.govTree.clearSelection()
        self.dlg.muniTree.clearSelection()
        self.dlg.orgTree.clearSelection()
        self.dlg.opendataTree.clearSelection()
        # self.dlg.userServicesTree.clearSelection()
            

    def preRequest(self, request):
        '''
        Before making url request check if proxy needs to be added
        '''
        url = request.url().toString()
        if url.startswith(self.proxyHandler["proxy_mask"]):
            url = self.proxyHandler["proxy"] + "?" + url
        request.setUrl(QUrl(url))
        
    def getOrgs(self, root):
        baseObject = {}
        files = os.listdir(root)
        for file in files:
            if file.endswith('.json'):
                org = os.path.splitext(file)
                path = os.path.join(root, file)
                with open(path, encoding='utf-8') as json_file:
                    baseObject[org] = json.load(json_file)
        return baseObject   


    def buildDataList(self):
        
        try:
            dataList = json.loads(codecs.decode(self.checkCredentials().decode('utf-8'),'rot_13'))

            if dataList:
                if dataList['type'] == 'paid':
                    self.dlg.teaserGov.hide()
                    self.dlg.teaserMuni.hide()
                    self.dlg.versionLabel.setText('משתמש משלם')
                else:
                    self.dlg.versionLabel.setText('משתמש רשום (ללא תשלום)')

                if dataList['type'] == 'free':
                    self.dlg.teaserGov.show()
                    self.dlg.teaserGov.setText('שכבות וארגונים נוספים זמינים למשתמשים משלמים')
                    self.dlg.teaserMuni.show()
                    self.dlg.teaserMuni.setText('שכבות וארגונים נוספים זמינים למשתמשים משלמים')

                if dataList['type'] == 'overLimit':
                    self.mb.pushWarning('אזהרה', 'הגעת למגבלת השימוש החודשית, ניתן ליצור קשר להרחבת השימוש')
                    self.dlg.teaserGov.show()
                    self.dlg.teaserGov.setText('שכבות וארגונים נוספים זמינים למשתמשים משלמים')
                    self.dlg.teaserMuni.show()
                    self.dlg.teaserMuni.setText('שכבות וארגונים נוספים זמינים למשתמשים משלמים')

                if dataList['type'] == 'notRegistered':
                    self.dlg.versionLabel.setText('משתמש לא רשום')
                    self.dlg.teaserGov.show()
                    self.dlg.teaserGov.setText('שכבות וארגונים נוספים זמינים למשתמשים רשומים')
                    self.dlg.teaserMuni.show()
                    self.dlg.teaserMuni.setText('שכבות וארגונים נוספים זמינים למשתמשים משלמים')

                    # Disabled feature
                    # self.dlg.teaserUserServices.show()
                    # self.dlg.teaserUserServices.setText('שכבות וארגונים נוספים זמינים למשתמשים רשומים')
                else:
                    self.dlg.loginsCounter.setText('החודש הוספת {} שכבות'.format(dataList['logins']))
                
                # Disabled feature
                # if 'orgName' in dataList:
                #     if len(dataList['orgName']) > 0:
                #         if 'userServices' in dataList['data'] and len(dataList['data']['userServices']) > 0:
                #             self.dlg.teaserUserServices.hide()
                #         else:
                #             self.dlg.teaserUserServices.setText('יש להוסיף סרביס ראשון\n  על מנת לפתוח את הטאב')
                #     else:
                #         self.dlg.teaserUserServices.setText('מוזמנים לפנות אלינו \n על מנת לפתוח את האפשרות')

        except:
            self.mb.pushMessage('תקלת חיבור','נראה שיש תקלה בחיבור לשרת, נא לדווח על השגיאה',Qgis.Warning, 5)
            # raise Exception
                
        return dataList['data']

    def getWithProxy(self, url, headers=None):
        s = QgsSettings()
        proxy = QNetworkProxy()
        proxyEnabled = s.value("proxy/proxyEnabled", "")
        proxyType = s.value("proxy/proxyType", "" )
        proxyHost = s.value("proxy/proxyHost", "" )
        proxyPort = s.value("proxy/proxyPort", "" )
        proxyUser = s.value("proxy/proxyUser", "" )
        proxyPassword = s.value("proxy/proxyPassword", "" )
        if not self.QNM:
            self.QNM = QgsNetworkAccessManager()
        self.QNM.setTimeout(20000)
        if proxyEnabled == "true":
            proxy.setType(QNetworkProxy.HttpProxy)
            proxy.setHostName(proxyHost)
            if proxyPort != "":
                proxy.setPort(int(proxyPort))
            proxy.setUser(proxyUser)
            proxy.setPassword(proxyPassword)
            QNetworkProxy.setApplicationProxy(proxy)
            self.QNM.setupDefaultProxyAndCache()
            self.QNM.setFallbackProxyAndExcludes(proxy,[""],[""])
        request = QNetworkRequest(QUrl(url))
        if headers:
            for header in headers.keys():
                request.setRawHeader(header, headers[header])
        reply = self.QNM.blockingGet(request)
        return reply.content().data()

    def postWithProxy(self, url, headers=None, data=None):
        s = QgsSettings()
        proxy = QNetworkProxy()
        proxyEnabled = s.value("proxy/proxyEnabled", "")
        proxyType = s.value("proxy/proxyType", "" )
        proxyHost = s.value("proxy/proxyHost", "" )
        proxyPort = s.value("proxy/proxyPort", "" )
        proxyUser = s.value("proxy/proxyUser", "" )
        proxyPassword = s.value("proxy/proxyPassword", "" )
        self.QNM = QgsNetworkAccessManager()
        self.QNM.setTimeout(20000)
        if proxyEnabled == "true":
            proxy.setType(QNetworkProxy.HttpProxy)
            proxy.setHostName(proxyHost)
            if proxyPort != "":
                proxy.setPort(int(proxyPort))
            proxy.setUser(proxyUser)
            proxy.setPassword(proxyPassword)
            QNetworkProxy.setApplicationProxy(proxy)
            self.QNM.setupDefaultProxyAndCache()
            self.QNM.setFallbackProxyAndExcludes(proxy,[""],[""])
        request = QNetworkRequest(QUrl(url))
        if headers:
            for header in headers.keys():
                headerKey = QByteArray()
                headerKey.append(header)
                headerVal = QByteArray()
                headerVal.append(headers[header])
                request.setRawHeader(headerKey, headerVal)
        postdata = QByteArray()
        if data:
            for key in data.keys():
                postdata.append(key).append('=').append(data[key]).append("&")
        return self.QNM.blockingPost(request, postdata)

    def loadLayers(self):
        """
        Initial list filling function
        """
        self.dlg.govTree.clear()
        self.dlg.muniTree.clear()
        self.dlg.orgTree.clear()
        self.dlg.opendataTree.clear()
        
        # Disabled feature
        # self.dlg.userServicesTree.clear()
        
        self.dataList = self.buildDataList()
        if "govOrgs" in self.dataList and len(self.dataList["govOrgs"]) > 0:
            orgI = -1
            for (orgI, org) in enumerate(self.dataList["govOrgs"].values()):
                orgItem = QTreeWidgetItem(None, [org["hebName"]])
                layers = org["layers"]
                for layer in layers:
                    TreeItemName = layer["layerHebName"]
                    layerItem = QTreeWidgetItem(orgItem,[TreeItemName])
                    self.defineLayerItemIcon(layer,layerItem)
                self.dlg.govTree.insertTopLevelItems(orgI, [orgItem])
                self.dlg.govTree.sortItems(0,Qt.AscendingOrder)
        
        if "municipalities" in self.dataList and len(self.dataList["municipalities"]) > 0:
            munisI = -1
            for (munisI, muni) in enumerate(self.dataList["municipalities"].values()):
                muniItem = QTreeWidgetItem(None, [muni["hebName"]])
                layers = muni["layers"]
                for layer in layers:
                    TreeItemName = layer["layerHebName"]
                    layerItem = QTreeWidgetItem(muniItem,[TreeItemName])
                    self.defineLayerItemIcon(layer,layerItem)

                self.dlg.muniTree.insertTopLevelItems(munisI, [muniItem])
                self.dlg.muniTree.sortItems(0,Qt.AscendingOrder)

        if "NGO" in self.dataList and len(self.dataList["NGO"]) > 0:
            NGOI = -1
            for (NGOI, ngo) in enumerate(self.dataList["NGO"].values()):
                NGOItem = QTreeWidgetItem(None, [ngo["hebName"]])
                layers = ngo["layers"]
                for layer in layers:
                    TreeItemName = layer["layerHebName"]
                    layerItem = QTreeWidgetItem(NGOItem,[TreeItemName])
                    self.defineLayerItemIcon(layer,layerItem)

                self.dlg.orgTree.insertTopLevelItems(NGOI, [NGOItem])
                self.dlg.orgTree.sortItems(0,Qt.AscendingOrder)

        if "ods" in self.dataList and len(self.dataList["ods"]) > 0:
            odsI = -1
            for (odsI, ods) in enumerate(self.dataList["ods"].values()):
                odsItem = QTreeWidgetItem(None, [ods["hebName"]])
                layers = ods["layers"]
                for layer in layers:
                    TreeItemName = layer["layerHebName"]
                    layerItem = QTreeWidgetItem(odsItem,[TreeItemName])
                    self.defineLayerItemIcon(layer,layerItem)
                self.dlg.opendataTree.insertTopLevelItems(odsI, [odsItem])
                self.dlg.opendataTree.sortItems(0,Qt.AscendingOrder)
        
        # Disabled feature
        # if "userServices" in self.dataList and len(self.dataList["userServices"]) > 0:
        #     self.dlg.tabWidget.setTabEnabled(4,True)
        #     for (LayerI, layer) in enumerate(self.dataList["userServices"]):
        #         layerItemName = layer['layerHebName']
        #         layerItem = QTreeWidgetItem(None,[layerItemName])
        #         self.defineLayerItemIcon(layer,layerItem)
        #         self.dlg.userServicesTree.insertTopLevelItems(LayerI, [layerItem])

    def checkCredentials(self):
        try:
            userEmail = self.dlg.emailInput.text()
            userKey = self.dlg.keyInput.text()
            userCreds = {'email':userEmail, 'key':userKey}
            self.reply = self.postWithProxy(f"{self.prefix}/json", self.headers, userCreds)
            obj = zlib.decompress(self.reply.content())
            self.storeCredentials()
        except Exception as e:
            error = QgsError("התרחשה תקלה בחיבור נא להעביר את ההודעה המצורפת","תקלה")
            error.append(str(e),"תקלת חיבור")
            QgsErrorDialog(error,"network error").show(error,"Israeli Open Data Loader Network Error")
            # raise Exception
        return obj
        
    def storeCredentials(self):
        s = QgsSettings()
        userEmail = self.dlg.emailInput.text()
        userKey = self.dlg.keyInput.text()
        s.setValue('opendata_loader/email', userEmail)
        s.setValue('opendata_loader/key', userKey)
    
    def loadCredentials(self):
        s = QgsSettings()
        userEmail = s.value('opendata_loader/email','')
        userKey = s.value('opendata_loader/key','')
        self.dlg.emailInput.setText(userEmail)
        self.dlg.keyInput.setText(userKey)

#    Disabled feature
#    def addExisting(self):
#        data = self.formdlg.ExistingLayers.currentData()
#        data['layerUrl'] = bytearray(data['layerUrl'],'utf-8').hex()
#        s = QgsSettings()
#        userEmail = s.value('opendata_loader/email','')
#        userKey = s.value('opendata_loader/key','')
#        userCreds = {'email':userEmail, 'key':userKey, 'layer': json.dumps(data, ensure_ascii=False), 'type': 'insert'}
#        self.reply = self.postWithProxy(f"{self.prefix}/service", self.headers, userCreds)
#        if self.reply.attribute(QNetworkRequest.HttpStatusCodeAttribute) == 200:
#            self.loadLayers()
#        else:
#            self.mb.pushMessage('תקלה','התרחשה תקלה בהוספת השכבה',Qgis.Warning, 5)

    def defineLayerItemIcon(self,layer,layerItem):
        # Legacy support
        if 'fileType' in layer.keys():
            typeFormat = 'fileType'
        else:
            typeFormat = 'connectionType'
        if layer[typeFormat] == 'GeoJSON' or layer[typeFormat] == 'shp' or layer[typeFormat] == 'vector':
            layerItem.setIcon(0,QIcon(":images/themes/default/mIconVector.svg"))
        if layer[typeFormat] == "connections-arcgismapserver" or layer[typeFormat] == "wms" or layer[typeFormat] == "connections-wms" or layer[typeFormat] == 'raster':
            layerItem.setIcon(0,QIcon(":images/themes/default/mIconWms.svg"))
        if layer[typeFormat] == "connections-arcgisfeatureserver" or layer[typeFormat] == "wfs" or layer[typeFormat] == "connections-wfs":
            layerItem.setIcon(0,QIcon(":images/themes/default/mIconWfs.svg"))

#    Disabled feature
#    def removeDuplicateUserServices(self, userServices):
#        seen = set()
#        result = []
#        for layer in userServices:
#            key = (layer['layerHebName'], layer['layerUrl'])
#            if key in seen:
#                continue
#
#            result.append(layer)
#            seen.add(key)
#        return result
#
#    Disabled feature
#    def filterServices(self, layer):
#        providerType = layer.providerType()
#        acceptedTypes = ['wms','WFS', 'arcgismapserver', 'arcgisfeatureserver']
#        if providerType in acceptedTypes:
#            return True

#    Disabled feature
#    def populateExistingUserServices(self):
#        self.formdlg.ExistingLayers.clear()
#        self.formdlg.ExistingLayers.line_edit = self.formdlg.ExistingLayers.lineEdit()
#        self.formdlg.ExistingLayers.line_edit.setAlignment(Qt.AlignCenter)
#        self.formdlg.ExistingLayers.line_edit.setReadOnly(True)
#
#        mapLayers = QgsProject.instance().mapLayers().values()
#        mapServices = [{'layerHebName':layer.name(),'layerUrl':layer.source(),'tempLayerType':layer.providerType(),'layerCRS':layer.crs().authid() } for layer in mapLayers if self.filterServices(layer)]
#        mapServices = sorted(mapServices, key=lambda k: k['layerHebName'])
#        for layer in mapServices:
#            if layer['tempLayerType'] == 'wms' or layer['tempLayerType'] == 'arcgismapserver':
#                icon = QIcon(":images/themes/default/mIconWms.svg")
#            else:
#                icon = QIcon(":images/themes/default/mIconWfs.svg")
#            name = layer['layerHebName']
#            self.formdlg.ExistingLayers.addItem(icon,name, userData = layer)
#        self.formdlg.ExistingLayers.insertSeparator(len(mapServices))
#        s = QgsSettings()
#        keys = s.allKeys()
#        mapserverKeys = [key for key in keys if key.startswith('qgis/connections-arcgismapserver') or key.startswith('qgis/connections-wms')]
#        featureserverKeys = [key for key in keys if key.startswith('qgis/connections-arcgisfeatureserver') or key.startswith('qgis/connections-wfs')]
#        mapserverservices = list(set([key.split('/')[2] for key in mapserverKeys]))
#        mapserverservices.sort()
#        for service in mapserverservices:
#            selectedKeys = [key for key in mapserverKeys if key.split('/')[2] == service]
#            layerName = service
#            layerUrl = ''
#            for key in selectedKeys:
#                if key.endswith('url'):
#                    layerUrl = s.value(key,'')
#                    continue
#                else:
#                    pass
#            #layerUrl = s.value([key for key in selectedKeys if key.endswith('url')][0],'')
#            tempLayerType = 'arcgismapserver'
#            connectionType = 'connections-arcgismapserver'
#            layer =  {
#                        "connectionType": connectionType,
#                        "layerEngName": layerName,
#                        "layerHebName": layerName,
#                        "layerUrl":layerUrl,
#                        "tempLayerType": tempLayerType
#                    }
#            if len(layerUrl) > 1:
#                self.formdlg.ExistingLayers.addItem(QIcon(":images/themes/default/mIconWms.svg"),service, userData = layer)
#
#        featureserverservices = list(set([key.split('/')[2] for key in featureserverKeys]))
#        featureserverservices.sort()
#        for service in featureserverservices:
#            selectedKeys = [key for key in featureserverKeys if key.split('/')[2] == service]
#            layerName = service
#            layerUrl = ''
#            for key in selectedKeys:
#                if key.endswith('url'):
#                    layerUrl = s.value(key,'')
#                    continue
#                else:
#                    pass
#            #layerUrl = s.value([key for key in selectedKeys if key.endswith('url')][0],'')
#            tempLayerType = 'arcgisfeatureserver'
#            connectionType = 'connections-arcgisfeatureserver'
#            layer =  {
#                        "connectionType": connectionType,
#                        "layerEngName": layerName,
#                        "layerHebName": layerName,
#                        "layerUrl":layerUrl,
#                        "tempLayerType": tempLayerType
#                    }
#            if len(layerUrl) > 1:
#                self.formdlg.ExistingLayers.addItem(QIcon(":images/themes/default/mIconWfs.svg"), service, userData = layer)

#    Disabled feature
#    def openUserServicesForm(self):
#        self.populateExistingUserServices()
#        self.formdlg.open()

#    Disabled feature
#    def addUserService(self):
#        layerName = self.formdlg.UserLayerHebName.text()
#        layerURL = bytearray(self.formdlg.UserLayerURL.text(),'utf-8').hex()
#        layer = {"layerHebName": layerName,
#            "layerUrl" : layerURL
#            }
#        s = QgsSettings()
#        userEmail = s.value('opendata_loader/email','')
#        userKey = s.value('opendata_loader/key','')
#        userCreds = {'email':userEmail, 'key':userKey, 'layer': json.dumps(layer, ensure_ascii=False), 'type': 'insert'}
#        self.reply = self.postWithProxy(f"{self.prefix}/service", self.headers, userCreds)
#        if self.reply.attribute(QNetworkRequest.HttpStatusCodeAttribute) == 200:
#            self.loadLayers()
#        else:
#            self.mb.pushMessage('תקלה','התרחשה תקלה בהוספת השכבה',Qgis.Warning, 5)

#    Disabled feature
#    def addPluginServiceToUserServices(self):
#
#        if self.dlg.tabWidget.currentWidget().findChildren(QTreeWidget)[0].selectedItems():
#            selectedItem = self.dlg.tabWidget.currentWidget().findChildren(QTreeWidget)[0].selectedItems()[0]
#        else:
#            self.mb.pushInfo('',"יש לבצע בחירה על מנת להוסיף שירותים")
#            return
#        if self.dlg.tabWidget.currentWidget().objectName() != 'userServicesTab':
#            if selectedItem.parent() is None:
#                self.mb.pushWarning('Warning',"הוספה של ארגונים שלמים אינה נתמכת")
#                return
#
#            parent = selectedItem.parent()
#            parent_text = parent.text(0)
#            layers = []
#            for muni in self.dataList["municipalities"].values():
#                if parent_text == muni["hebName"]:
#                    layers.extend(muni["layers"])
#
#            for govOrg in self.dataList["govOrgs"].values():
#                if parent_text == govOrg["hebName"]:
#                    layers.extend(govOrg["layers"])
#
#            for org in self.dataList["NGO"].values():
#                if parent_text == org["hebName"]:
#                    layers.extend(org["layers"])
#
#            for ods in self.dataList["ods"].values():
#                if parent_text == ods["hebName"]:
#                    layers.extend(ods["layers"])
#
#            for layer in layers:
#                if selectedItem.text(0) == layer["layerHebName"]:
#                    selectedLayer = layer
#
#            layerName = parent_text + ' - ' + selectedLayer["layerHebName"]
#            layerURL = bytearray(selectedLayer["layerUrl"],'utf-8').hex()
#
#            layer = {
#                "layerHebName": layerName,
#                "layerUrl" : layerURL,
#                "tempLayerType": selectedLayer['tempLayerType']
#            }
#            s = QgsSettings()
#            userEmail = s.value('opendata_loader/email','')
#            userKey = s.value('opendata_loader/key','')
#            userCreds = {'email':userEmail, 'key':userKey, 'layer': json.dumps(layer, ensure_ascii=False), 'type': 'insert'}
#            self.reply = self.postWithProxy(f"{self.prefix}/service", self.headers, userCreds)
#            if self.reply.attribute(QNetworkRequest.HttpStatusCodeAttribute) == 200:
#                self.loadLayers()
#            else:
#                self.mb.pushMessage('תקלה','התרחשה תקלה בהוספת השכבה',Qgis.Warning, 5)
#
    def addTempArcgisFeature(self, layer, crs=None, group=False, allGroup=None):
        """
        :param crs: A coordinate reference system, preferebly using EPSG codes.
            for example: 'EPSG:3857'
        """
        url = layer["layerUrl"]
        metdataUrl = "{}?f=pjson".format(url)

        # if layer includes proxy create a preporcess request
        if "proxy" in layer.keys():
            self.proxyHandler['proxy_mask'] = layer['proxy_mask']
            self.proxyHandler['proxy'] = layer['proxy']
            nam = QgsNetworkAccessManager.instance()
            nam.setRequestPreprocessor(self.preRequest)

        try:
            answer_from_proxy = self.getWithProxy(metdataUrl)
            if answer_from_proxy:
                data = json.loads(answer_from_proxy)
            else:
                self.mb.pushCritical('Error',"The layer is not available, please report layer: \"{}\" is not available".format(layer["layerHebName"]))
                return "error"
            
        except rq.exceptions.SSLError:
            self.mb.pushCritical('SSL Error'," could not get layers from url")
            return "error"
        except:
            self.mb.pushCritical('Error',"The layer is not available, please report layer: \"{}\" is not available".format(layer["layerHebName"]))
            return "error"

        else:
            if crs is None:
                spatialReference = data.get("spatialReference", {})
                latestWkid = spatialReference.get("latestWkid", 3857)
                crs = "EPSG:{}".format(latestWkid)
            if "layers" in data:
                layers = data["layers"]
                if len(layers) == 1:
                    l = layers[0]
                    layerUrl = "crs='{}' url='{}/{}'".format(crs, url,l["id"])
                    vlayer = QgsVectorLayer(layerUrl, l["name"], "arcgisfeatureserver")
                    if group:
                        QgsProject.instance().addMapLayer(vlayer, False)
                        allGroup.addLayer(vlayer)
                    else:
                        QgsProject.instance().addMapLayer(vlayer)
                else:
                    groupName=layer["layerHebName"]
                    #root = QgsProject.instance().layerTreeRoot()
                    layerTree = self.iface.layerTreeCanvasBridge().rootGroup()
                    group1 = QgsLayerTreeGroup(groupName, False)
                    group0 = group1
                    if group:
                        allGroup.insertChildNode(-1,group0)
                    else:
                        layerTree.insertChildNode(0, group1)
                    for l in layers:
                        if 'type' in l and l["type"] == 'Group Layer':
                            if l == layers[0]:
                                continue
                            else:
                                group1 = QgsLayerTreeGroup(l["name"], False)
                                group0.insertChildNode(-1,group1)
                        layerUrl = "crs='{}' url='{}/{}'".format(crs, url,l["id"])
                        vlayer = QgsVectorLayer(layerUrl, l["name"], "arcgisfeatureserver")
                        QgsProject.instance().addMapLayer(vlayer, False)
                        group0.addLayer(vlayer)
                    #group = root.addGroup(groupName)
            else:
                layerUrl = "crs='{}' url='{}'".format(crs, url)
                vlayer = QgsVectorLayer(layerUrl,  layer["layerHebName"], "arcgisfeatureserver")
                if group:
                    QgsProject.instance().addMapLayer(vlayer, False)
                    allGroup.addLayer(vlayer)
                else:
                    QgsProject.instance().addMapLayer(vlayer)
            return data

    def addTempArcgisMap(self, layer, crs=None, group=False, allGroup=None):
        """
        :param url: arcgis server mapserver address.
        
        :param crs: A coordinate reference system, preferebly using EPSG codes.
            for example: 'EPSG:3857'
        """
        url = layer["layerUrl"]
        metdataUrl = "{}?f=pjson".format(url)
        try:

            answer_from_proxy = self.getWithProxy(metdataUrl)
            if answer_from_proxy:
                data = json.loads(self.getWithProxy(metdataUrl))
            else:
                self.mb.pushCritical('Error',"The layer is not available, please report layer: \"{}\" is not available".format(layer["layerHebName"]))
                return "error"
        except rq.exceptions.SSLError:
            self.mb.pushCritical('SSL Error'," could not get layers from url")
            return "error"
        except:
            self.mb.pushCritical('Error',"The layer is not available, please report layer: \"{}\" is not available".format(layer["layerHebName"]))
            return "error"
        else:
            if crs is None:
                spatialReference = data.get("spatialReference", {})
                latestWkid = spatialReference.get("latestWkid", 3857)
                crs = "EPSG:{}".format(latestWkid)
                
            if "layers" in data:
                layers = data["layers"]
                if len(layers) == 1:
                    l = layers[0]    
                    layerUrl = "crs='{}' format='PNG32' layer='{}' url='{}'".format(crs, l["id"], url)
                    rlayer = QgsRasterLayer(layerUrl, l["name"], "arcgismapserver")
                    if group:
                        QgsProject.instance().addMapLayer(rlayer, False)
                        allGroup.addLayer(rlayer)
                    else:
                        QgsProject.instance().addMapLayer(rlayer)
                else:
                    groupName=layer["layerHebName"]
                    #root = QgsProject.instance().layerTreeRoot()
                    layerTree = self.iface.layerTreeCanvasBridge().rootGroup()
                    groupLayer = QgsLayerTreeGroup(groupName, False)
                    group0 = groupLayer
                    if group:
                        allGroup.insertChildNode(-1,group0)
                    else:
                        layerTree.insertChildNode(0, group0)
                    for l in layers:
                        if 'type' in l and l["type"] == 'Group Layer':
                            if l == layers[0]:
                                continue
                            else:
                                group = QgsLayerTreeGroup(l["name"], False)
                                group0.insertChildNode(-1,group)
                        layerUrl = "crs='{}' format='PNG32' layer='{}' url='{}'".format(crs, l["id"], url)
                        rlayer = QgsRasterLayer(layerUrl, l["name"], "arcgismapserver")
                        QgsProject.instance().addMapLayer(rlayer, False)
                        group0.addLayer(rlayer)
            else:
                layerUrl = "crs='{}' format='PNG32' url='{}'".format(crs, url)
                rlayer = QgsRasterLayer(layerUrl,  layer["layerHebName"], "arcgismapserver")
                if group:
                    QgsProject.instance().addMapLayer(rlayer, False)
                    allGroup.addLayer(rlayer)
                else:
                    QgsProject.instance().addMapLayer(rlayer)

    def esriJsonToGeoJson(self, url):
        base_object = {'type': "FeatureCollection",'features':[]}
        r = json.load(self.getWithProxy(url))
        esriJson = r.json()
        geomType = esriJson['geometryType']
        if 'spatialRefernce' in esriJson:
            wkid = esriJson['spatialReference']['wkid']
        else:
            wkid = False
        #wkid = esriJson.get('spatialReference'.{}).get('wkid', False)
        if wkid and wkid != 4326:
            base_object["crs"] = {"type": "link",
                          "properties": {
                            "href": "https://www.spatialreference.org\
                            /ref/esri/{}/proj4/".format(wkid),
                            "type": "proj4"}}
        if geomType == 'esriGeometryPolyline':
            for f in esriJson["features"]:
                feature = {'type' : 'Feature','properties' : {},'geometry' : {}}
                paths = f['geometry']['paths']
                #if any([len(p) > 1 for p in paths]):
                #    feature['geometry']['type'] = 'MultiLineString'
                #else:
                #    feature['geometry']['type'] = 'LineString'
                feature['geometry']['type'] = \
                    'MultiLineString' if any([len(p) > 1 for p in paths]) else 'LineString'
                feature['geometry']['coordinates'] = paths

        elif geomType == 'esriGeometryPoint':
            for f in esriJson["features"]:
                feature = {'type' : 'Feature','properties' : {},'geometry' : {}}
                feature['geometry']['type'] = 'Point'
                geom = [f['geometry']['x'],f['geometry']['y']]
                feature['geometry']['coordinates'] = geom

        elif geomType == 'esriGeometryMultipoint':
            for f in esriJson["features"]:
                feature = {'type' : 'Feature','properties' : {},'geometry' : {}}
                feature['geometry']['type'] = 'MultiPoint'
                geometries = f['geometries']
                points = []
                for geom in geometries:
                    points.extend(geom['points'])
                feature['geometry']['coordinates'] = points

        elif geomType == 'esriGeometryPolygon':
            for f in esriJson["features"]:
                feature = {'type' : 'Feature','properties' : {},'geometry' : {}}
                rings = f['geometry']['rings']
                #paths = f['geometry']['paths']
                #if any([len(r) > 1 for r in rings]):
                #    feature['geometry']['type'] = 'MultiPolygon'
                #else:
                #    feature['geometry']['type'] = 'Polygon'
                feature['geometry']['type'] = \
                    'MultiPolygon' if any([len(p) > 1 for p in rings]) else 'Polygon'
                feature['geometry']['coordinates'] = [rings]
                
        if 'attributes' in f:
                feature['properties'] = f['attributes']
                base_object['features'].append(feature)
        return base_object

    def addTempShapefile(self, layer, group=False, allGroup=None):
        '''Format to load shp in a zip files is /vsizip//vsicurl/https//.../folder.zip/file.shp
           Format to load shp without a zip is /vsicurl/https//.../file.shp
        '''
        url = layer["layerUrl"]

        if '.zip' in url.lower():
            shp_url = f'/vsizip//vsicurl/{url}'
        else:
            # Case no zip, direct call to shp
            shp_url = f'/vsicurl/{url}'
        vlayer = QgsVectorLayer(shp_url, layer['layerHebName'], 'ogr')
        if group:
            QgsProject.instance().addMapLayer(vlayer, False)
            if allGroup:
                allGroup.addLayer(vlayer)
        else:
            QgsProject.instance().addMapLayer(vlayer)
    def addWMSService(self, layer, group=False, allGroup=None):
        rlayer = QgsRasterLayer(layer["layerUrl"], layer["layerHebName"], layer["tempLayerType"])
        if group:
            QgsProject.instance().addMapLayer(rlayer, False)
            allGroup.addLayer(rlayer)
        else:
            QgsProject.instance().addMapLayer(rlayer)
            layerTree = self.iface.layerTreeCanvasBridge().rootGroup()
            layerTree.insertChildNode(0, QgsLayerTreeLayer(rlayer))

    def addGeoserverGeoJson(self, layer, group=False, allGroup=None):
        name = layer["layerHebName"]
        if "layerUrl2" in layer:
            url = layer["layerUrl2"]
            type = "ogr"
            vlayer = QgsVectorLayer(url,name,type)
            if group:
                QgsProject.instance().addMapLayer(vlayer, False)
                if allGroup:
                    allGroup.addLayer(vlayer)
            else:
                QgsProject.instance().addMapLayer(vlayer)
        else:
            self.mb.pushCritical('Error',"Unable to load WMS as vector layer")

    def addTempGeoJson(self, layer, group=False, allGroup=None):
        name = layer["layerHebName"]
        url = layer["layerUrl"]
        type = layer["tempLayerType"]
        vlayer = QgsVectorLayer(url,name,type)
        if group:
            QgsProject.instance().addMapLayer(vlayer, False)
            if allGroup:
                allGroup.addLayer(vlayer)
        else:
            QgsProject.instance().addMapLayer(vlayer)
    
    def loadTemps(self):
        """
        loading temporary layers to the project
        """
        sending_button = self.dlg.sender()

        # Disabled add to group button
        # group = self.dlg.addAsGroup.isChecked()
        group = False
        selectedLayers = []
        queryLayerID = []
        layers = []
        
        # Layer selection is inserted into a stack from earlier version when multi-selection was enabled 
        if self.dlg.tabWidget.currentWidget().findChildren(QTreeWidget)[0].selectedItems():
            selectedItem = self.dlg.tabWidget.currentWidget().findChildren(QTreeWidget)[0].selectedItems()[0]
            if selectedItem.parent() is None:
                self.mb.pushWarning('Warning',"Adding entire organizations is not supported")
            else:
                parent = selectedItem.parent()
                parent_text = parent.text(0)

                '''
                * When a layer is selected by a user, the layer has both it's name and it's parent branch
                * The layer is selected by comparing it's name from the ui and the layer name in the json
                * To Avoid mistakes the potential canidates are inserted into the layers stack
                * The stack is populated by potential layers by comparing the parent branch (organisation name) in the UI and the organisation in the json layer structure
                * The following code iterate through the 4 main categories and for each category iterates through
                    the suborganisations, if the suborganization is equal to the parent text in the ui then insert all the layers into the layers stack
                '''
                # Add potetntial layers based on parent text
                # Iterate each of the muncipliaties if muni is eqaul to parent label add to potential layers stack
                for org_name in (["municipalities", "govOrgs", "NGO", "ods"]):
                    for item in self.dataList[org_name].values():
                        if parent_text == item["hebName"]:
                            layers.extend(item["layers"])

                # Iterate through the potential layers after it was filtered using the organisation
                for layer in layers:
                    if selectedItem.text(0) == layer["layerHebName"]:
                        # if LayerID attribute exist it is preffered
                        if "layerID" in layer:
                            queryLayerID.append(layer['layerID'])
                        #Support for legacy code, (should be delted when refactoring)
                        else:
                            selectedLayers.append(layer)

                # Avoid duplicated layer ids
                queryLayerID = list(set(queryLayerID))

                #Support for legacy code, (should be delted when refactoring)
                seen = set()
                new_l = []
                for d in selectedLayers:
                    t = tuple(d.items())
                    if t not in seen:
                        seen.add(t)
                        new_l.append(d)
                selectedLayers = new_l

                # Disabled feature
                # if "userServices" in self.dataList:
                #     s = QgsSettings()
                #     userServices = self.dataList["userServices"]
                #     for i in self.dlg.userServicesTree.selectedItems():
                #         for layer in userServices:
                #             if i.text(0) == layer['layerHebName']:
                #                 selectedLayers.append(layer)

                # Query the server for data layers and add them to selectedLayers
                url = f"{self.prefix}/get_layer_data"
                layerID = queryLayerID[0]

                s = QgsSettings()
                userEmail = s.value('opendata_loader/email','')
                userKey = s.value('opendata_loader/key','')
                payload = {'layerID': layerID, 'email':userEmail, 'key':userKey}
                response =  self.postWithProxy(url, self.headers, payload)
                try:
                    response_json = json.loads(response.content().data())
                except:
                    response = 'error'

                # Check if response from server returns ok - layer infomration found
                if "answer" in response_json.keys():
                    
                    # Check if server returned a url
                    if response_json["answer"] == 1:
                        
                        # Check if the URL retrieved is valid if so add layer infromation to stack
                        if self.is_valid_url(response_json['layer_data']["layerUrl"]):
                            selectedLayers += [response_json['layer_data']]
                        else:
                            # layer URL  is not valid
                            self.mb.pushWarning('Warning', 'Layer URL not valid')
                    else:
                        # Layer url is missing
                        self.mb.pushWarning('Warning', 'URL could not be retrieved')
                    if "logins" in response_json.keys():
                        self.dlg.loginsCounter.setText('החודש הוספת {} שכבות'.format(response_json['logins']))
                else:
                    # Layer was not found
                    self.mb.pushWarning('Warning', 'Layer information not found')


                allGroup = QgsLayerTreeGroup("Group", False)
                if group is not False:
                    layerTree = self.iface.layerTreeCanvasBridge().rootGroup()
                    layerTree.insertChildNode(0, allGroup)
                if sending_button.objectName() == "addToMapV":
                    self.drawTempVLayer(selectedLayers, group, allGroup)
                elif sending_button.objectName() == "addToMapR":
                    self.drawTempRLayer(selectedLayers, group, allGroup)
            self.clearSelection()
            allGroup = None
    
    def drawTempVLayer(self, selectedLayers, group, allGroup):
        """
        Draw temporary vector layers
        """
        layersAdded = 0
        for layer in selectedLayers:
            layerUrl = layer["layerUrl"]
            layerCRS = layer["layerCRS"] if "layerCRS" in layer else None
            tempLayerType = layer["tempLayerType"]
            connectionType = layer["connectionType"]
            if connectionType == 'connections-arcgisfeatureserver':
                self.addTempArcgisFeature(layer,layerCRS, group, allGroup)
                layersAdded += 1
            elif connectionType == 'connections-arcgismapserver':
                self.mb.pushWarning('Warning', 'Rendering Arcgis MapServer layer as features might not work')
                self.addTempArcgisFeature(layer,layerCRS, group, allGroup)    
                layersAdded += 1
            elif connectionType == 'GeoJSON':
                self.addTempGeoJson(layer, group, allGroup)
                layersAdded += 1
            elif connectionType == 'shp':
                self.addTempShapefile(layer, group, allGroup)
                layersAdded += 1
            elif connectionType == 'esriJson':
                gj = self.esriJsonToGeoJson(layerUrl)
                layer['layerUrl'] = json.dumps(gj)
                self.addTempGeoJson(layer, group, allGroup)
                layersAdded += 1
            elif connectionType == 'wms':
                self.addGeoserverGeoJson(layer, group, allGroup)
                layersAdded += 1
            else:
                self.mb.pushCritical('Error',"Unrecognised layer type {}".format(tempLayerType))
                # raise TypeError("Unrecognised layer type {}".format(tempLayerType))
        if layersAdded > 0:
            self.mb.pushInfo('{} Layers added'.format(str(layersAdded)),"")
        else:
            self.mb.pushInfo('No layers to add',"")
        self.clearSelection()
    
    def drawTempRLayer(self, selectedLayers, group, allGroup):
        """
        Draw temporary raster layers
        """
        layersAdded = 0
        for layer in selectedLayers:
            layerUrl = layer["layerUrl"]
            layerCRS = layer["layerCRS"] if "layerCRS" in layer else None
            tempLayerType = layer["tempLayerType"]
            if tempLayerType == 'arcgisfeatureserver':
                layersAdded += 1
                self.mb.pushWarning('Warning', 'Rendering Arcgis FeatureServer layer as raster might not work')
                self.addTempArcgisMap(layer,layerCRS, group, allGroup)
            elif tempLayerType == 'arcgismapserver':
                layersAdded += 1
                self.addTempArcgisMap(layer,layerCRS, group, allGroup)
            elif tempLayerType == 'ogr':
                self.mb.pushWarning('Error',"Unable to render layer type {} as raster".format(layer["connectionType"]))
            elif tempLayerType == 'wms':
                layersAdded += 1
                self.addWMSService(layer,group, allGroup)
            else:
                self.mb.pushCritical('Error',"Unrecognised layer type {}".format(tempLayerType))
                break
        if layersAdded > 0:
            self.mb.pushInfo('{} Layers added'.format(str(layersAdded)),"")
        else:
            self.mb.pushInfo('No layers to add',"")
        self.clearSelection()

    def closeFormDialog(self):
        self.formdlg.reject()

    def resizeTabWidget(self):
        self.dlg.tabWidget.setGeometry(self.dlg.tabWidget.x(),self.dlg.tabWidget.y(),self.dlg.width()-60,self.dlg.tabWidget.height())
        self.dlg.govTree.setGeometry(self.dlg.govTree.x(),self.dlg.govTree.y(),self.dlg.tabWidget.width()-5,self.dlg.govTree.height())
        self.dlg.teaserGov.setGeometry(self.dlg.teaserGov.x(),self.dlg.teaserGov.y(),self.dlg.tabWidget.width()-5,self.dlg.teaserGov.height())
        self.dlg.muniTree.setGeometry(self.dlg.muniTree.x(),self.dlg.muniTree.y(),self.dlg.tabWidget.width()-5,self.dlg.muniTree.height())
        self.dlg.teaserMuni.setGeometry(self.dlg.teaserMuni.x(),self.dlg.teaserMuni.y(),self.dlg.tabWidget.width()-5,self.dlg.teaserMuni.height())
        self.dlg.orgTree.setGeometry(self.dlg.orgTree.x(),self.dlg.orgTree.y(),self.dlg.tabWidget.width()-5,self.dlg.orgTree.height())
        self.dlg.opendataTree.setGeometry(self.dlg.opendataTree.x(),self.dlg.opendataTree.y(),self.dlg.tabWidget.width()-5,self.dlg.opendataTree.height())
        
        # Disabled feature
        # self.dlg.teaserUserServices.setGeometry(self.dlg.teaserUserServices.x(),self.dlg.teaserUserServices.y(),self.dlg.tabWidget.width()-5,self.dlg.teaserUserServices.height())
        # self.dlg.userServicesTree.setGeometry(self.dlg.userServicesTree.x(),self.dlg.userServicesTree.y(),self.dlg.tabWidget.width()-5,self.dlg.userServicesTree.height())
    def reportNewService(self):
        # Open Github repository to add new services, upon click on button
        url = QUrl("https://github.com/KaplanOpenSource/qgis-open-data/issues/new")
        QDesktopServices.openUrl(url)
    def is_valid_url(self, url):
        # standard way to check if url is correct
        try:
            result = urlparse(url)
            return all([result.scheme, result.netloc])  # Check if scheme and netloc are present
        except ValueError:
            return False
    def run(self):
        """Run method that performs all the real work"""

        if self.first_start:
            self.first_start = False
            self.dlg = opendata_loaderDialog()

            # Load credentials to UI (if saved in settings)
            self.loadCredentials()
            
            # Load the layers based on the credntial, this enables reopening fast
            self.loadLayers()

        self.dlg.resized.connect(self.resizeTabWidget)
        self.dlg.registerButton.clicked.connect(self.openRegisterButton)
        self.dlg.submitCredntials.clicked.connect(self.loadLayers)
        self.dlg.addToMapV.clicked.connect(self.loadTemps)
        self.dlg.addToMapR.clicked.connect(self.loadTemps)

        # Disabled feature
        # self.dlg.addToMyServices.clicked.connect(self.addPluginServiceToUserServices)
        # self.dlg.addServiceButton.clicked.connect(self.openUserServicesForm)
        self.dlg.vLabel.setText(self.version)
        
        # Disabled feature
        # self.formdlg.AddUserLayer.clicked.connect(self.addUserService)
        # self.formdlg.addExisting.clicked.connect(self.addExisting)
        #self.formdlg.closeForm.clicked.connect(self.closeFormDialog)
        self.dlg.reportNewSourcebutton.clicked.connect(self.reportNewService)
        self.clearSelection()
        self.dlg.show()
        result = self.dlg.exec_()
        if result:
            self.clearSelection()
            
            

