# -*- coding: utf-8 -*-
"""
/***************************************************************************
 IrmisJsonLoader
                                 A QGIS plugin
 This plugin imports IRMIS Json files into QGIS
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-04-04
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Marco Lechner / Bundesamt für Strahlenschutz
        email                : mlechner@bfs.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QVariant, QDateTime, Qt, QMetaType
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

from qgis.core import Qgis, QgsProject, QgsField, QgsVectorLayer, QgsCoordinateReferenceSystem, QgsFeature, QgsPointXY, QgsGeometry, QgsMessageLog
# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .irmis_json_loader_dialog import IrmisJsonLoaderDialog
import os.path
import json
from datetime import datetime, UTC


class IrmisJsonLoader:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'IrmisJsonLoader_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&IRMIS Json Loader')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('IrmisJsonLoader', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/irmis_json_loader/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'IRMIS Json Loader'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&IRMIS Json Loader'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = IrmisJsonLoaderDialog()
        # set default values on every load of the GUI
        self.dlg.mQgsFileWidget.setFilePath(None)
        self.dlg.lineEditLayerName.setText(datetime.strftime(datetime.now(UTC), '%Y%m%d-%H%M_irmis'))

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            jsonfile_path = self.dlg.mQgsFileWidget.filePath() # Get path from QgsFileWidget
            # read file
            with open(jsonfile_path, 'r', encoding='utf-8-sig') as irmisfile:
                irmisdata = irmisfile.read()
            # parse file
            irmisobj = json.loads(irmisdata)
            # create layer
            vl_name = self.dlg.lineEditLayerName.text() if (self.dlg.lineEditLayerName.text() and len(self.dlg.lineEditLayerName.text()) > 0) else "irmis_json"
            vl = QgsVectorLayer("Point", vl_name, "memory")
            vl.setCrs(QgsCoordinateReferenceSystem('EPSG:4326'))

            pr = vl.dataProvider()
            # add fields default fields
            pr.addAttributes([QgsField("startTime", QMetaType.Type.QDateTime),
                    QgsField("time", QMetaType.Type.QDateTime),
                    QgsField("locationId", QMetaType.Type.Int),
                    QgsField("location", QMetaType.Type.QString),
                    QgsField("longitude", QMetaType.Type.Double),
                    QgsField("latitude", QMetaType.Type.Double),
                    QgsField("value", QMetaType.Type.Double),
                    QgsField("countryIsoCode",  QMetaType.Type.QString),
                    QgsField("surveyTypeId",  QMetaType.Type.Double),
                    QgsField("reportContextId",  QMetaType.Type.Double),
                    QgsField("measurementTypeId",  QMetaType.Type.Double),
                    QgsField("measurementUnitTypeId",  QMetaType.Type.Double),
                    QgsField("reportId",  QMetaType.Type.Int)])
                    
            vl.updateFields() # tell the vector layer to fetch changes from the provider
            featurelist = []
            for irmisfeature in irmisobj.get("data"):
                feature = QgsFeature()
                feature.setFields(vl.fields())
                if 'lon' in irmisfeature and 'lat' in irmisfeature:
                    feature.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(float(irmisfeature["lon"]),float(irmisfeature["lat"]))))
                elif 'longitude' in irmisfeature and 'latitude' in irmisfeature:
                    feature.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(float(irmisfeature["longitude"]),float(irmisfeature["latitude"]))))
                for key in feature.fields().names():
                    if key in irmisfeature:
                        if vl.fields().field(key).type() == QVariant.Int:
                            feature.setAttribute(key, int(irmisfeature[key]))
                        elif vl.fields().field(key).type() == QVariant.Double:
                            try:
                                feature.setAttribute(key, float(irmisfeature[key]))
                            except:
                                feature.setAttribute(key, None)
                                QgsMessageLog.logMessage("Float conversion failed for " + str(key) + " on " + str(irmisfeature["locationId"]), level=Qgis.MessageLevel.Info)
                        elif vl.fields().field(key).type() == QVariant.DateTime:
                            #2022-04-04 06:00:00 or 2024-10-18T06:00:00
                            #QDateTime.fromString(irmisfeature[key], "yyyy-MM-dd hh:mm:ss") or QDateTime.fromString(irmisfeature[key], Qt.ISODate)
                            try:
                                feature.setAttribute(key, QDateTime.fromString(irmisfeature[key], Qt.DateFormat.ISODate))
                            except:
                                try:
                                    feature.setAttribute(key, QDateTime.fromString(irmisfeature[key], "yyyy-MM-dd hh:mm:ss"))
                                except:
                                    QgsMessageLog.logMessage("Both datetime conversions failed for " + str(key) + " on " + str(irmisfeature["locationId"]), level=Qgis.MessageLevel.Warning)
                        else:
                            feature.setAttribute(key, str(irmisfeature[key]))
                if len(irmisfeature) == len(vl.fields()):
                    featurelist.append(feature)
                else:
                    self.iface.messageBar().pushMessage("Error", "irmisfeature len is " + str(len(irmisfeature)) + " on " + str(irmisfeature["locationId"] + " but table size is " + str(len(vl.fields()))), level=Qgis.MessageLevel.Critical)
            pr.addFeatures(featurelist)
            # update layer's extent when new features have been added
            # because change of extent in provider is not propagated to the layer
            vl.updateExtents()
            QgsProject.instance().addMapLayer(vl)
            self.iface.mapCanvas().refresh()