"""
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""

import os

from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterBand,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterFileDestination)
from qgis import processing

from .mesh import MeshFactory


class RasterToObjProcessingAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    BAND = 'BAND'
    NORMALIZE_XY = 'NORMALIZE_XY'
    NORMALIZE_Z = 'NORMALIZE_Z'
    EXPORT_RGB = 'EXPORT_RGB'
    FACE_TYPE = 'FACE_TYPE'
    OUTPUT = 'OUTPUT'

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return RasterToObjProcessingAlgorithm()

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'rastertoobj'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr('Raster to Obj')

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return ''

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def shortHelpString(self):
        return self.tr(
            """
            This processing algorithm creates a Wavefront .obj file from a raster layer. 

            <h3>Parameters</h3>
            <dl>
                <dt><b>Input raster layer</b></dt>
                <dd>The raster layer to export.</dd>
                <dt><b>Z values</b></dt>
                <dd>The values from this raster band will be exported the vertices Z values.</dd>
                <dt><b>Normalize XY values</b></dt>
                <dd>Sets the raster cell coordinates to 1x1 units and shifts the lower left to the origin (0,0). If not checked, the actual XY coordinates of the raster cell centroids will be exported.</dd>
                <dt><b>Normalize Z values</b></dt>
                <dd>Maps the Z values to a range between 0 and 1. If not set, the actual Z values from the data will be exported.</dd>
                <dt><b>Export RGB colors</b></dt>
                <dd>Export the RGB values based on the layer's symbology for every vertex. (Currently only works for Singleband pseudocolor renderers.)</dd>
                <dt><b>Face type</b></dt>
                <dd>Type of faces to be generated. NONE: Vertices only, TRI: Triangles only, QUAD: Quadrilaterals (squares) only, TRI+QUAD: Mixed types as needed.</dd>
                <dt><b>Output .obj file</b></dt>
                <dd>Filename of the .obj file to be generated.</dd>
            </dl>
            """
        )

    def initAlgorithm(self, config=None):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                self.tr('Input raster layer')
            )
        )

        self.addParameter(
            QgsProcessingParameterBand(
                self.BAND,
                self.tr("Z values"),
                parentLayerParameterName='INPUT'
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.NORMALIZE_XY,
                self.tr("Normalize XY values"),
                True
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.NORMALIZE_Z,
                self.tr("Normalize Z values"),
                True
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.EXPORT_RGB,
                self.tr("Export RGB colors"),
                True
            )
        )

        self.addParameter(
            QgsProcessingParameterEnum(
                self.FACE_TYPE,
                self.tr('Face type'), 
                ['NONE', 'TRI', 'QUAD', 'TRI+QUAD'],
                defaultValue=3
            )
        )

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT,
                self.tr('Output .obj file'),
                self.tr('OBJ files (*.obj)')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        # feedback.pushInfo(f'parameters: {str(parameters)}')

        raster_layer = self.parameterAsRasterLayer(parameters, self.INPUT, context)
        if raster_layer is None:
            raise QgsProcessingException(
                self.invalidRasterError(parameters, self.INPUT)
            )
        band = parameters['BAND']
        normalize_xy = parameters['NORMALIZE_XY']
        normalize_z = parameters['NORMALIZE_Z']
        export_rgb = parameters['EXPORT_RGB']
        face_type = parameters['FACE_TYPE']
        out_fn = self.parameterAsFileOutput(parameters, self.OUTPUT, context)
        feedback.pushInfo(out_fn)

        mesh_factory = MeshFactory()
        mesh_factory.from_raster(raster_layer, normalize_xy=normalize_xy, normalize_z=normalize_z, band_z=band, parse_rgb=export_rgb, face_type=face_type)

        feedback.pushInfo(str(mesh_factory))
        # feedback.pushInfo(str(mesh_factory.vertex_ids))

        mesh_factory.write_obj(out_fn, max_rgb=1)

        return {self.OUTPUT: out_fn}

    def icon(self):
        plugin_dir = os.path.dirname(__file__)
        icon_fn = os.path.join(plugin_dir, 'img', 'icon_raster_to_obj.svg')
        icon = QIcon(icon_fn)
        return icon