"""
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""

import os

from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterField,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterVectorLayer,
                       QgsProcessingParameterFileDestination)
from qgis import processing

from .mesh import MeshFactory


class PointLayerToObjProcessingAlgorithm(QgsProcessingAlgorithm):
    INPUT = 'INPUT'
    Z_SOURCE = 'Z_SOURCE'
    Z_FIELD = 'Z_FIELD'
    MOVE_TO_ORIGIN = 'MOVE_TO_ORIGIN'
    SCALE_XY = 'SCALE_XY'
    SCALE_Z = 'SCALE_Z'
    NORMALIZE_Z = 'NORMALIZE_Z'
    OUTPUT = 'OUTPUT'

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return PointLayerToObjProcessingAlgorithm()

    def name(self):
        return 'pointlayertoobj'

    def displayName(self):
        return self.tr('Point Layer to Obj')

    def group(self):
        return ''

    def groupId(self):
        return ''

    def shortHelpString(self):
        return self.tr(
            """
            This processing algorithm creates a Wavefront .obj file from a point layer. 

            <h3>Parameters</h3>
            <dl>
                <dt><b>Input point layer</b></dt>
                <dd>The point layer to export.</dd>
                <dt><b>Z source</b></dt>
                <dd>Set the source of the Z values, either the geometry's Z value (Z_FROM_POINTZ) or the value from an attribute field (Z_DEFINED_BY_FIELD).</dd>
                <dt><b>Z field</b></dt>
                <dd>If Z source is set to Z_DEFINED_BY_FIELD, pick the field that contains the Z values.</dd>
                <dt><b>Scale XY values</b></dt>
                <dd>Multiply the X and Y values by this factor before exporting (0.001 converts from meters to km)</dd>
                <dt><b>Output .obj file</b></dt>
                <dd>Filename of the .obj file to be generated.</dd>
            </dl>
            """
        )

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT,
                self.tr('Input point layer')
            )
        )
        
        self.addParameter(
            QgsProcessingParameterEnum(
                self.Z_SOURCE,
                self.tr('Z source'), 
                ['Z_FROM_POINTZ', 'Z_DEFINED_BY_FIELD'],
                defaultValue=1
            )
        )
        
        self.addParameter(
            QgsProcessingParameterField(
                self.Z_FIELD,
                self.tr("Z field"),
                parentLayerParameterName='INPUT'
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.MOVE_TO_ORIGIN,
                self.tr("Move data to origin"),
                False
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.SCALE_XY,
                self.tr("Scale XY values"),
                True,
                defaultValue=1
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.NORMALIZE_Z,
                self.tr("Normalize Z values"),
                True
            )
        )

        self.addParameter(
            QgsProcessingParameterFileDestination(
                self.OUTPUT,
                self.tr('Output .obj file'),
                self.tr('OBJ files (*.obj)')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        # feedback.pushInfo(f'parameters: {str(parameters)}')

        vector_layer = self.parameterAsVectorLayer(parameters, self.INPUT, context)
        if vector_layer is None:
            raise QgsProcessingException(
                self.invalidRasterError(parameters, self.INPUT)
            )
        feedback.pushInfo(str(vector_layer))
        z_source = parameters['Z_SOURCE']
        feedback.pushInfo(str(z_source))
        z_field = parameters['Z_FIELD']
        feedback.pushInfo(z_field)
        move_to_origin = parameters['MOVE_TO_ORIGIN']
        feedback.pushInfo(z_field)
        scale_xy = parameters['SCALE_XY']
        feedback.pushInfo(str(scale_xy))
        #normalize_z = parameters['NORMALIZE_Z']
        out_fn = self.parameterAsFileOutput(parameters, self.OUTPUT, context)
        feedback.pushInfo(out_fn)

        if z_source == 0:
            z_field = None
        feedback.pushInfo(z_field)

        mesh_factory = MeshFactory()
        import_result = mesh_factory.from_point_layer(vector_layer, z_field)
        feedback.pushInfo(str(import_result))

        feedback.pushInfo(str(mesh_factory))
        # feedback.pushInfo(str(mesh_factory.vertex_ids))

        mesh_factory.write_obj2(out_fn, move_to_origin=move_to_origin, scale_xy=scale_xy)

        return {self.OUTPUT: out_fn}

    def icon(self):
        plugin_dir = os.path.dirname(__file__)
        icon_fn = os.path.join(plugin_dir, 'img', 'icon_points_to_obj.svg')
        icon = QIcon(icon_fn)
        return icon