# -*- coding: utf-8 -*-
"""
/***************************************************************************
 FloodplainCalculator
                                 A QGIS plugin
 This plugin calculates inundation durations in floodplains.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-02-16
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Ute Susanne Kaden
        email                : ute-susanne.kaden@ufz.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.core import QgsProject, Qgis, QgsRasterLayer

from osgeo import gdal
from osgeo.gdalconst import *

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .inundation_calculator_dialog import FloodplainCalculatorDialog

import os.path
import numpy, sys

class FloodplainCalculator:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'FloodplainCalculator_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Floodplain Inundation Calculator')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('FloodplainCalculator', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/inundation_calculator/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'calculate floodplain inundation durations'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True
    
    # conversion tool: raster to array
    def raster_to_array(self, layer, band=1):
            # convert raster layer to a numpy array
            provider = layer.dataProvider()
            width, height = layer.width(), layer.height()
            block = provider.block(band, provider.extent(), width, height)

            # supported types
            types = {Qgis.Float64 : numpy.float64}

            try:
                T = types[block.dataType()]
            except KeyError:
                self.iface.messageBar().pushMessage("ERROR","raster type {} is not supported".format(block.dataType()),level=2,duration=10)
                raise NotImplementedError("raster type {} is not supported".format(block.dataType()))

            return numpy.ndarray((height, width), dtype=T, buffer=block.data().data()).round(2)
    
    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Floodplain Inundation Calculator'),
                action)
            self.iface.removeToolBarIcon(action)
    
    # output = tif file
    def select_output_file(self):
        filename, _filter = QFileDialog.getSaveFileName(
            self.dlg, "Select   output file ","", '*.tif')
        self.dlg.lineEdit_2.setText(filename)
        
    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = FloodplainCalculatorDialog()
            self.dlg.pushButton.clicked.connect(self.select_output_file)
                        
        # Get the currently loaded layers
        layers = QgsProject.instance().layerTreeRoot().children()
        # Clear the contents of the comboBox from previous runs
        self.dlg.comboBox.clear()
        self.dlg.comboBox_2.clear()
        # Fill the comboBox with names of all the loaded layers
        self.dlg.comboBox.addItems([layer.name() for layer in layers])
        self.dlg.comboBox_2.addItems([layer.name() for layer in layers])
        
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        
        # See if OK was pressed
        if result:                
            # use of layers listed in dropdown menus
            dem = layers[self.dlg.comboBox.currentIndex()].layer() # digital elevation model
            watlev = layers[self.dlg.comboBox_2.currentIndex()].layer() # daily water levels
            if self.dlg.lineEdit.text()=="":
                corr = 0
            else:
                corr = float(self.dlg.lineEdit.text())/100.00 # correction factor
            years = watlev.featureCount()/365 # daily water levels / 365 = considered years

            # waterlevel-dictionary: representation of the distribution of waterlevel data (csv) 
            # by dividing the entire range of values into a series of 1cm intervals  
            # and then count how many values fall into each interval
            waterlevel_dict = {}
            for wl in watlev.getFeatures():
                wl_r = round(float(wl.attributes()[0])+corr,2) # get all possible height steps
                if wl_r in waterlevel_dict:
                    waterlevel_dict[wl_r]+=1 # if height step is part of dict, dict is raised by 1
                else: 
                    waterlevel_dict[wl_r]=1 # else dict is set to 1 (height step is newly registered)
            
            # add up those lying below
            hmax = max(waterlevel_dict)
            hmin = min(waterlevel_dict)
            step = 0.01
            counter = 0
            h_dict = {} #new dictionary
            for i in range(int((hmax-hmin)/step)+1): # range of waterlevels 
                h = round((hmax-step*i),2) #count from top to bottom
                try:
                    counter = waterlevel_dict[h]+counter #if height step x is inundated, all height steps <x are inundated too
                except KeyError:
                    pass
                h_dict[h]=counter #inundation frequency for each height step 
            
            matrix = self.raster_to_array(dem)

            lx=matrix.shape[0]
            ly=matrix.shape[1]

            inundmatrix = [] #empty inundation matrix (subsequently filled)
            for x in range(lx):
                spalte = []
                for y in range(ly): # xy range of dem 
                    z = matrix[x][y]
                    if not(z==-9999): # only if raster is defined
                        if z > hmax:
                            f = 0 # never inundated
                        elif z <= hmin:
                            f = 365 # always inundated 
                        else:
                            f = h_dict[z]/years # f = sum of inundated days divided by number of years under consideration
                    else: # if raster is not defined
                        f = -9999 # inundation not defined, too
                    spalte.append(f)
                inundmatrix.append(spalte)
                
       
            # register all of the GDAL drivers
            gdal.AllRegister()

            # open the dem to get information 
            input_name = self.dlg.comboBox.currentText()
            inputlayer = QgsProject().instance().mapLayersByName(input_name)[0]
            inputlayer_path = inputlayer.source()
            inDs = gdal.Open(inputlayer_path)
            if inDs is None:
              print('Could not open dem')
              sys.exit(1)

            # read in the dem data to get information for output
            band1 = inDs.GetRasterBand(1)
            rows = inDs.RasterYSize
            cols = inDs.RasterXSize      
            
            # create the output image
            driver = inDs.GetDriver()
            
            # driver
            outDs = driver.Create(self.dlg.lineEdit_2.text(), cols, rows, 1, GDT_Int32)
            if outDs is None:
                print('Could not create')
                sys.exit(1)

            outBand = outDs.GetRasterBand(1)
            outData = numpy.matrix(inundmatrix)

            # write data
            outBand.WriteArray(outData, 0, 0)

            # flush data to disk, set the NoData value and calculate stats
            outBand.FlushCache()
            outBand.SetNoDataValue(-9999)

            # georeference the image and set the projection
            outDs.SetGeoTransform(inDs.GetGeoTransform())
            outDs.SetProjection(inDs.GetProjection())
            
            # finish process
            outBand = None
            outDs = None
            
            # load result if checkbox is checked
            if self.dlg.checkBox.isChecked():
                # get the path to the new created tif file
                resultname = self.dlg.lineEdit_2.text().split("/")[-1].split(".")[0]
                path_to_tif = self.dlg.lineEdit_2.text()
                # load the resultlayer
                resultlayer = QgsRasterLayer(path_to_tif, resultname)
                self.iface.messageBar().pushMessage("CONGRATULATIONS","The result is succesfully added as a layer to the current project.",level=3,duration=1000)
                if not resultlayer.isValid():
                    print("Layer failed to load!")
                QgsProject.instance().addMapLayer(resultlayer)
            else:
                print("Find result under specified path")
                self.iface.messageBar().pushMessage("CONGRATULATIONS","Find result under specified path",level=3,duration=1000)

        # end of calculation 
        pass
