import math

from qgis.PyQt.QtCore import QMetaType
from qgis.core import (
    QgsVectorLayer,
    QgsProject,
    QgsWkbTypes,
    QgsFeature,
    QgsField,
    QgsSpatialIndex,
    QgsDistanceArea,
    QgsCoordinateTransform,
    QgsFeatureRequest,
    QgsGeometry,
    QgsCoordinateReferenceSystem,
)


def build_intersection_memory_layer(layer_a, layer_b):
    """
    Gera uma camada temporária (memory) com a interseção entre duas camadas poligonais.
    Campos: area_m2, area_ha, perim_m

    - Suporta camadas em CRS diferentes (transforma geometrias de B para o CRS de A)
    - Mede área/perímetro SEMPRE em EPSG:4326 (WGS84), evitando NaN ao inverter camadas
    """
    if not _is_polygon_layer(layer_a) or not _is_polygon_layer(layer_b):
        return None

    if not layer_a.crs().isValid() or not layer_b.crs().isValid():
        return None

    # Saída no CRS da camada A
    crs_auth = layer_a.crs().authid()
    out_layer = QgsVectorLayer(f"Polygon?crs={crs_auth}", "intersecao_temp", "memory")
    if not out_layer.isValid():
        return None

    pr = out_layer.dataProvider()

    # Campos + updateFields
    pr.addAttributes([
        QgsField("area_m2", QMetaType.Type.Double),
        QgsField("area_ha", QMetaType.Type.Double),
        QgsField("perim_m", QMetaType.Type.Double),
    ])
    out_layer.updateFields()

    # Medidas geodésicas (Cookbook usa QgsDistanceArea + WGS84) :contentReference[oaicite:3]{index=3}
    d = QgsDistanceArea()
    d.setEllipsoid("WGS84")

    # Vamos medir SEMPRE em EPSG:4326 (Cookbook mostra EPSG:4326 e transform) :contentReference[oaicite:4]{index=4} :contentReference[oaicite:5]{index=5}
    crs_geo = QgsCoordinateReferenceSystem("EPSG:4326")
    ctx = QgsProject.instance()

    # Transformação B -> CRS de A (para fazer interseção corretamente)
    same_crs_ab = layer_a.crs() == layer_b.crs()
    xform_b_to_a = None
    if not same_crs_ab:
        xform_b_to_a = QgsCoordinateTransform(layer_b.crs(), layer_a.crs(), ctx)

    # Transformação CRS de A -> EPSG:4326 (para medir SEMPRE no mesmo CRS)
    need_a_to_geo = layer_a.crs() != crs_geo
    xform_a_to_geo = None
    if need_a_to_geo:
        xform_a_to_geo = QgsCoordinateTransform(layer_a.crs(), crs_geo, ctx)

    # Índice espacial (sempre no CRS de A)
    transformed_b = {}  # fid -> QgsGeometry no CRS de A, só se CRS diferente

    if same_crs_ab:
        index_b = QgsSpatialIndex(layer_b.getFeatures())
    else:
        index_b = QgsSpatialIndex()
        for fb in layer_b.getFeatures():
            gb = fb.geometry()
            if not gb or gb.isEmpty():
                continue

            gb_t = QgsGeometry(gb)
            try:
                gb_t.transform(xform_b_to_a)
            except Exception:
                continue

            transformed_b[fb.id()] = gb_t

            fb_idx = QgsFeature(fb)
            fb_idx.setGeometry(gb_t)
            index_b.addFeature(fb_idx)

    out_feats = []

    # Loop em A
    for fa in layer_a.getFeatures():
        ga = fa.geometry()
        if not ga or ga.isEmpty():
            continue

        candidate_ids = index_b.intersects(ga.boundingBox())
        if not candidate_ids:
            continue

        req = QgsFeatureRequest().setFilterFids(list(candidate_ids))

        for fb in layer_b.getFeatures(req):
            gb = fb.geometry()
            if not gb or gb.isEmpty():
                continue

            if same_crs_ab:
                gb_use = gb
            else:
                gb_use = transformed_b.get(fb.id())
                if gb_use is None:
                    gb_use = QgsGeometry(gb)
                    try:
                        gb_use.transform(xform_b_to_a)
                    except Exception:
                        continue

            if not ga.intersects(gb_use):
                continue

            inter = ga.intersection(gb_use)
            if not inter or inter.isEmpty():
                continue

            # ✅ Medir sempre em EPSG:4326
            inter_for_measure = QgsGeometry(inter)
            if need_a_to_geo:
                try:
                    inter_for_measure.transform(xform_a_to_geo)  # Cookbook mostra transform via QgsCoordinateTransform :contentReference[oaicite:6]{index=6}
                except Exception:
                    continue

            perim_m = float(d.measurePerimeter(inter_for_measure))
            area_m2 = float(d.measureArea(inter_for_measure))

            # Proteção: se por algum motivo vier NaN, ignora
            if math.isnan(perim_m) or math.isnan(area_m2):
                continue

            area_ha = area_m2 / 10000.0

            f_out = QgsFeature(out_layer.fields())
            f_out.setGeometry(inter)  # geometria de saída continua no CRS da camada A
            f_out.setAttributes([area_m2, area_ha, perim_m])
            out_feats.append(f_out)

    pr.addFeatures(out_feats)
    out_layer.updateExtents()
    return out_layer


def _is_polygon_layer(layer):
    return (
        isinstance(layer, QgsVectorLayer)
        and layer.isValid()
        and layer.geometryType() == QgsWkbTypes.GeometryType.PolygonGeometry
    )
