# -*- coding: utf-8 -*-
"""
/***************************************************************************
 InteractiveMap
                                 A QGIS plugin
 Bringing interactive maps to QGIS
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-08-06
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Kofi Agyekum
        email                : adukofi73@yahoo.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon, QColor
from qgis.PyQt.QtWidgets import QAction
from qgis.core import ( QgsVectorLayer, QgsProject, QgsSingleSymbolRenderer, 
                       QgsCategorizedSymbolRenderer, QgsGraduatedSymbolRenderer, 
                       QgsRuleBasedRenderer, QgsMarkerSymbol, QgsLineSymbol, QgsFillSymbol)
from qgis.PyQt.QtWidgets import QMessageBox, QFileDialog
from osgeo import ogr
from qgis.utils import plugins, iface

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .interactive_map_dialog import InteractiveMapDialog
from .feature_click_tool import FeatureClickTool
from .basemap_manager import BasemapManager
from .export_dialog import ExportDialog
from .exporter import Exporter
import os
import os.path
import random

class InteractiveMap:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'InteractiveMap_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Interactive Map')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        ##click tool
        self.click_tool = None

        ##basemap selector
        self.basemap_manager = BasemapManager()

        ##vectors
        self.loaded_vector_layers = []

        
    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('InteractiveMap', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/interactive_map/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Interactive Map'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Interactive Map'),
                action)
            self.iface.removeToolBarIcon(action)

    ##load basemap
    def load_selected_basemap(self):
        selected = self.dlg.basemapComboBox.currentText()
        try:
            self.basemap_manager.add_basemap(selected)
        except Exception as e:
            QMessageBox.warning(self.iface.mainWindow(), "Basemap Error", str(e))

    ##add custom map
    def add_custom_basemap(self):
        name = self.dlg.customNameInput.text().strip()
        url = self.dlg.customUrlInput.text().strip()

        ##add custom map
        #self.basemap_manager.add_custom_basemap(name, url)

        if not name or not url:
            QMessageBox.warning(self.iface.mainWindow(), "Missing Info", "Please enter both a name and URL.")
            return
        
        try:
            self.basemap_manager.add_custom_basemap(name, url)
            self.dlg.basemapComboBox.addItem(name)

            ##automatically loads the recently added custom map
            self.basemap_manager.add_basemap(name)

            QMessageBox.information(self.iface.mainWindow(), "Success", f"Basemap '{name}' added!")
            self.dlg.customNameInput.clear()
            self.dlg.customUrlInput.clear()
        except Exception as e:
            QMessageBox.warning(self.iface.mainWindow(), "Error", str(e))

    def load_vector_layer(self):
        file_paths, _ = QFileDialog.getOpenFileNames(
            self.iface.mainWindow(),
            "Select Vector Layer(s)",
            "",
            "Vector Files (*.shp *.geojson *.gpkg *.json *.kml *.kmz);;All Files (*)"
        )
        if not file_paths:
            return

        if not hasattr(self, 'loaded_vector_layers'):
            self.loaded_vector_layers = []

        def random_color():
            return QColor(random.randint(50, 200), random.randint(50, 200), random.randint(50, 200))

        def apply_simple_symbology(layer: QgsVectorLayer):
            """Only tweak when single-symbol; leave categorized/graduated/rule-based alone."""
            try:
                renderer = layer.renderer()
                geom = layer.geometryType()  

                if isinstance(renderer, QgsSingleSymbolRenderer):
                    sym = renderer.symbol()
                    if geom == 0:
                        sym.setSize(4)
                        sym.setColor(random_color())
                    elif geom == 1:
                        sym.setWidth(1.5)
                        sym.setColor(random_color())
                    elif geom == 2:
                        sym.setColor(random_color())
                        sym.setOpacity(0.6)
                    layer.triggerRepaint()
                else:
                    pass
            except Exception as e:
                QMessageBox.warning(self.iface.mainWindow(), "Styling warning",
                                    f"Could not style '{layer.name()}': {e}")

        def load_gpkg_layers(gpkg_path: str) -> int:
            """Open a GeoPackage and load all vector sublayers."""
            loaded = 0
            try:
                ds = ogr.Open(gpkg_path)
                if ds is None:
                    QMessageBox.warning(self.iface.mainWindow(), "Load Error",
                                        f"Could not open GeoPackage: {os.path.basename(gpkg_path)}")
                    return 0

                for i in range(ds.GetLayerCount()):
                    ogr_lyr = ds.GetLayerByIndex(i)
                    if ogr_lyr is None:
                        continue
                    name = ogr_lyr.GetName() or f"layer_{i}"
                    uri = f"{gpkg_path}|layername={name}"
                    v = QgsVectorLayer(uri, name, "ogr")
                    if not v.isValid():
                        print(f"Failed to load sublayer '{name}' from {gpkg_path}")
                        continue

                    QgsProject.instance().addMapLayer(v)
                    self.loaded_vector_layers.append(v)
                    apply_simple_symbology(v)
                    print(f"Loaded GPKG sublayer: {name}")
                    loaded += 1
            except Exception as e:
                QMessageBox.warning(self.iface.mainWindow(), "GeoPackage Error",
                                    f"Problem reading {os.path.basename(gpkg_path)}:\n{e}")
            return loaded

        total_loaded = 0
        for file_path in file_paths:
            ext = os.path.splitext(file_path)[1].lower()

            if ext == ".gpkg":
                loaded = load_gpkg_layers(file_path)
                total_loaded += loaded
                if loaded == 0:
                    print(f"ℹNo layers loaded from {file_path}")
                continue

            layer_name = os.path.basename(file_path)
            v = QgsVectorLayer(file_path, layer_name, "ogr")
            if not v.isValid():
                QMessageBox.warning(self.iface.mainWindow(), "Error", f"Failed to load: {layer_name}")
                continue

            QgsProject.instance().addMapLayer(v)
            self.loaded_vector_layers.append(v)
            apply_simple_symbology(v)
            print(f"Loaded: {v.name()}")
            total_loaded += 1

        if total_loaded:
            QMessageBox.information(self.iface.mainWindow(), "Layers Loaded",
                                    f"Successfully loaded {total_loaded} layer(s).")
        else:
            QMessageBox.warning(self.iface.mainWindow(), "No Layers Loaded",
                                "Nothing was loaded. If this was a GeoPackage, it may be empty or unreadable.")


    def export_to_web_map(self):
        try:
            dlg = MainDialog(self.iface)
            dlg.show()
            dlg.exec_()
        except Exception as e:
            QMessageBox.critical(self.iface.mainWindow(), "Error", f"Failed to launch qgis2web:\n{str(e)}")

    def open_export_dialog(self):
        dlg = ExportDialog()
        if dlg.exec_():
            settings = dlg.get_settings()

            if not hasattr(self, "loaded_vector_layers") or not self.loaded_vector_layers:
                self.iface.messageBar().pushWarning("Export Warning", "No layers loaded through the plugin.")
                return

            exporter = Exporter(self.iface, self.plugin_dir, self.loaded_vector_layers)
            exporter.export_to_web_map(settings)
            print("Exporting with layers:", [l.name() for l in self.loaded_vector_layers])

    def run(self):
        ##click tool
        if not self.click_tool:
            self.click_tool = FeatureClickTool(self.iface.mapCanvas())
        self.iface.mapCanvas().setMapTool(self.click_tool)

        #load a built-in basemap
        if not hasattr(self, 'dlg'):
            self.dlg = InteractiveMapDialog()

            ##populate dropdown with predefined basemaps
            self.dlg.basemapComboBox.clear()
            self.dlg.basemapComboBox.addItems(self.basemap_manager.basemaps.keys())

            ##connect basemap button
            self.dlg.loadBasemapButton.clicked.connect(self.load_selected_basemap)
            self.dlg.addCustomBasemapButton.clicked.connect(self.add_custom_basemap)

            ##load vector layer
            self.dlg.loadVectorLayerButton.clicked.connect(self.load_vector_layer)

            ##export to web map for qgis2web
            ##self.dlg.exportToWebButton.clicked.connect(self.export_to_web_map)

            ##export to my custom template
            self.dlg.exportToWebButton.clicked.connect(self.open_export_dialog)


        ##show dialog
        self.dlg.show()

       