# -*- coding: utf-8 -*-
"""
/***************************************************************************
 InclinometryCalc
                                 A QGIS plugin
 Plugin for inclinometry processing
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-01-26
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Michael Chernov
        email                : mihail.tchernov@yandex.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .inclinometry_calc_dialog import InclinometryCalcDialog
import os.path

from PyQt5.QtWidgets import *

from math import cos, sin, radians
from typing import List

import numpy as np

from qgis.core import QgsCoordinateTransform, QgsCoordinateTransformContext
from qgis.core import QgsCoordinateReferenceSystem

from .dialogs import show_folder_dialog
from .dialogs import show_file_dialog


def transform_coordinates(x, y, crs_src, crs_dest):
    if crs_src == crs_dest:
        return x, y
    transform = QgsCoordinateTransform(crs_src, crs_dest,
                                       QgsCoordinateTransformContext())
    transform_point = transform.transform(x, y)
    return [transform_point.x_local(), transform_point.y_local()]


def calc_meridian_correction(x_point: float, y_point: float) -> float:
    zone_number = int((x_point + 6) / 6)
    central_meridian = 6 * zone_number - 3
    return (x_point - central_meridian) * sin(radians(y_point))


def rotate(point, center, angle: float, is_rad=False):
    x_c, y_c, = center
    x, y = point
    if not is_rad:
        angle = radians(angle)
    x_transform = x_c + cos(angle) * (x - x_c) - sin(angle) * (y - y_c)
    y_transform = y_c + sin(angle) * (x - x_c) + cos(angle) * (y - y_c)
    return x_transform, y_transform


def get_middle_angle(alpha, beta):
    if abs(alpha - beta) <= 180:
        return (alpha + beta) / 2
    result = min(alpha, beta) - (360 - abs(alpha - beta)) / 2
    if result >= 0:
        return result
    return 360 + result


def calc_inclinometry_by_md(data: np.ndarray) -> np.ndarray:
    result = np.array([[0, 0, 0]])
    for i in range(1, data.shape[0]):
        inclination_0, azimuth_0 = data[i - 1, 1:]
        mid_az = radians(get_middle_angle(azimuth_0, data[i][2]))
        mid_incl = radians((inclination_0 + data[i][1]) / 2)
        dmd = data[i, 0] - data[i - 1, 0]
        dx = dmd * sin(mid_incl) * sin(mid_az)
        dy = dmd * sin(mid_incl) * cos(mid_az)
        dz = dmd * cos(mid_incl)
        delta = [dx, dy, dz]
        x, y, z = [result[-1, i] + dl for i, dl in enumerate(delta)]
        result = np.vstack((result, [x, y, z]))
    return result


def real_coordinates_inclinometry(data, well_head, azimuth_correction):
    result = np.zeros(shape=(data.shape[0], 4))
    result[0] = np.array(well_head[:2] + [0, well_head[2]])
    for i in range(1, len(data)):
        x, y, z = data[i]
        x_t, y_t = rotate((x, y), data[0, :2], azimuth_correction)
        x_t += well_head[0]
        y_t += well_head[1]
        z_alt = well_head[2] - z
        result[i] = np.array([x_t, y_t, z, z_alt])
        data[i][:2] = x_t, y_t
    return result


class InclinometryCalc:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'InclinometryCalc_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Inclinometry calculator')
        self.dlg = InclinometryCalcDialog()

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        self.file_path = ''
        self.file_columns: List[str] = []
        self.file_data = np.array([[]])
        self.well_head_coords = [0, 0, 0]
        self.input_coords_system = 0
        self.export_folder = '/'
        self.processing_type = 'DxDy'
        self.magnetic_azimuth_correction = 0.0

        self.dlg.bOpenFile.clicked.connect(self.open_input_file)
        self.dlg.cbProcessingType.currentTextChanged.connect(
            self.columns_enabling_control)
        self.dlg.eOpenFolder.clicked.connect(self.open_export_folder)
        self.dlg.bApply.clicked.connect(self.proc)

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('InclinometryCalc', message)

    def add_action(self, icon_path, text, callback, enabled_flag=True,
                   add_to_menu=True, add_to_toolbar=True, status_tip=None,
                   whats_this=None, parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/inclinometry_calc/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Inclinometry calculator'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Inclinometry calculator'),
                action)
            self.iface.removeToolBarIcon(action)

    @staticmethod
    def load_file(file_path):
        if len(file_path) == 0:
            return [], np.array([[]])

        with open(file_path, 'r') as f:
            file_columns = f.readline().rstrip().split('\t')

        file_data = np.loadtxt(file_path, skiprows=1, delimiter='\t')
        return file_columns, file_data

    def fill_column_selectors(self, items):
        if len(items) == 0:
            return
        self.dlg.cbDxColumn.addItems(items)
        self.dlg.cbDyColumn.addItems(items)
        self.dlg.cbVerticalDepthColumn.addItems(items)
        self.dlg.cbMDColumn.addItems(items)
        self.dlg.cbAzimuthColumn.addItems(items)
        self.dlg.cbInclinationColumn.addItems(items)

    def columns_enabling_control(self):
        index = self.dlg.cbProcessingType.currentIndex()
        if index == 0:
            self.dlg.cbDxColumn.setEnabled(True)
            self.dlg.cbDyColumn.setEnabled(True)
            self.dlg.cbVerticalDepthColumn.setEnabled(True)
            self.dlg.cbMDColumn.setEnabled(False)
            self.dlg.cbAzimuthColumn.setEnabled(False)
            self.dlg.cbInclinationColumn.setEnabled(False)
            self.processing_type = 'DxDy'
            self.magnetic_azimuth_correction = 0
            self.dlg.sbMagneticAzimuthCorrection.setEnabled(False)
        else:
            self.dlg.cbDxColumn.setEnabled(False)
            self.dlg.cbDyColumn.setEnabled(False)
            self.dlg.cbVerticalDepthColumn.setEnabled(False)
            self.dlg.cbMDColumn.setEnabled(True)
            self.dlg.cbAzimuthColumn.setEnabled(True)
            self.dlg.cbInclinationColumn.setEnabled(True)
            self.processing_type = 'MD'
            self.dlg.sbMagneticAzimuthCorrection.setEnabled(True)

    def get_form_data(self):
        x, y = self.dlg.sbXValue.value(), self.dlg.sbYValue.value()
        z = self.dlg.sbWellHeadAltitude.value()
        self.well_head_coords = [x, y, z]
        self.input_coords_system = self.dlg.mProjection.crs()
        self.magnetic_azimuth_correction = \
            self.dlg.sbMagneticAzimuthCorrection.value()

    def open_input_file(self):
        self.dlg.eFilePath.clear()
        self.file_path = ''
        path = show_file_dialog()
        if path is None:
            return

        self.file_path = path
        self.dlg.eFilePath.setText(path)

        self.file_columns, self.file_data = self.load_file(path)
        self.fill_column_selectors(items=self.file_columns)

    def get_pulkovo1942_coords(self):
        crs_src = self.input_coords_system
        crs_dest = QgsCoordinateReferenceSystem(4284)
        x, y, _ = self.well_head_coords
        return transform_coordinates(x, y, crs_src, crs_dest)

    def get_wgs84_coords(self):
        crs_dest = QgsCoordinateReferenceSystem(4326)
        crs_src = self.input_coords_system
        x, y, _ = self.well_head_coords
        return transform_coordinates(x, y, crs_src, crs_dest)

    def get_gk_zone_number(self) -> int:
        x, _ = self.get_wgs84_coords()
        return int((x + 6) / 6)

    def get_gk_projection(self):
        zone = str(self.get_gk_zone_number())
        zone_str = '0' * (2 - len(zone)) + str(zone)
        epsg_code = int('284' + zone_str)
        return QgsCoordinateReferenceSystem(epsg_code)

    def get_gk_coords(self):
        crs_src = self.input_coords_system
        crs_dest = self.get_gk_projection()
        x, y, _ = self.well_head_coords
        return transform_coordinates(x, y, crs_src, crs_dest)

    def open_export_folder(self):
        self.export_folder = show_folder_dialog()
        self.dlg.eExportFolder.setText(self.export_folder)

    def get_used_column_indexes(self):
        column_indexes = []
        if self.processing_type == 'DxDy':
            column_indexes.append(self.dlg.cbDxColumn.currentIndex())
            column_indexes.append(self.dlg.cbDyColumn.currentIndex())
            column_indexes.append(
                self.dlg.cbVerticalDepthColumn.currentIndex())
        else:
            column_indexes.append(self.dlg.cbMDColumn.currentIndex())
            column_indexes.append(self.dlg.cbInclinationColumn.currentIndex())
            column_indexes.append(self.dlg.cbAzimuthColumn.currentIndex())
        return column_indexes

    def get_export_data(self, incl_data, meridian_correction):
        result_array = self.file_data[:, self.get_used_column_indexes()]
        gk_zone_number = self.get_gk_zone_number()
        file_attribute_lines = f'Meridian correction =' \
                               f' {meridian_correction}\n'
        if self.processing_type == 'MD':
            header = ['MD', 'Incl', 'Geographic_azimuth',
                      f'x_GK{gk_zone_number}', f'y_GK{gk_zone_number}',
                      'Depth', 'Altitude']
            file_attribute_lines += f'Magnetic declination = ' \
                                    f'{self.magnetic_azimuth_correction}\n'
            result_array[:, 2] = result_array[:, 2] + self.magnetic_azimuth_correction
            result_array = np.column_stack((result_array, incl_data))
        else:
            header = ['dx', 'dy', 'Depth', f'x_GK{gk_zone_number}',
                      f'y_GK{gk_zone_number}', 'Altitude']
            result_array = np.column_stack((result_array,
                                            incl_data[:, [0, 1, 3]]))
        return header, file_attribute_lines, result_array

    def proc(self):
        self.get_form_data()

        column_indexes = self.get_used_column_indexes()

        data = self.file_data[:, column_indexes]
        if self.processing_type == 'MD':
            data = calc_inclinometry_by_md(data)

        x_sc42, y_sc42 = self.get_pulkovo1942_coords()
        meridian_correction = calc_meridian_correction(x_sc42, y_sc42)

        total_correction = meridian_correction + self.magnetic_azimuth_correction

        well_gk_coords = self.get_gk_coords()
        well_coords = [well_gk_coords[0], well_gk_coords[1],
                       self.well_head_coords[2]]
        transform_data = real_coordinates_inclinometry(data, well_coords,
                                                       total_correction)

        header, attribute_lines, export_data = \
            self.get_export_data(transform_data, meridian_correction)
        header = '\t'.join(header)

        export_path = os.path.join(self.export_folder, 'Incl.dat')
        with open(export_path, 'w') as f:
            f.write(attribute_lines)
            np.savetxt(f, export_data, '%f', '\t', header=header,
                       comments='')

    def run(self):
        """Run method that performs all the real work"""
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # self.get_form_data()
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass
