# Import basic libs
import json

# Import PyQt libs
from qgis.PyQt.QtCore import QObject, QUrl, pyqtSignal
from qgis.PyQt.QtNetwork import QNetworkReply, QNetworkRequest

from inaturalist_extractor.__about__ import __plugin_name__, __version__


class MaxObs(QObject):
    finished_dl = pyqtSignal()
    """Launch a request to fetch only one obs in the extent and get the total number of
    observation in th extent as a property."""

    def __init__(
        self,
        network_manager=None,
        extent=None,
        url=None,
        dlg=None,
    ):
        super().__init__()
        self.network_manager = network_manager
        self.extent = extent
        self.url = url
        self.dlg = dlg

        self.nb_obs = 0

        self.download()

    def download(self):
        # If the user only wants the verified value, the url is changed.
        if self.dlg.verifiable_checkbox.isChecked():
            verifiable = "verifiable=true&"
        else:
            verifiable = ""
        url = "{url}/?{verifiable}spam=false&swlng={xmin}&swlat={ymin}&nelng={xmax}&nelat={ymax}&locale=fr&per_page=1".format(  # noqa: E501
            url=self.url,
            xmin=self.extent.xMinimum(),
            ymin=self.extent.yMinimum(),
            xmax=self.extent.xMaximum(),
            ymax=self.extent.yMaximum(),
            verifiable=verifiable,
        )
        url = QUrl(url)
        request = QNetworkRequest(url)
        # Add an user-agent to the request so the website maintainer can check where
        # the requests are from.
        request.setRawHeader(
            b"User-Agent", bytes(__plugin_name__ + "/" + __version__, encoding="utf-8")
        )
        request.setHeader(QNetworkRequest.ContentTypeHeader, "application/json")
        reply = self.network_manager.get(request)
        reply.finished.connect(lambda: self.handle_finished(reply))

    def handle_finished(self, reply):
        # Check if the request raises an error.
        if reply.error() != QNetworkReply.NetworkError.NoError:
            print(f"code: {reply.error()} message: {reply.errorString()}")
            if reply.error() == 403:
                print("Service down")
        else:
            # Decode data fetch from the get request and create a dictionnary.
            data_request = reply.readAll().data().decode()
            res = json.loads(data_request)
            # Get the observation number in the extent based on a specific property
            self.nb_obs = res["total_results"]
            self.finished_dl.emit()
