# -*- coding: utf-8 -*-
"""
InaSAFE Disaster risk assessment tool developed by AusAid -
**metadata module.**

Contact : ole.moller.nielsen@gmail.com

.. note:: This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
"""

__author__ = 'marco@opengis.ch'
__revision__ = 'f16353426abc9c5fd8f65e2eb0e87e11c4159468'
__date__ = '27/05/2015'
__copyright__ = ('Copyright 2012, Australia Indonesia Facility for '
                 'Disaster Reduction')


from safe.metadata.provenance import ProvenanceStep


class Provenance(object):
    """
    Class to store a list of provenance steps.

    .. versionadded:: 3.2

    """

    def __init__(self):
        # private members
        self._steps = []

    def __str__(self):
        return str(self._steps)

    def __iter__(self):
        return iter(self._steps)

    @property
    def dict(self):
        """
        the python object for rendering json.

        It is called dict to be
        coherent with the other modules but it actually returns a list

        :return: the python object for rendering json
        :rtype: list
        """

        json_list = []
        for step in self.steps:
            json_list.append(step.dict)
        return json_list

    @property
    def xml(self):
        """
        the xml string representation.

        :return: the xml string
        :rtype: str
        """

        xml = '<inasafe_provenance>\n'
        for step in self.steps:
            xml += step.xml
        xml += '</inasafe_provenance>\n'
        return xml

    @property
    def steps(self):
        """
        the steps list.

        :return: the steps list
        :rtype: list
        """

        return self._steps

    @property
    def count(self):
        """
        the size of the list.

        :return: the size
        :rtype: int
        """
        return len(self._steps)

    @property
    def last(self):
        """
        the last step of the list.

        :return: the last step
        :rtype: ProvenanceStep
        """
        return self._steps[-1]

    def get(self, index):
        """
        the step at index position of the list.

        :return: the step at index
        :rtype: ProvenanceStep
        """
        return self._steps[index]

    def append_step(self, title, description, timestamp=None):
        """
        Append a new provenance step.

        :param title: the title of the ProvenanceStep
        :type title: str
        :param description: the description of the ProvenanceStep
        :type description: str
        :param timestamp: the time of the ProvenanceStep
        :type timestamp: datetime
        :return: the time of the ProvenanceStep
        :rtype: datetime
        """
        step = ProvenanceStep(title, description, timestamp)
        self._steps.append(step)
        return step.time
