# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ImageVectorisator
                                 A QGIS plugin
 Plugin for image vectorisation
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-03-06
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Euk0r
        email                : gekalets2014@yandex.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import time

from qgis.PyQt.QtWidgets import QWidget, QFileDialog, QMessageBox
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon, QImage
from qgis.PyQt.QtWidgets import QAction
from qgis.core import QgsProcessing, QgsRasterLayer, QgsRasterBlock, QgsRasterBandStats, QgsSingleBandGrayRenderer, \
    QgsContrastEnhancement, QgsProject, QgsVectorLayer
import numpy as np

# test
import os
from qgis import processing
from PIL import Image

from .ModulesManager import ModulesManager

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .image_vectorisator_dialog import ImageVectorisatorDialog
import os.path


class ImageVectorisator:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ImageVectorisator_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Image vectorisator')

        projectPath = os.path.dirname(os.path.abspath(__file__))

        self.modulesManager = ModulesManager(projectPath)

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        self.layers = None
        self.index_selected_layer = None
        self.index_selected_overlay = None
        self.index_selected_model = None
        self.save_filepath = None

        self.init_values()

    def init_values(self):
        self.index_selected_layer = -1
        self.index_selected_overlay = -1
        self.index_selected_model = 0
        self.save_filepath = ""

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ImageVectorisator', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = ':/plugins/image_vectorisator/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Vectorise image'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Image vectorisator'),
                action)
            self.iface.removeToolBarIcon(action)

    def selectOutputFile(self):
        filename = QFileDialog.getSaveFileName(self.dlg, "Select output file ", "", '*.shp')
        if filename:
            self.dlg.lineEdit_filepath.setText(filename[0])

    def comboBoxSLChanged(self):
        """Selected layer changed event"""
        self.index_selected_layer = self.dlg.comboBox_source_layer.currentIndex() - 1

    def comboBoxMChanged(self):
        """Selected model changed event"""
        self.index_selected_model = self.dlg.comboBox_model.currentIndex()

    def run(self):
        """Run method that performs all the real work"""
        # breakpoints
        if len(QgsProject.instance().mapLayers().values()) == 0:
            self.errorWindow("There is no layers!")
            return

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = ImageVectorisatorDialog()
            self.dlg.pushButton_cancel.clicked.connect(self.closeDialog)
            self.dlg.pushButton_start.clicked.connect(self.doneDialog)
            self.dlg.toolButton.clicked.connect(self.selectOutputFile)
            self.dlg.comboBox_source_layer.activated.connect(self.comboBoxSLChanged)
            self.dlg.comboBox_model.activated.connect(self.comboBoxMChanged)
        else:
            self.beforeExit()

        # Adding menu items
        self.layers = [layer for layer in QgsProject.instance().mapLayers().values()]
        names = [layer.name() for layer in self.layers]
        names.insert(0, "")
        self.dlg.comboBox_source_layer.addItems(names)
        self.dlg.comboBox_overlay.addItems(names)
        module_names = self.modulesManager.module_names
        self.dlg.comboBox_model.addItems(module_names)

        # init indexes
        if self.index_selected_layer >= 0:
            self.dlg.comboBox_source_layer.setCurrentIndex(self.index_selected_layer + 1)
        if self.index_selected_overlay >= 0:
            self.dlg.comboBox_overlay.setCurrentIndex(self.index_selected_overlay + 1)
        if self.index_selected_model >= 0:
            self.dlg.comboBox_model.setCurrentIndex(self.index_selected_model)

        # show the dialog
        self.dlg.show()

        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            print('complete')

        self.beforeExit()

    def raster2ndarray(self, lyr):
        '''
            input: lyr(QgsMapLayerType.QgsRasterLayer)
            output: _(ndarray)
        '''
        provider = lyr.dataProvider()
        # How many bands
        blocks = []
        for c in range(lyr.bandCount()):
            blocks.append(provider.block((c + 1), lyr.extent(), lyr.width(), lyr.height()))
        values = []
        for i in range(lyr.width()):
            tmps = []
            for j in range(lyr.height()):
                tmps.append([blocks[k].value(i, j) for k in range(lyr.bandCount())])
            values.append(tmps)
        return np.array(values)

    def errorWindow(self, info):
        msg = QMessageBox()
        msg.setIcon(QMessageBox.Critical)
        msg.setText("Error")
        msg.setInformativeText(info)
        msg.setWindowTitle("Error")
        msg.exec_()

    def addGrayLayerFromNArray(self, selected_layer, array, layer_name):
        params = {
            'EXTENT': selected_layer.dataProvider().extent(),
            'TARGET_CRS': selected_layer.crs(),
            'PIXEL_SIZE': int(selected_layer.rasterUnitsPerPixelX()),
            'NUMBER': 1,
            'OUTPUT_TYPE': 0,
            # Qgis.DataType.Byte, https://docs.qgis.org/3.22/en/docs/user_manual/processing_algs/qgis/rastercreation.html
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        r = processing.run('qgis:createconstantrasterlayer', params)['OUTPUT']
        rlayer = QgsRasterLayer(r, layer_name, 'gdal')
        provider = rlayer.dataProvider()

        w = rlayer.width()
        h = rlayer.height()

        # dataType = Qgis.DataType.Byte
        dataType = provider.dataType(1)
        block = QgsRasterBlock(dataType, w, h)
        for i in range(0, w):
            for j in range(0, h):
                block.setValue(i, j, int(array[i][j]))

        provider.setEditable(True)
        provider.writeBlock(block, band=1)
        provider.setEditable(False)
        provider.reload()

        # this is needed for the min and max value to refresh in the layer panel
        stats = provider.bandStatistics(1, QgsRasterBandStats.All, selected_layer.dataProvider().extent())
        min = stats.minimumValue
        max = stats.maximumValue

        renderer = rlayer.renderer()
        myType = renderer.dataType(1)
        GrayRenderer = QgsSingleBandGrayRenderer(provider, 1)
        contrastEnhancement = QgsContrastEnhancement.StretchToMinimumMaximum
        myEnhancement = QgsContrastEnhancement()
        myEnhancement.setContrastEnhancementAlgorithm(contrastEnhancement, True)
        myEnhancement.setMinimumValue(min)
        myEnhancement.setMaximumValue(max)
        rlayer.setRenderer(GrayRenderer)
        rlayer.renderer().setContrastEnhancement(myEnhancement)

        QgsProject.instance().addMapLayer(rlayer)

    def beforeExit(self):
        self.dlg.comboBox_source_layer.clear()
        self.dlg.comboBox_overlay.clear()
        self.dlg.comboBox_model.clear()

    def closeDialog(self):
        self.dlg.done(0)

    def doneDialog(self):
        if self.index_selected_layer < 0:
            self.errorWindow("Did not selected layer")
            return

        if not type(self.layers[self.index_selected_layer]) == QgsRasterLayer:
            self.errorWindow("The layer is not Raster")
            return

        # if self.index_selected_model < 0:
        #    self.errorWindow("Did not selected model")
        #    self.run()
        #    return

        if self.dlg.lineEdit_filepath.text():
            shp_save_path = self.dlg.lineEdit_filepath.text()
        else:
            self.errorWindow("Shape file save path is empty")
            return

        # START PROCESSING
        try:
            save_all = self.dlg.checkBox_save_all.isChecked()
            open_at_the_end = self.dlg.checkBox_open_after.isChecked()
            selected_layer = self.layers[self.index_selected_layer]

            # convert raster to nparray
            #data = self.raster2ndarray(selected_layer)

            selected_layer_path = selected_layer.dataProvider().dataSourceUri()

            # binarize image
            imgpath_segmented = self.modulesManager.processModule(self.index_selected_model, selected_layer_path,
                                                                  self.dlg.progressBar)

            # open binarized image, convert to gray and add to project as temporary
            if save_all:
                img = Image.open(imgpath_segmented)
                img_array = np.asarray(img)
                img_gray_array = np.dot(img_array[..., :3], [0.299, 0.587, 0.114])
                w = selected_layer.width()
                h = selected_layer.height()
                self.addGrayLayerFromNArray(selected_layer, img_gray_array, "Binarized image")

            # vectorise image
            img_name = imgpath_segmented.split('/')[-1]
            self.modulesManager.vectorise(img_name, imgpath_segmented, self.dlg.comboBox_model.currentIndex())

            # save temporal shp with offset
            temp_dir = self.modulesManager.get_working_dir(img_name)
            temp_shp_name = shp_save_path.split('/')[-1]
            temp_shp_save_path = os.path.join(temp_dir, temp_shp_name)
            offset = [selected_layer.dataProvider().extent().xMinimum(),
                      selected_layer.dataProvider().extent().yMaximum()]
            shp_file = self.modulesManager.save_vector(img_name, temp_shp_save_path, self.dlg.comboBox_model.currentIndex(), offset)

            # reproject gained shpfile and save
            parameter = {'INPUT': shp_file,
                         'EXTENT': selected_layer.dataProvider().extent(),
                         'TARGET_CRS': selected_layer.crs(),
                         'OUTPUT': shp_save_path}

            processing.run('qgis:reprojectlayer', parameter)

            # delete temp objects
            '''
                There is a problem with deleting additional temp shp files
                so for now deleting will ignore errors
            '''
            if not save_all:
                time.sleep(1)
                self.modulesManager.clean_up(img_name)
            else:
                os.startfile(temp_dir)

            # open shp layer
            if open_at_the_end:
                vlayer = QgsVectorLayer(shp_save_path, temp_shp_name, "ogr")
                if not vlayer.isValid():
                    print("Layer failed to load!")
                else:
                    QgsProject.instance().addMapLayer(vlayer)

            self.save_filepath = ""
            #self.dlg.done(1)

        except Exception as e:
            self.errorWindow(e)
