"""/***************************************************************************
 IbgeDataDownloader
                                 A QGIS plugin
 This plugin downloads data from IBGE
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-11-17
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Vinicius Etchebeur Medeiros Dória
        email                : vinicius_etchebeur@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from __future__ import annotations

import http
import http.client
import os
import socket
import tarfile
import urllib
import zipfile
from pathlib import Path
from typing import TYPE_CHECKING

from qgis.core import Qgis, QgsApplication, QgsProject, QgsVectorLayer
from qgis.PyQt.QtCore import (
    QCoreApplication,
    QItemSelection,
    QItemSelectionModel,
    QModelIndex,
    QSettings,
    Qt,
    QTranslator,
)
from qgis.PyQt.QtGui import QBrush, QColor, QIcon, QStandardItem, QStandardItemModel
from qgis.PyQt.QtWidgets import (
    QAbstractItemView,
    QAction,
    QComboBox,
    QDialogButtonBox,
    QFileDialog,
    QHeaderView,
    QListWidget,
    QProgressBar,
    QPushButton,
    QToolButton,
    QWidget,
)

from ibgedatadownloader.__about__ import DIR_PLUGIN_ROOT
from ibgedatadownloader.HelpDialog import HelpDialog
from ibgedatadownloader.ibgeDataDownloader_dialog import IbgeDataDownloaderDialog
from ibgedatadownloader.MyHTMLParser import MyHTMLParser
from ibgedatadownloader.MyProgressDialog import MyProgressDialog
from ibgedatadownloader.WorkerDownloadManager import WorkerDownloadManager
from ibgedatadownloader.WorkerSearchManager import WorkerSearchManager

if TYPE_CHECKING:
    from collections.abc import Callable

    from qgis.gui import QgisInterface


class IbgeDataDownloader:
    """Manages the main functionality of the IBGE Data Downloader plugin.

    This class handles the plugin's lifecycle, including initialization of the GUI,
    user interactions, and communication with background worker tasks for searching
    and downloading data from IBGE's FTP servers.
    """

    def __init__(self, iface: QgisInterface) -> None:
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgisInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = Path(DIR_PLUGIN_ROOT / "i18n" / f"ibgeDataDownloader_{locale}.qm")

        if locale_path.exists() and locale_path.is_file():
            self.translator = QTranslator()
            self.translator.load(str(locale_path))
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&Download data from IBGE")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.firstStart = None

        # Plugin icon
        self.pluginIcon = QIcon(str(DIR_PLUGIN_ROOT / "icon.png"))

        # Access to QGIS status bar and progress button
        self.statusBar = self.iface.statusBarIface()
        for i in self.statusBar.children():
            if isinstance(i, QToolButton):
                for j in i.children():
                    if isinstance(j, QProgressBar):
                        self.qgisProgressButton = i
                        break

        # Avoid headers and maxlines limit error
        http.client._MAXHEADERS = 999999999999999999
        http.client._MAXLINE = 999999999999999999

        # Saving references
        self.msg_bar = self.iface.messageBar()
        self.task_manager = QgsApplication.taskManager()
        self.settings = QSettings()
        self.html_parser = MyHTMLParser()
        self.geobase_url = "https://geoftp.ibge.gov.br/"
        self.statbase_url = "https://ftp.ibge.gov.br/"
        self.items_expanded = []
        self.selected_products_url = []
        self.selected_search = ""
        self.dir_output = ""
        self.item_last_check_state = None
        self.items_highlighted = []

    # noinspection PyMethodMayBeStatic
    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("IbgeDataDownloader", message)

    def add_action(
        self,
        icon_path: str,
        text: str,
        callback: Callable,
        enabled_flag: bool = True,
        add_to_menu: bool = True,
        add_to_toolbar: bool = True,
        status_tip: str | None = None,
        whats_this: str | None = None,
        parent: QWidget | None = None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self) -> None:
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = str(DIR_PLUGIN_ROOT / "icon.png")
        self.add_action(
            icon_path,
            text=self.tr("IBGE Data Downloader"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.firstStart = True

    def unload(self) -> None:
        """Remove the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&Download data from IBGE"), action)
            self.iface.removeToolBarIcon(action)

    def progress_dialog(self, value: int, text: str) -> tuple[MyProgressDialog, QProgressBar]:
        """Create and configure a custom progress dialog.

        :param value: The initial value for the progress bar.
        :type value: int

        :param text: The label text to display in the dialog.
        :type text: str

        :returns: A tuple containing the configured progress dialog and progress bar.
        :rtype: tuple(MyProgressDialog, QProgressBar)
        """
        dialog = MyProgressDialog()
        dialog.setWindowTitle(self.tr("Processing. Please, wait..."))
        dialog.setLabelText(text)
        dialog.setWindowIcon(self.pluginIcon)
        progress_bar = QProgressBar(dialog)
        progress_bar.setTextVisible(True)
        progress_bar.setValue(value)
        dialog.setBar(progress_bar)
        dialog.setMinimumWidth(300)
        dialog.setModal(True)
        dialog.setWindowFlag(Qt.WindowCloseButtonHint, False)
        dialog.setAutoClose(False)
        dialog.setAutoReset(False)
        dialog.canceled.connect(self.canceled_progress_dialog)
        for i in dialog.children():
            if isinstance(i, QPushButton):
                self.dlg_bar_cancel_button = i
                break
        return dialog, progress_bar

    def set_maximum_progress_value(self, max_n: float) -> None:
        """Set the maximum value for the progress bar.

        :param maxN: The maximum value.
        :type maxN: float
        """
        # self.progressBar.reset()
        self.progress_value = 0
        self.progress_bar.setRange(self.progress_value, int(max_n))
        self.progress_bar.setValue(self.progress_value)
        # print(int(maxN))

    def canceled_progress_dialog(self) -> None:
        """Handle the cancellation of the progress dialog."""
        self.dlg_bar_cancel_button.setEnabled(False)
        self.dlg_bar.setLabelText(self.tr("{}\nCanceling...").format(self.dlg_bar.labelText().split("\n")[0]))
        self.dlg_bar.show()
        if not self.thread_task.isCanceled():
            self.thread_task.cancel()

    def set_progress_value(self, val: int) -> None:
        """Set the current value of the progress bar.

        :param val: The progress value.
        :type val: int
        """
        self.progress_value = int(val)
        self.progress_bar.setValue(self.progress_value)

    def set_progress_text(self, txt: str) -> None:
        """Set the label text of the progress dialog.

        If the text indicates file extraction, the progress bar is set to indeterminate.

        :param txt: The text to display.
        :type txt: str
        """
        self.dlg_bar.setLabelText(txt)
        if self.tr("Extracting files") in txt:
            self.progress_bar.setRange(0, 0)
            self.progress_bar.setValue(0)

    def dlg_dir_output(self, checked: bool) -> None:
        """Open a dialog to select the output directory.

        :param checked: The state of the button that triggered the slot.
        :type checked: bool
        """
        self.dir_output = QFileDialog.getExistingDirectory(QFileDialog(), self.tr("Output directory"), "")
        self.dlg.lineEdit_Saida.setText(self.dir_output)

        if self.dir_output != "":
            self.check_ok_button()
        else:
            self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(False)

    def thread_result(self, result: list) -> None:
        """Store the result from a background worker thread.

        :param result: The result list emitted by the worker task.
        :type result: list
        """
        self.plugin_result = result
        # self.endingProcess()

    def end_process(self) -> None:
        """Finalize the process after a background task completes.

        This method closes the progress dialog, displays messages to the user,
        and handles post-processing tasks like adding layers or highlighting search results.
        """
        # Close progress dialog
        self.dlg_bar.setClose(True)
        self.dlg_bar.close()

        # Define message title and deal with remaining data, if necessary
        if self.plugin_result[4] == "download":
            # If the method callback came from WorkerDownloadManager
            if self.tr("canceled") in self.plugin_result[0]:
                msg_type = self.tr("Warning")
                # Delete remaining data
                file_name = os.path.basename(self.plugin_result[3])
                if os.path.isfile(os.path.join(self.dir_output, file_name)):
                    os.remove(os.path.join(self.dir_output, file_name))
            elif self.plugin_result[1] == Qgis.Critical:
                msg_type = self.tr("Error")
            else:
                msg_type = self.tr("Success")
                # Add layer (if possible)
                if self.dlg.checkBox_AddLayer.isChecked():
                    for i in self.plugin_result[3]:
                        file_name = os.path.basename(i[1])
                        file = os.path.join(self.dir_output, file_name)
                        files = None
                        if file_name.endswith(".zip"):
                            files = zipfile.ZipFile(file).namelist()
                        elif file_name.endswith(".tar"):
                            files = tarfile.TarFile(file).getnames()
                        if files:
                            for j in files:
                                if j.endswith(".shp"):
                                    layer = QgsVectorLayer(os.path.join(self.dir_output, j), j, "ogr")
                                    # Extent enlarged of 1/25
                                    extent = layer.extent()
                                    x_min = extent.xMinimum()
                                    y_min = extent.yMinimum()
                                    x_max = extent.xMaximum()
                                    y_max = extent.yMaximum()
                                    diagonal = ((x_max - x_min) ** 2 + (y_max - y_min) ** 2) ** 0.5
                                    buffer = 0.04 * diagonal
                                    extent = extent.buffered(buffer)
                                    # Add layer
                                    QgsProject.instance().addMapLayer(layer)
                                    self.iface.mapCanvas().setExtent(extent)
        else:
            # If the method callback came from WorkerSearchManager
            if self.tr("canceled") in self.plugin_result[0]:
                msg_type = self.tr("Warning")
            elif self.plugin_result[1] == Qgis.Critical:
                msg_type = self.tr("Error")
            else:
                msg_type = self.tr("Success")

            if len(self.plugin_result[3]) > 0:
                self.dlg_bar, self.progress_bar = self.progress_dialog(
                    0,
                    self.tr("Adding products to Products Tree. This may take several minutes..."),
                )
                self.progress_bar.setRange(0, 0)
                self.dlg_bar.show()
                for i in self.plugin_result[3]:
                    dirs = i[0].split("/")[2:]
                    # print(i, dirs)
                    if "geo" in dirs[0]:
                        self.dlg.tabWidget.setCurrentIndex(0)
                        tree_view = self.dlg.treeView_Geo
                        model = tree_view.model()
                    else:
                        self.dlg.tabWidget.setCurrentIndex(1)
                        tree_view = self.dlg.treeView_Stat
                        model = tree_view.model()
                    for d in dirs:
                        if d != "":
                            items = model.findItems(d, Qt.MatchExactly | Qt.MatchRecursive)
                            for n, item in enumerate(items):
                                model_index = model.indexFromItem(item)
                                item_url = self.get_item_url(model_index)
                                # print(itemUrl, i[0])
                                if item_url in i[0]:
                                    # Highlight item found
                                    text = self.dlg.lineEdit_SearchWord.text()
                                    if text.lower() in model_index.data().lower():
                                        # Set backgroud color to highlight item
                                        item.setBackground(QBrush(QColor(255, 255, 100)))
                                        self.items_highlighted.append(item)
                                    if os.path.splitext(item.text())[1] in ("", ".br"):
                                        # If item is expandable
                                        tree_view.expand(model_index)
                                        # Scroll to first item found
                                        if n == 0:
                                            tree_view.scrollTo(
                                                model_index,
                                                QAbstractItemView.PositionAtTop,
                                            )

                self.dlg_bar.setClose(True)
                self.dlg_bar.close()

        self.msg_bar.clearWidgets()
        if self.plugin_result[2] != []:
            self.msg_bar.pushMessage(
                msg_type,
                self.plugin_result[0],
                "\n\n".join(self.plugin_result[2]),
                self.plugin_result[1],
                duration=0,
            )
        else:
            self.msg_bar.pushMessage(msg_type, self.plugin_result[0], self.plugin_result[1], duration=20)

    def populateComboListBox(
        self,
        objeto: QComboBox | QListWidget,
        lista: list,
        coluna: str = "",
        inicial: str = "",
    ) -> None:
        """Populate a QComboBox or QListWidget with items from a list.

        :param objeto: The widget to populate (QComboBox or QListWidget).
        :type objeto: QtWidgets.QComboBox or QtWidgets.QListWidget

        :param lista: The list of items to add.
        :type lista: list

        :param coluna: The key to access a specific column in a list of dicts/tuples.

        :param inicial: An optional initial item to add.
        """
        objeto.clear()
        if inicial != "":
            objeto.addItem(inicial)
        for elemento in lista:
            e = elemento if coluna == "" else elemento[0] + " - " + elemento[coluna]
            item = str(e)
            objeto.addItem(item)

    def get_current_objects(self, clear: bool = False) -> tuple[str, QAbstractItemView, QItemSelectionModel]:
        """Get the relevant objects based on the currently active tab.

        :param clear: If True, clears the selection model of the inactive tab.
        :type clear: bool

        :returns: A tuple containing the base URL, the active QTreeView,
            and its selection model.
        :rtype: tuple(str, QtWidgets.QTreeView, QtCore.QItemSelectionModel)
        """
        if self.dlg.tabWidget.currentIndex() == 0:
            if clear:
                self.stat_selection_Model.clear()
            base_url = self.geobase_url
            tree_view = self.dlg.treeView_Geo
            selection_model = self.geo_selection_model
        else:
            if clear:
                self.geo_selection_model.clear()
            base_url = self.statbase_url
            tree_view = self.dlg.treeView_Stat
            selection_model = self.stat_selection_Model

        return base_url, tree_view, selection_model

    def get_item_url(self, model_index: QModelIndex) -> str:
        """Construct the full URL for a given item in the tree view.

        :param modelIndex: The QModelIndex of the item.
        :type modelIndex: QtCore.QModelIndex

        :returns: The constructed URL for the item.
        :rtype: str
        """
        base_url, _, _ = self.get_current_objects()

        if model_index.column() == 0:
            # Gets all parents and the item to create the URL
            parents = (
                [model_index.data()]
                if [f"/{model_index.data()}" if os.path.splitext(model_index.data())[1] == "" else model_index.data()][
                    0
                ]
                not in base_url
                else []
            )
            parent = model_index.parent()
            # print(modelIndex.parent(), modelIndex.parent().data())
            while (
                parent.data() is not None
                and [f"/{parent.data()}" if os.path.splitext(parent.data())[1] == "" else parent.data()][0]
                not in base_url
            ):
                parents.insert(0, parent.data())
                parent = parent.parent()
            product_url = f"{base_url}{'/'.join(parents)}"
        else:
            product_url = ""

        return product_url

    def clear_highlighted_items(self) -> None:
        """Remove the background highlight from all previously highlighted items."""
        while self.items_highlighted:
            for i in self.items_highlighted:
                i.setBackground(QBrush())
                self.items_highlighted.remove(i)

    def tree_view_pressed(self, model_index: QModelIndex) -> None:
        """Store the check state of an item before it is clicked.

        This is used to detect changes in the check state in the `treeViewClicked` slot.

        :param modelIndex: The QModelIndex of the pressed item.
        :type modelIndex: QtCore.QModelIndex
        """
        item = model_index.model().itemFromIndex(model_index)
        self.item_last_check_state = item.checkState() if item.isCheckable() else None

        # Set background of highlighted items to default
        self.clear_highlighted_items()

    def tree_view_clicked(self, model_index: QModelIndex) -> None:
        """Handle a click event on an item in the tree view.

        Manages the list of selected products, updates UI elements accordingly,
        and checks if the download (OK) button should be enabled.

        :param modelIndex: The QModelIndex of the clicked item.
        :type modelIndex: QtCore.QModelIndex
        """
        _, tree_view, selection_model = self.get_current_objects()

        # Set background of highlighted items to default
        self.clear_highlighted_items()

        if model_index.column() == 0:
            # Get url of clicked item
            product_url = self.get_item_url(model_index)

            # Add or remove from products variable
            model = model_index.model()
            item = model.itemFromIndex(model_index)
            # print(item.checkState())
            if self.item_last_check_state and self.item_last_check_state != item.checkState():
                selection_model.clear()
            if item.checkState() == Qt.CheckState.Checked:
                self.selected_products_url.append([model_index, product_url, tree_view])
                qtd_products = len(self.selected_products_url)
            else:
                for p in self.selected_products_url:
                    if p[1] == product_url:
                        self.selected_products_url.remove(p)
                qtd_products = len(self.selected_products_url)
            self.dlg.label_ProductsSelected.setText("{p} {t}".format(p=qtd_products, t=self.tr("Product(s) selected")))

            # Check if the unzip option can be enabled
            if self.selected_products_url:
                self.dlg.checkBox_AddLayer.setEnabled(True)
                for p in self.selected_products_url:
                    if any(p[1].endswith(ext) for ext in (".zip", ".tar")):
                        self.dlg.checkBox_Unzip.setEnabled(True)
                        break
                    self.dlg.checkBox_Unzip.setEnabled(False)
            else:
                self.dlg.checkBox_Unzip.setEnabled(False)
                self.dlg.checkBox_AddLayer.setEnabled(False)

            # Check if OK button can be enabled
            self.check_ok_button()

    def tree_view_expanded(self, model_index: QModelIndex) -> None:
        """Handle the expansion of an item in the tree view.

        Fetches the directory listing for the expanded item from the IBGE FTP server
        and populates the tree with its children (subdirectories and files).

        :param model_index: The QModelIndex of the expanded item.
        :type model_index: QtCore.QModelIndex
        """
        _, tree_view, _ = self.get_current_objects()

        if model_index not in self.items_expanded:
            # Deletes first empty child
            model = model_index.model()
            item = model.itemFromIndex(model_index)
            child = item.child(0)
            if child and child.text() == "":
                item.removeRow(0)

            # Adds item's children
            # print('/'.join(parents))
            url = self.get_item_url(model_index)
            # print(url)
            self.html_parser.resetParent()
            self.html_parser.resetChildren()
            self.html_parser.resetChild()
            # Set timeout for requests
            socket.setdefaulttimeout(15)
            try:
                with urllib.request.urlopen(url) as response:
                    self.html_parser.feed(response.read().decode("utf-8", errors="ignore"))
                # Set timeout for requests to default
                socket.setdefaulttimeout(None)
            except TimeoutError as e:
                self.msg_bar.pushMessage(
                    self.tr("Warning"),
                    self.tr("The expand operation of an item fails due to a server timeout."),
                    e,
                    Qgis.Warning,
                    duration=0,
                )
            # Set timeout for requests to default
            socket.setdefaulttimeout(None)
            children = self.html_parser.getChildren()

            # Add children to the tree
            # print(children)
            if children:
                for child in children:
                    if child[0] != "https://www.ibge.gov.br/":
                        print("adicionando {} ao item {}".format(child[0].replace("/", ""), model_index.data()))
                        child[0] = child[0].replace("/", "")
                        self.add_tree_view_parent_child_item(tree_view, model_index, child)

            # Add the item to expanded list
            self.items_expanded.append(model_index)

    def search_exact_state_changed(self, state: Qt.CheckState) -> None:
        """Enable or disable the match score spinbox based on the 'Contains' checkbox state.

        :param state: The new state of the checkbox.
        :type state: QtCore.Qt.CheckState
        """
        if state == Qt.Checked:
            self.dlg.label_Match.setEnabled(False)
            self.dlg.doubleSpinBox_MatchValue.setEnabled(False)
        else:
            self.dlg.label_Match.setEnabled(True)
            self.dlg.doubleSpinBox_MatchValue.setEnabled(True)

    def add_tree_view_parent_child_item(self, tree_view: QAbstractItemView, parent, child: list | None = None) -> None:
        """Add a parent or child item to the QTreeView.

        :param treeView: The QTreeView to which the item will be added.
        :type treeView: QtWidgets.QTreeView

        :param parent: The parent item or its text.

        :param child: A list containing the child item's data.
        """
        model = tree_view.model()
        if not model:
            model = QStandardItemModel(0, 3)
            model.setHorizontalHeaderLabels(
                [
                    self.tr("Products Tree"),
                    self.tr("File size"),
                    self.tr("Last modified"),
                ],
            )
            tree_view.setModel(model)

        # Creates standard empty item
        empty_item = QStandardItem("")

        if not child:
            parent_item = QStandardItem(parent)
            parent_item.appendRow([empty_item, empty_item, empty_item])
            model.appendRow([parent_item, empty_item, empty_item])
        else:
            if isinstance(parent, QModelIndex):
                # print(u'é QModelIndex', child)
                parent_item = model.itemFromIndex(parent)
                child_item = QStandardItem(child[0])
            if "." not in child_item.text():
                if not child_item.hasChildren():
                    child_item.appendRow(empty_item)
            else:
                child_item.setCheckable(True)
            # print(child)
            try:
                child_item_size = QStandardItem(child[2])
            except IndexError:
                child_item_size = empty_item
            try:
                child_item_sate = QStandardItem(child[1])
            except IndexError:
                child_item_sate = empty_item
            parent_item.appendRow([child_item, child_item_size, child_item_sate])
            # print(childItem.text(), childItem.row(), childItem.column())
            # print(childItemSize.text(), childItemSize.row(), childItemSize.column())
            # print(childItemDate.text(), childItemDate.row(), childItemDate.column())

    def uncheck_all(self) -> None:
        """Uncheck all selected products and reset related UI options.

        This is the slot for the 'Uncheck All' button.
        """
        # Iterate trough products list
        for p in self.selected_products_url:
            model = p[2].model()
            model_index = p[0]
            item = model.itemFromIndex(model_index)
            item.setCheckState(False)

        # Set label of selected products
        self.dlg.label_ProductsSelected.setText(self.tr("0 Product(s) selected"))
        # Clear products urls list
        self.selected_products_url = []
        # Clear selections
        self.geo_selection_model.clear()
        self.stat_selection_Model.clear()
        self.selected_search = ""
        # Uncheck options
        self.dlg.checkBox_SearchSelectedOnly.setChecked(False)
        self.dlg.checkBox_SearchSelectedOnly.setEnabled(False)
        self.dlg.checkBox_Unzip.setChecked(False)
        self.dlg.checkBox_Unzip.setEnabled(False)
        self.dlg.checkBox_AddLayer.setChecked(False)
        self.dlg.checkBox_AddLayer.setEnabled(False)
        # Disable OK button
        self.check_ok_button()

    def search_word_text_changed(self, text: str) -> None:
        """Handle text changes in the search input field.

        Standardizes the input text (replaces spaces with underscores) and enables
        the search button only if the text is valid.

        :param text: The current text in the search field.
        """
        # Starndardize text
        text = text.replace(" ", "_")
        self.dlg.lineEdit_SearchWord.setText(text)

        # Check if text respect standard structure
        if (
            not any(
                c in text.lower() or c.replace(".", "") == text.lower()
                for c in (
                    " ",
                    ",",
                    ".",
                    ";",
                    "?",
                    ".zip",
                    ".tar",
                    ".shp",
                    ".xls",
                    ".ods",
                    ".pdf",
                )
            )
            and text != ""
        ):
            self.dlg.pushButton_Search.setEnabled(True)
        else:
            self.dlg.pushButton_Search.setEnabled(False)

    def tree_view_selection_changed(self, selected: QItemSelection, deselected: QItemSelection) -> None:
        """Handle changes in the tree view's selection.

        Updates the `selectedSearch` attribute with the URL of the selected directory,
        which can be used to limit the scope of a search.

        :param selected: The newly selected items.
        :type selected: QtCore.QItemSelection
        :param deselected: The previously selected items.
        """
        _, tree_view, selection_model = self.get_current_objects(True)

        if selected.indexes():
            item = tree_view.model().itemFromIndex(selected.indexes()[0])
            if item.column() > 0:
                selection_model.clear()
                self.selected_search = ""
            else:
                # Define selected search
                product_url = self.get_item_url(selected.indexes()[0])
                self.selected_search = product_url if os.path.splitext(product_url)[1] == "" else ""
            # Enable or disable selected item search option
            self.radio_button_search_geo_toggled(self.dlg.radioButton_SearchGeo.isChecked())

    def radio_button_search_geo_toggled(self, checked: bool) -> None:
        """Enable or disable the 'Search in selected only' checkbox.

        The checkbox is enabled only if a directory is selected in the tree view
        that corresponds to the currently selected FTP server (Geo or Stat).
        :param checked: True if the 'Geo' radio button is checked, False otherwise.
        """
        if self.selected_search != "":
            if checked:
                if self.selected_search.startswith("https://geo"):
                    self.dlg.checkBox_SearchSelectedOnly.setEnabled(True)
                else:
                    self.dlg.checkBox_SearchSelectedOnly.setEnabled(False)
            elif self.selected_search.startswith("https://ftp"):
                self.dlg.checkBox_SearchSelectedOnly.setEnabled(True)
            else:
                self.dlg.checkBox_SearchSelectedOnly.setEnabled(False)
        else:
            self.dlg.checkBox_SearchSelectedOnly.setEnabled(False)

    def search_clicked(self) -> None:
        """Initiate a background search for products.

        This is the slot for the 'Search' button.
        """
        # Set background of highlighted items to default
        self.clear_highlighted_items()

        # Search params
        root = self.geobase_url if self.dlg.radioButton_SearchGeo.isChecked() else self.statbase_url
        text = self.dlg.lineEdit_SearchWord.text()
        match_contains = self.dlg.checkBox_SearchContains.isChecked()
        match_score = self.dlg.doubleSpinBox_MatchValue.value()

        # Collapsing all item in the three
        tree_view = self.dlg.treeView_Geo if "geo" in root else self.dlg.treeView_Stat
        tree_view.collapseAll()

        # Preparing product search
        self.dlg_bar, self.progress_bar = self.progress_dialog(0, self.tr("Searching data..."))
        self.dlg_bar.show()
        self.msg_bar.pushMessage(
            self.tr("Processing"),
            self.tr(f"Searching products with '{text}' word.\nThis may take several minutes..."),
            Qgis.Info,
            duration=0,
        )
        # Adjusting button text
        self.dlg_bar_cancel_button.setText(self.tr("Interrupt"))

        # Instantiate the background worker and connects slots to signals
        task_desc = "{} {}.\n{}...".format(self.tr("Searching"), text, self.tr("The search may take several minutes"))
        self.thread_task = WorkerSearchManager(
            self.iface,
            task_desc,
            (
                self.selected_search
                if self.dlg.checkBox_SearchSelectedOnly.isEnabled()
                and self.dlg.checkBox_SearchSelectedOnly.isChecked()
                else root
            ),
            text,
            match_contains,
            match_score,
        )
        self.thread_task.begun.connect(lambda: self.set_progress_text(task_desc))
        self.thread_task.progressChanged.connect(self.set_progress_value)
        self.thread_task.bar_max.connect(self.set_maximum_progress_value)
        self.thread_task.text_progress.connect(self.set_progress_text)
        self.thread_task.process_result.connect(self.thread_result)
        self.thread_task.taskCompleted.connect(self.end_process)
        self.thread_task.taskTerminated.connect(self.end_process)
        self.task_manager.addTask(self.thread_task)
        # Hide QGIS native progress button
        self.qgisProgressButton.hide()

    def button_help_clicked(self) -> None:
        """Show the help dialog."""
        self.help_dialog.show()

    def config_dialogs(self) -> None:
        """Configure the main plugin dialog and the help dialog.

        This method sets up initial UI elements, connects signals to slots,
        and initializes models for the tree views.
        """
        # Set window icon
        self.help_dialog.setWindowIcon(self.pluginIcon)
        self.dlg.setWindowIcon(self.pluginIcon)

        # Add top parent to the tree
        self.add_tree_view_parent_child_item(
            self.dlg.treeView_Geo,
            os.path.basename(os.path.normpath(self.geobase_url)),
        )
        self.add_tree_view_parent_child_item(
            self.dlg.treeView_Stat,
            os.path.basename(os.path.normpath(self.statbase_url)),
        )

        # Adjusting headers size mode
        self.dlg.treeView_Geo.header().setSectionResizeMode(QHeaderView.ResizeToContents)
        self.dlg.treeView_Stat.header().setSectionResizeMode(QHeaderView.ResizeToContents)

        # Set selection mode
        self.dlg.treeView_Geo.setSelectionMode(QAbstractItemView.SingleSelection)
        self.dlg.treeView_Stat.setSelectionMode(QAbstractItemView.SingleSelection)

        # Keep references to selection models
        self.geo_selection_model = self.dlg.treeView_Geo.selectionModel()
        self.stat_selection_Model = self.dlg.treeView_Stat.selectionModel()

        # Connect signals to slots
        self.dlg.pushButton_Dir.clicked.connect(self.dlg_dir_output)
        self.dlg.treeView_Geo.clicked.connect(self.tree_view_clicked)
        self.dlg.treeView_Geo.expanded.connect(self.tree_view_expanded)
        self.dlg.treeView_Geo.pressed.connect(self.tree_view_pressed)
        self.dlg.treeView_Stat.clicked.connect(self.tree_view_clicked)
        self.dlg.treeView_Stat.expanded.connect(self.tree_view_expanded)
        self.dlg.treeView_Stat.pressed.connect(self.tree_view_pressed)
        self.dlg.pushButton_UncheckAll.clicked.connect(self.uncheck_all)
        self.dlg.checkBox_SearchContains.stateChanged.connect(self.search_exact_state_changed)
        self.dlg.lineEdit_SearchWord.textChanged.connect(self.search_word_text_changed)
        self.dlg.pushButton_Search.clicked.connect(self.search_clicked)
        self.dlg.radioButton_SearchGeo.toggled.connect(self.radio_button_search_geo_toggled)
        self.geo_selection_model.selectionChanged.connect(self.tree_view_selection_changed)
        self.stat_selection_Model.selectionChanged.connect(self.tree_view_selection_changed)
        self.dlg.button_box.button(QDialogButtonBox.Help).clicked.connect(self.button_help_clicked)

        # Populate tree for product selection
        # url = '{}organizacao_do_territorio/'.format(self.geobaseUrl)

        # Disable OK button
        self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(False)
        self.dlg.button_box.button(QDialogButtonBox.Cancel).setText(self.tr("Cancel"))
        self.dlg.button_box.button(QDialogButtonBox.Help).setText(self.tr("Help"))
        # Disable options
        self.dlg.checkBox_Unzip.setEnabled(False)
        self.dlg.checkBox_AddLayer.setEnabled(False)
        self.dlg.pushButton_Search.setEnabled(False)

    def check_ok_button(self) -> None:
        """Enable or disable the OK (download) button based on the current state."""
        if self.selected_products_url:
            for p in self.selected_products_url:
                if os.path.splitext(p[1])[1] != "" and self.dir_output != "":
                    # Enable OK button
                    self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(True)
                else:
                    # Disable OK button
                    self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(False)
        else:
            # Disable OK button
            self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(False)

    def execute(self) -> None:
        """Initiate the background download process for the selected products.

        This method is called when the user confirms the download by clicking the 'OK'
        button in the main dialog. It sets up a progress dialog, instantiates the
        :class:`~ibgedatadownloader.WorkerDownloadManager.WorkerDownloadManager` to handle
        the download and extraction in a separate thread, connects the necessary signals
        and slots for UI updates, and adds the task to the QGIS task manager.
        """
        # Preparing product download
        self.dlg_bar, self.progress_bar = self.progress_dialog(0, self.tr("Downloading data..."))
        self.dlg_bar.show()
        self.msg_bar.pushMessage(
            self.tr("Processing"),
            self.tr("Working on selected data..."),
            Qgis.Info,
            duration=0,
        )

        # Instantiate the background worker and connects slots to signals
        task_desc = self.tr("Processing selected data.")
        self.thread_task = WorkerDownloadManager(
            self.iface,
            task_desc,
            self.selected_products_url,
            self.dir_output,
            [self.dlg.checkBox_Unzip.isEnabled(), self.dlg.checkBox_Unzip.isChecked()],
        )
        self.thread_task.begun.connect(lambda: self.set_progress_text(task_desc))
        self.thread_task.progressChanged.connect(self.set_progress_value)
        self.thread_task.barMax.connect(self.set_maximum_progress_value)
        self.thread_task.textProgress.connect(self.set_progress_text)
        self.thread_task.processResult.connect(self.thread_result)
        self.thread_task.taskCompleted.connect(self.end_process)
        self.thread_task.taskTerminated.connect(self.end_process)
        self.task_manager.addTask(self.thread_task)
        # Hide QGIS native progress button
        self.qgisProgressButton.hide()

    def run(self) -> None:
        """Run method - plugin callback."""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.firstStart:
            self.firstStart = False
            self.dlg = IbgeDataDownloaderDialog()
            self.help_dialog = HelpDialog(self.dlg)
            self.config_dialogs()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            self.execute()
