# -*- coding: utf-8 -*-
"""
/***************************************************************************
 IBGECartographyIndexMap
                                 A QGIS plugin
 This plugin displays in QGIS the index layers from IBGE Cartography Department.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-05-17
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Marcel Rotunno (IBGE)
        email                : marcelgaucho@yahoo.com.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QThread, pyqtSignal
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialogs
from .download_ibge_cartography_index_map_dialog import DownloadIBGECartographyIndexMapDialog

# Import from QGIS library
from qgis.core import QgsProject, Qgis, QgsVectorLayer

# Import the Index Layer Map structure
from .groups_layers import groups_idxlayers

# Import other libraries
import requests
import zipfile
import os.path


class DownloadThread(QThread):
    _signal = pyqtSignal(int)
    def __init__(self, filename_download):
        super(DownloadThread, self).__init__()
        self.filename_download = filename_download

    def __del__(self):
        self.wait()

    def run(self):
        # Download URL
        url = (u'https://geoftp.ibge.gov.br/'
               u'cartas_e_mapas/mapa_indice_digital/mapa_indice_digital_5ed_2021/'
               u'mapa_indice_digital_5ed_2021.zip') 
        
        # Get response from the server
        res = requests.get(url, stream=True)
        
        try:
            res.raise_for_status()
        except Exception:
            # If any error in http request occurs, it returns -1 
            self._signal.emit(-1)
            
        # Download file
        total_length = int(res.headers.get('content-length'))
        
        # Progress bar starts at 0
        dl = 0
        self._signal.emit(dl)
        
        # Do the work while update the progress bar
        try:
            with open(self.filename_download, "wb") as f:
                for chunk in res.iter_content(chunk_size=4096):
                    f.write(chunk)
                    dl = dl + len(chunk)
                    done = int(dl / total_length * 100)
                    self._signal.emit(done)
                    
        except FileNotFoundError:
            # Returns -2 if file not found
            self._signal.emit(-2)
           
           
class IBGECartographyIndexMap:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # index map file directory
        self.file_dir = os.path.join(self.plugin_dir, 'gpkg_indexmap_file')
        if not os.path.exists(self.file_dir):
            os.mkdir(self.file_dir)
        
        # Index Map Geopackage path
        self.filename = os.path.join(self.file_dir, 'mapa_indice_digital_v2021.gpkg')
        
        # Variable to store if download has already started (avoid 2 download threads)
        self.download_started = False
        
        # List of added layers
        self.layers_list = []
        self.invalid_layers = [] 
        
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'IBGECartographyIndexMap_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&IBGE Cartography Index Map')
        

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        self.first_start_download = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('IBGECartographyIndexMap', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/ibge_cartography_index_map/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Display IBGE Cartography Index Layers'),
            callback=self.run,
            parent=self.iface.mainWindow())
        
        # will be set False in run()
        self.first_start = True
        self.first_start_download = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&IBGE Cartography Index Map'),
                action)
            self.iface.removeToolBarIcon(action)
            
    def start_download(self):
        self.thread = DownloadThread(os.path.join(self.file_dir, 'mapa_indice_digital_v2021.zip'))
        self.thread._signal.connect(self.signal_accept)
        self.thread.start()
        
        
    def signal_accept(self, msg):
        # Error in http request
        if msg == -1:
            self.dlg_download.reject()
            self.iface.messageBar().pushMessage(
                "Error", "Error while retrieving URL",
                level=Qgis.Critical, duration=3)
            return    

        # Error in filename    
        if msg == -2:
            self.dlg_download.reject()
            self.iface.messageBar().pushMessage(
                "Error", "Invalid filename",
                level=Qgis.Critical, duration=3)
            return
        
        self.dlg_download.progressBar.setValue(msg)
        if self.dlg_download.progressBar.value() == 100:
            self.unzip_indexmap()
            self.dlg_download.progressBar.setValue(0)
            self.dlg_download.accept()
            self.run()
            
    
    def add_groups_layers(self, group_dict, group):
        """Adds the groups and layers presented in a dictionary to a parent group."""
    
        # Iterate over the dictionary and use recursion
        for key, value in group_dict.items():
            # If the group value is a dict, there are subgroups.
            # In this case, it adds the group and repeats the process
            # to add its subgroups with recursion
            if isinstance(value, dict):
                temp_group = group.addGroup(key)
                self.add_groups_layers(value, temp_group)
            # If the group value is of list type, it adds the group
            # and the layers of this group
            else:
                temp_group = group.addGroup(key)
                for layer in value:
                    gpkg_layer = self.filename + "|layername=" + layer
                    vlayer = QgsVectorLayer(gpkg_layer, layer, "ogr")
                    self.layers_list.append(vlayer) # It's necessary to store the layers in a list, I think it is just
                                                    # to keep reference to them
                    temp_group.addLayer(vlayer)
                    

    def unzip_indexmap(self):
        filename = os.path.join(self.file_dir, 'mapa_indice_digital_v2021.zip')
        
        with zipfile.ZipFile(filename, 'r') as zip_file:
            zip_file.extractall(self.file_dir)                    
                    
    def run(self):
        """Run method that performs all the real work"""
        
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start_download == True:
            self.first_start_download = False
            self.dlg_download = DownloadIBGECartographyIndexMapDialog()
            
        # Treatment if gpkg file doesn't exists 
        if not os.path.exists(self.filename):
            # show the dialog
            self.dlg_download.show()
            # Show dialog
            self.dlg_download.show()
                
            # Start download
            if not self.download_started:
                self.download_started = True
                self.start_download()
                                   
            return
            
        # Add main group of IBGE Index Map Layers
        root = QgsProject.instance().layerTreeRoot()
        main_group = root.addGroup('IBGE Index Map Layers') 

        # Clear list of added layers
        self.layers_list.clear()
        self.invalid_layers.clear()
        
        # Add Index Map Structure and Layers
        try:
            self.add_groups_layers(groups_idxlayers, main_group)
        except Exception as e:
            self.iface.messageBar().pushMessage(
                "Error", str(e),
                level=Qgis.Critical, duration=3)
        else:
            self.iface.messageBar().pushMessage(
                "Success", "Index Map Layers Loaded",
                level=Qgis.Success, duration=3)
        
        # Warn about invalid layers
        invalid_layers_names = [layer.name() for layer in self.layers_list if not layer.isValid()] 
        if invalid_layers_names:
            self.iface.messageBar().pushMessage(
                "Warning", "The following layers are invalid: {}".format(', '.join(invalid_layers_names)),
                level=Qgis.Warning, duration=3)
            
