# -*- coding: utf-8 -*-
"""
/***************************************************************************
 HypsometricCurve
                                 A QGIS plugin
 Calculate and draw the hypsometric curve of a drainage basin starting from a vector layer of contour lines and a vector layer that delimits the basin itself.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-12-17
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Dr. Geol. Faustino Cetraro
        email                : geol-faustino@libero.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

import os
import numpy as np
from qgis.core import QgsRasterLayer, QgsRasterBandStats, QgsProject, QgsUnitTypes
from qgis.core import QgsRectangle
from qgis.core import QgsGeometry, QgsPointXY
from qgis.core import QgsFeature, QgsField, QgsFields, QgsVectorLayer
from qgis.core import QgsWkbTypes
from qgis.core import QgsRasterDataProvider
from qgis.core import Qgis
from qgis.core import QgsCoordinateTransform

from PyQt5.QtCore import QVariant
from PyQt5.QtWidgets import QFileDialog, QTableWidgetItem
from PyQt5.QtWidgets import QGraphicsScene
from PyQt5 import QtWidgets
from PyQt5.QtWidgets import QMessageBox
from PyQt5.QtWidgets import QProgressBar
from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QColorDialog
from PyQt5.QtGui import QColor
from PyQt5.QtWidgets import QApplication

from matplotlib import pyplot as plt
from matplotlib.backends.backend_qt5agg import FigureCanvasQTAgg as FigureCanvas
import csv
from osgeo import gdal
from PyQt5.QtCore import QCoreApplication


# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .hypsometric_curve_dialog import HypsometricCurveDialog
import os.path

class HypsometricCurve:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]

        # initialize locale path
        locale_path = os.path.join(self.plugin_dir, 'i18n', f'hypsometric_curve_{locale}.qm')

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            if self.translator.load(locale_path):  # Check if load was successful
                QCoreApplication.installTranslator(self.translator)
                # self.show_message("Success", f"The locale file '{locale}' has been loaded successfully.")
            else:
                self.show_message(self.tr("Errore"), self.tr("Impossibile caricare il file locale: {locale_path}").format(locale_path=locale_path))
        
        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Hypsometric Curve')

        # Check if plugin was started the first time in current QGIS session
        self.first_start = None

    def show_message(self, title, message):
        """Show a dialog message to inform the user."""
        # Initialize a dictionary to track shown messages
        if not hasattr(self, '_shown_messages'):
            self._shown_messages = set()

        # Check if the message has already been shown
        if message not in self._shown_messages:
            msg = QMessageBox()
            msg.setIcon(QMessageBox.Information)
            msg.setText(message)
            msg.setWindowTitle(title)
            msg.exec_()
            self._shown_messages.add(message)  # Aggiunge il messaggio gia' mostrato

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('HypsometricCurve', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        self.add_action(icon_path, text=self.tr(u'Hypsometric Curve'), callback=self.run, parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&Hypsometric Curve'),
                action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
        
        if not hasattr(self, 'dlg') or self.dlg is None:
            self.dlg = HypsometricCurveDialog()

            # Collegamenti dei segnali (fatti solo la prima volta)
            self.dlg.pushButton_calc.clicked.connect(self.calculate_hypsometric_curve)
            self.dlg.pushButton_canc.clicked.connect(self.reset_fields)
            self.dlg.pushButton_salva_tab.clicked.connect(self.save_table)
            self.dlg.pushButton_salva_graph.clicked.connect(self.save_graph)
            self.dlg.pushButton_refresh.clicked.connect(self.pushButton_refresh)
            self.dlg.pushButton_close.clicked.connect(self.handle_close)
            
            # Connect the color select button
            self.dlg.pushButton_color.clicked.connect(self.select_color)
            
            # min and max class intervals
            self.dlg.spinBox_classi.setMinimum(10)
            self.dlg.spinBox_classi.setMaximum(500)

            # Suggested class value
            self.dlg.spinBox_classi.setValue(255)
        
            # Set the first tab as active
            self.dlg.tabWidget.setCurrentIndex(0)

            # Default color
            self.selected_color = QColor("blue")

        #progress bar
        self.dlg.progressBar.setValue(0)

        # Reset fields at every run
        self.reset_fields() 

        # Populate DEM combobox
        self.dlg.cmb_dem.clear()
        dem_layers = [layer.name() for layer in QgsProject.instance().mapLayers().values() if isinstance(layer, QgsRasterLayer)]
        if not dem_layers:
            QMessageBox.warning(self.dlg, self.tr("Errore"), self.tr("Nessun layer DEM disponibile."))
            return
        self.dlg.cmb_dem.addItems(dem_layers)

        # Populate band combobox
        self.dlg.cmb_band.clear()
        raster_name = self.dlg.cmb_dem.currentText()
        raster_layer = next((layer for layer in QgsProject.instance().mapLayers().values() if layer.name() == raster_name), None)
        if raster_layer:
            for band in range(raster_layer.bandCount()):
                self.dlg.cmb_band.addItem(str(band + 1), band + 1)  # Aggiunge testo e dati

        # Populate polygon layers combobox
        self.dlg.cmb_polibac.clear()
        polygon_layers = [layer.name() for layer in QgsProject.instance().mapLayers().values() if isinstance(layer, QgsVectorLayer) and layer.geometryType() == QgsWkbTypes.PolygonGeometry]
        if not polygon_layers:
            QMessageBox.warning(self.dlg, self.tr("Errore"), self.tr("Nessun layer poligonale disponibile."))
            return
        self.dlg.cmb_polibac.addItems(polygon_layers)       

        # Initialize the graph with the default view (axes 0 to 1)
        self.initialize_graph()

        # Resize columns
        self.resize_columns()

        #unity csr
        self.update_units_label()

        # show the dialog
        self.dlg.show()
        
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass

    def select_color(self):
        """Open the color selection dialog and apply the selected color."""
        # Controlla se la finestra di dialogo e' gia' aperta
        if hasattr(self, '_color_dialog_open') and self._color_dialog_open:
            return  # Previeni l'apertura multipla

        # Imposta il flag di apertura
        self._color_dialog_open = True

        try:
            # Usa il dialogo per selezionare un colore
            color = QColorDialog.getColor()

            # Controlla se un colore e' stato selezionato
            if color.isValid():
                # Applica il colore selezionato
                self.dlg.lbl_color.setStyleSheet(f"background-color: {color.name()}; border: 1px solid black;")
                self.selected_color = color

                # Aggiorna il grafico con il nuovo colore se necessario
                if self.dlg.tableWidget_tabella.rowCount() > 0:
                    self.update_graph_color()
        finally:
            # Reset del flag dopo la chiusura della finestra
            self._color_dialog_open = False

    def pushButton_refresh(self):
        # Check if the table is not empty
            if self.dlg.tableWidget_tabella.rowCount() > 0:
                # After selecting the color, update the chart
                self.update_graph_color()

    def update_units_label(self):
        """Update the label with the measurement units based on the CRS of the DEM layer."""
        raster_name = self.dlg.cmb_dem.currentText()
        raster_layer = next((layer for layer in QgsProject.instance().mapLayers().values() if layer.name() == raster_name), None)

        if not raster_layer:
            self.iface.messageBar().pushMessage(self.tr("Errore"), self.tr("Seleziona un layer DEM valido"), level=3)
            return
        
        # Check if the CRS of the layers is geographic
        dem_crs = raster_layer.crs()

        # Get the linear unit
        distance_unit = dem_crs.mapUnits()

        if distance_unit == QgsUnitTypes.DistanceMeters:
            self.dlg.lbl_hmin.setText("m")
            self.dlg.lbl_hmax.setText("m")
            self.dlg.lbl_hmed.setText("m")
            self.dlg.lbl_A.setText("m^2")

        elif distance_unit == QgsUnitTypes.DistanceFeet:
            self.dlg.lbl_hmin.setText("ft")
            self.dlg.lbl_hmax.setText("ft")
            self.dlg.lbl_hmed.setText("ft")
            self.dlg.lbl_A.setText("ft^2")
        else:
            self.dlg.lbl_hmin.setText("grad")
            self.dlg.lbl_hmax.setText("grad")
            self.dlg.lbl_hmed.setText("grad")
            self.dlg.lbl_A.setText("---")
  
    #nuovo
    def calculate_hypsometric_curve(self):
        """Perform hypsometric calculations."""

        # Progress bar
        self.dlg.progressBar.setValue(0)

        # Check if there is data in the table before proceeding
        if self.dlg.tableWidget_tabella.rowCount() > 0:
            reply = QMessageBox.question(self.dlg, self.tr('Conferma'),
                                        self.tr("Vuoi procedere con un nuovo calcolo?"),
                                        QMessageBox.Yes | QMessageBox.No, QMessageBox.No)
            if reply == QMessageBox.No:
                return

        # Get selected DEM layer
        raster_name = self.dlg.cmb_dem.currentText()
        raster_layer = next((layer for layer in QgsProject.instance().mapLayers().values() if layer.name() == raster_name), None)

        if not raster_layer:
            self.iface.messageBar().pushMessage(self.tr("Errore"), self.tr("Seleziona un layer DEM valido"), level=3)
            return

        # Check if the CRS of the layer is geographic
        dem_crs = raster_layer.crs()
        if dem_crs.isGeographic():
            QMessageBox.warning(None, self.tr("CRS Geografico"), self.tr("Utilizzare un sistema proiettato, come UTM."))
            return

        # Progress bar
        self.dlg.progressBar.setValue(10)

        # Get the linear unit
        distance_unit = dem_crs.mapUnits()
        if distance_unit == QgsUnitTypes.DistanceMeters:
            self.dlg.lbl_hmin.setText("m")
            self.dlg.lbl_hmax.setText("m")
            self.dlg.lbl_hmed.setText("m")
            self.dlg.lbl_A.setText("m^2")
        elif distance_unit == QgsUnitTypes.DistanceFeet:
            self.dlg.lbl_hmin.setText("ft")
            self.dlg.lbl_hmax.setText("ft")
            self.dlg.lbl_hmed.setText("ft")
            self.dlg.lbl_A.setText("ft^2")
        else:
            self.dlg.lbl_hmin.setText("grad")
            self.dlg.lbl_hmax.setText("grad")
            self.dlg.lbl_hmed.setText("grad")
            self.dlg.lbl_A.setText("---")

        # Get the selected band
        band_index = self.dlg.cmb_band.currentData()
        if band_index is None:
            self.dlg.progressBar.setValue(0)
            QMessageBox.critical(self.dlg, self.tr("Errore"), self.tr("Nessuna banda selezionata."))
            return

        # Progress bar
        self.dlg.progressBar.setValue(20)

        provider = raster_layer.dataProvider()

        # Get selected vector polygon layer
        basin_name = self.dlg.cmb_polibac.currentText()
        basin_layer = next((layer for layer in QgsProject.instance().mapLayers().values() if layer.name() == basin_name), None)

        if not basin_layer:
            self.dlg.progressBar.setValue(0)
            self.iface.messageBar().pushMessage(self.tr("Errore"), self.tr("Seleziona un layer vettoriale valido"), level=3)
            return

        # Ensure CRS compatibility
        if raster_layer.crs() != basin_layer.crs():
            QMessageBox.critical(None, self.tr("Errore"), self.tr("I layer DEM e vettoriale devono avere lo stesso CRS."))
            return

        # Extract vector geometry
        basin_features = list(basin_layer.getFeatures())
        if len(basin_features) == 0:
            self.dlg.progressBar.setValue(0)
            QMessageBox.critical(None, self.tr("Errore"), self.tr("Il layer vettoriale e' vuoto."))
            return

        # Progress bar
        self.dlg.progressBar.setValue(30)

        basin_geom = basin_features[0].geometry()

        # Control which function to use based on the combobox index
        index = self.dlg.cmb_statist_raster.currentIndex()

        if index == 0:  # Full raster
            file_path = raster_layer.dataProvider().dataSourceUri()
            dataset = gdal.Open(file_path)
            if not dataset:
                self.dlg.progressBar.setValue(0)
                QMessageBox.critical(self.dlg, self.tr("Errore"), self.tr("Impossibile aprire il file raster con GDAL."))
                return

            band = dataset.GetRasterBand(band_index)
            width = dataset.RasterXSize  # Set width
            height = dataset.RasterYSize  # Set height
            block = band.ReadAsArray(0, 0, width, height)  # Ottieni l'intero array di dati

            # Verifica il tipo di dato
            # dtype = np.float32 if band.DataType == gdal.GDT_Float32 else np.float64

            # Mappa dei tipi GDAL a NumPy
            gdal_to_numpy_dtype = {
                gdal.GDT_Byte: np.uint8,
                gdal.GDT_UInt16: np.uint16,
                gdal.GDT_Int16: np.int16,
                gdal.GDT_UInt32: np.uint32,
                gdal.GDT_Int32: np.int32,
                gdal.GDT_Float32: np.float32,
                gdal.GDT_Float64: np.float64
            }

            # Verifica il tipo di dato GDAL e mappa al tipo NumPy
            gdal_dtype = band.DataType
            dtype = gdal_to_numpy_dtype.get(gdal_dtype, np.float64)  # Default a np.float64 se non riconosciuto

            try:
                data = block.astype(dtype)
            except Exception as e:
                self.dlg.progressBar.setValue(0)
                QMessageBox.critical(self.dlg, self.tr("Errore"), self.tr(f"Errore durante la lettura del raster: {e}"))
                return

            stats = band.GetStatistics(True, True)
            min_value, max_value = stats[0], stats[1]
            
            # Definisci extent come l'estensione completa del raster
            extent = raster_layer.extent()

        else:  # Current view
            stats = provider.bandStatistics(band_index, QgsRasterBandStats.All)
            if stats.elementCount == 0:
                self.dlg.progressBar.setValue(0)
                QMessageBox.critical(self.dlg, self.tr("Errore"), self.tr("Nessun dato disponibile nella banda selezionata."))
                return

            min_value, max_value = stats.minimumValue, stats.maximumValue
            extent = raster_layer.extent()
            width, height = raster_layer.width(), raster_layer.height()

            # Get raster block for analysis extent
            block = provider.block(band_index, extent, width, height)

            # Verifica il tipo di dato del raster e utilizza il tipo appropriato per NumPy
            #try:
                #dtype = np.float32 if provider.dataType(band_index - 1) == 5 else np.float64
            #except Exception as e:
                #QMessageBox.critical(self.dlg, self.tr("Errore"), self.tr(f"Errore nell'ottenere il tipo di dati: {str(e)}"))
                #return
            
            # Verifica il tipo di dato del raster e utilizza il tipo appropriato per NumPy
            try:
                provider_dtype_map = {
                    1: np.uint8,
                    2: np.uint16,
                    3: np.int16,
                    4: np.uint32,
                    5: np.float32,
                    6: np.float64,
                }
                gdal_provider_type = provider.dataType(band_index - 1)
                dtype = provider_dtype_map.get(gdal_provider_type, np.float64)  # Default a np.float64 se non riconosciuto
            except Exception as e:
                QMessageBox.critical(self.dlg, self.tr("Errore"), self.tr(f"Errore nell'ottenere il tipo di dati: {str(e)}"))
                return

            data = np.frombuffer(block.data(), dtype=dtype)
        
        # Verifica che extent sia definito
        if extent is None:
            self.dlg.progressBar.setValue(0)
            QMessageBox.critical(self.dlg, self.tr("Errore"), self.tr("L'estensione del raster non e' stata inizializzata correttamente."))
            return

        # Verifica che il numero di celle nel blocco corrisponda al numero atteso
        expected_size = width * height
        actual_size = data.size

        if actual_size != expected_size:
            if actual_size > expected_size:
                # Trunca i dati se sono troppi
                data = data[:expected_size]
            else:
                # Calcola il numero di righe mancanti
                missing_rows = expected_size - actual_size

                # Crea un array di padding con valori di NoData
                no_data_value = provider.sourceNoDataValue(band_index)
                padding = np.full((missing_rows,), no_data_value, dtype=dtype)

                # Concatena i dati esistenti con il padding
                data = np.concatenate((data, padding))

        # Rimodella i dati correttamente
        try:
            data = data.reshape((height, width))
        except ValueError as e:
            self.dlg.progressBar.setValue(0)
            QMessageBox.critical(self.dlg, self.tr("Errore"), self.tr(f"Errore di rimodellamento: {e}"))
            return

        # Progress bar
        self.dlg.progressBar.setValue(40)

        # Mask NoData values
        no_data_value = provider.sourceNoDataValue(band_index)
        valid_data = np.ma.masked_equal(data, no_data_value)

        # Create a mask using the polygon
        mask = np.zeros((height, width), dtype=bool)

        # Total iterations
        total_steps = height * width
        completed_steps = 0

        for row in range(height):
            for col in range(width):
                x = extent.xMinimum() + col * raster_layer.rasterUnitsPerPixelX()
                y = extent.yMaximum() - row * raster_layer.rasterUnitsPerPixelY()
                point = QgsPointXY(x, y)

                if basin_geom.contains(QgsGeometry.fromPointXY(point)):
                    mask[row, col] = True

                # Increment the completed steps counter
                completed_steps += 1

                # Update the progress bar based on the completed steps
                progress = int((completed_steps / total_steps) * 100)  # Percentage completed
                self.dlg.progressBar.setValue(progress)

           # Progress bar
        self.dlg.progressBar.setValue(50)

        # Apply the mask to the raster data
        masked_data = np.ma.masked_where(~mask, valid_data)

        # Calculate statistics
        h_min = min_value  # masked_data.min()
        h_max = max_value  # masked_data.max()
        self.dlg.lineEdit_hmin.setText(f"{h_min:.2f}")
        self.dlg.lineEdit_hmax.setText(f"{h_max:.2f}")
        
        # Dimensione della cella
        cell_area = abs(raster_layer.rasterUnitsPerPixelX() * raster_layer.rasterUnitsPerPixelY())

        # Dimensione totale dei dati
        data_size = masked_data.shape[0]  # Numero di righe (o un asse rilevante per iterare)
        chunk_size = 100  # Definire il numero di righe da processare in ogni iterazione
        total_area = 0  # Area totale inizializzata a 0

        # Inizializza la barra di progresso
        self.dlg.progressBar.setValue(0)

        # Iterazione sui dati in blocchi per calcolare progressivamente l'area
        for start in range(0, data_size, chunk_size):
            end = min(start + chunk_size, data_size)  # Limita l'intervallo all'interno del dataset
            chunk = masked_data[start:end]  # Estrai il blocco corrente

            # Calcola il numero di celle non mascherate nel blocco
            total_area += np.count_nonzero(~chunk.mask) * cell_area

            # Aggiorna la barra di progresso
            progress = int(((start + chunk_size) / data_size) * 100)  # Percentuale completata
            self.dlg.progressBar.setValue(progress)

            # Aggiorna l'interfaccia utente
            QCoreApplication.processEvents()

        self.dlg.progressBar.setValue(60)

        self.dlg.lineEdit_A.setText(f"{total_area:.2f}")

        # Calculate cumulative areas and intervals
        num_classes = self.dlg.spinBox_classi.value()
        intervals = np.linspace(h_min, h_max, num_classes + 1)

        # Calculate areas using the updated `calculate_area_in_range`
        partial_areas, cumulative_areas = self.calculate_area_in_range(raster_layer, cell_area, band_index, basin_geom, dtype, index)

        if partial_areas is None or cumulative_areas is None:
            # Gestisci l'errore, ad esempio, fermando il processo o mostrando un messaggio all'utente
            self.dlg.progressBar.setValue(0)
            self.clear_memory()
            return

        # Modifica del TypeError: cumulative_areas convertito in lista mutabile.
        cumulative_areas = list(cumulative_areas)
        cumulative_areas[0] = total_area  # Ora supporta l'assegnazione

        # Progress bar
        self.dlg.progressBar.setValue(65)

        # Ensure the first cumulative area equals the total area
        if total_area != cumulative_areas[0]:
            cumulative_areas[0] = total_area

        # Calculate h_med
        statistical_means = self.calculate_statistical_means(valid_data, intervals)
        h_med = self.calculate_h_med(valid_data, intervals)

        self.dlg.lineEdit_hmed.setText(f"{h_med:.2f}")

        # Progress bar
        self.dlg.progressBar.setValue(70)

        # hypsometric
        hi = self.calculate_hypsometric_mean(statistical_means, h_min, h_max)
        
        self.dlg.lineEdit_HI.setText(f"{hi:.3f}")

        # Progress bar
        self.dlg.progressBar.setValue(90)

        # Correzione: utilizzo della funzione corretta per popolare la tabella
        self.populate_table_with_corrected_values(intervals, partial_areas, cumulative_areas, total_area, statistical_means)

        # Progress bar
        self.dlg.progressBar.setValue(100)

        #plot
        self.plot_graph(hi)

        # Reset progress bar
        self.dlg.progressBar.setValue(0)

    def create_contour_polygon(self, valid_mask, raster_layer):
        """Creates a boundary polygon based on valid pixels."""
        # Find the coordinates of valid pixels (pixels that are 'True' in the valid_mask)
        valid_coords = np.column_stack(np.where(valid_mask))

        # Get the CRS of the current project
        project_crs = QgsProject.instance().crs()

        points = []
        for coord in valid_coords:
            if len(coord) == 2:  # Check that 'coord' has 2 values
                row, col = coord  # Now valid_coords is two-dimensional (row, col)
            else:
                continue  # If coord' does not have 2 values, skip the loop for this element

            # Calculate the x, y position in the raster reference system
            x = raster_layer.extent().xMinimum() + col * raster_layer.rasterUnitsPerPixelX()
            y = raster_layer.extent().yMaximum() - row * raster_layer.rasterUnitsPerPixelY()
            point = QgsPointXY(x, y)

            # Transforming geometry into the project CRS
            point = point.transform(raster_layer.crs(), project_crs)
            points.append(point)

        if len(points) == 0:
            return None

        # Create a polygon from the coordinate contour
        contour_polygon = QgsGeometry.fromPolygonXY([points])

        return contour_polygon

    def add_polygon_to_map(self, polygon):
        """Adds the contour polygon to the map."""
        # Get the CRS of the current project
        project_crs = QgsProject.instance().crs()

        # Create a temporary vector layer for the polygon
        fields = QgsFields()
        fields.append(QgsField("id", QVariant.Int))
        layer = QgsVectorLayer("Polygon?crs=" + project_crs.toWkt(), "Contour Polygon", "memory")
        layer.dataProvider().addAttributes(fields)
        layer.updateFields()

        # Create a feature with the polygon and add it to the layer
        feature = QgsFeature()
        feature.setGeometry(polygon)
        feature.setAttributes([1])  # ID attributo
        layer.dataProvider().addFeature(feature)

        # Add layer to map
        QgsProject.instance().addMapLayer(layer)

    
    def calculate_area_in_range(self, raster_layer, cell_area, band_index, basin_geom, dtype, index):
        """
        Calcola le aree parziali e cumulative dei pixel in intervalli di elevazione.
        """
        dataset = None  # Inizializza come None

        if index==0: 
            # Full raster analysis
            file_path = raster_layer.dataProvider().dataSourceUri()
            dataset = gdal.Open(file_path)
            if not dataset:
                QMessageBox.critical(self.dlg, self.tr("Errore"), self.tr("Impossibile aprire il file raster con GDAL."))
                return None, None

            band = dataset.GetRasterBand(band_index)
            width, height = dataset.RasterXSize, dataset.RasterYSize
            block = band.ReadAsArray(0, 0, width, height).astype(dtype)

            # Estensione completa del raster
            extent = raster_layer.extent()

        else:
            # Analisi della vista corrente
            provider = raster_layer.dataProvider()
            extent = raster_layer.extent()
            width, height = raster_layer.width(), raster_layer.height()
            block = provider.block(band_index, extent, width, height)
            
            # Verifica che i dati abbiano la dimensione corretta
            raw_data = np.frombuffer(block.data(), dtype=dtype)
            expected_size = width * height
            actual_size = raw_data.size

            if actual_size != expected_size:
                QMessageBox.warning(
                    self.dlg,
                    self.tr("Attenzione"),
                    self.tr(f"Dimensione dei dati ({actual_size}) non corrisponde a quella attesa ({expected_size}).")
                )
                return None, None  # Interrompi il calcolo se le dimensioni non sono corrette

            # Rimodella i dati
            block = raw_data.reshape((height, width))

        # Maschera i valori NoData
        no_data_value = raster_layer.dataProvider().sourceNoDataValue(band_index)
        data = np.ma.masked_equal(block, no_data_value)

        # Maschera i pixel esterni alla geometria del bacino
        mask = np.zeros((height, width), dtype=bool)
        for row in range(height):
            for col in range(width):
                x = extent.xMinimum() + col * raster_layer.rasterUnitsPerPixelX()
                y = extent.yMaximum() - row * raster_layer.rasterUnitsPerPixelY()
                point = QgsPointXY(x, y)
                if basin_geom.contains(QgsGeometry.fromPointXY(point)):
                    mask[row, col] = True

            # Aggiorna la progress bar
            progress = int((row / height) * 100)
            self.dlg.progressBar.setValue(progress)

        masked_data = np.ma.masked_where(~mask, data)

        # Calcola le statistiche
        num_classes = self.dlg.spinBox_classi.value()
        h_min, h_max = masked_data.min(), masked_data.max()
        intervals = np.linspace(h_min, h_max, num_classes + 1)

        partial_areas = []
        cumulative_areas = []

        for i in range(num_classes):
            lower, upper = intervals[i], intervals[i + 1]
            interval_mask = (masked_data >= lower) & (masked_data < upper)
            interval_area = np.sum(interval_mask) * cell_area

            # Aggiungi l'area parziale
            partial_areas.append(interval_area)

        # Calcolo delle aree cumulative invertendo l'ordine
        cumulative_sum = 0
        for area in reversed(partial_areas):
            cumulative_sum += area
            cumulative_areas.insert(0, cumulative_sum)

        return partial_areas, cumulative_areas

    def calculate_statistical_means(self, valid_data, intervals):
        """
        Calcola il valore medio del raster per ciascun intervallo.
        """
        means = []

        # Verifica che valid_data sia un array NumPy mascherato
        if not isinstance(valid_data, np.ma.MaskedArray):
            valid_data = np.ma.masked_array(valid_data)

        if valid_data.count() == 0:
            raise ValueError(self.tr("Non ci sono dati validi nel raster."))

        num_intervals = len(intervals) - 1  # Numero di intervalli

        for i in range(num_intervals):
            lower, upper = intervals[i], intervals[i + 1]

            # Crea una maschera booleana per selezionare i valori nell'intervallo specificato
            interval_mask = (valid_data >= lower) & (valid_data < upper)

            # Applica la maschera per ottenere i valori nell'intervallo
            interval_values = valid_data[interval_mask]

            # Calcola il valore medio dell'intervallo (se ci sono valori, altrimenti 0)
            mean_value = interval_values.mean() if interval_values.size > 0 else 0
            means.append(mean_value)

            # Calcola e aggiorna il progresso della progress bar
            progress = int((i + 1) / num_intervals * 100)  # Percentuale completata
            self.dlg.progressBar.setValue(progress)

            # Aggiorna l'interfaccia per riflettere il progresso
            QCoreApplication.processEvents()

        return means
    
    # Funzione per calcolare h_med richiamando i valori medi
    def calculate_h_med(self, valid_data, intervals):
        """
        Calcola il valore medio finale (h_med) utilizzando i valori medi degli intervalli.
        """
        # Richiama la funzione calculate_statistical_means per ottenere i valori medi
        means = self.calculate_statistical_means(valid_data, intervals)
        # Calcola il valore medio finale (h_med) come la media dei valori medi
        h_med = sum(means) / len(means) if means else 0
        return h_med

    def calculate_hypsometric_mean(self, statistical_means, h_min, h_max):
        """
        Calcola l'HI (Hypsometric Integral) usando i valori medi normalizzati per ciascun intervallo.
        """
        # Controlla se gli input sono validi
        if not statistical_means or h_max == h_min:
            return 0

        # Normalizza i valori medi rispetto all'intervallo totale (h_min, h_max)
        partial_hi = [
            (mean_value - h_min) / (h_max - h_min)  # Normalizzazione rispetto a h_min e h_max
            for mean_value in statistical_means
        ]

        # Calcola il valore medio di HI
        hi_mean = np.mean(partial_hi) if partial_hi else 0
        return hi_mean

    def populate_table_with_corrected_values(self, intervals, partial_areas, cumulative_areas, total_area, valid_data):
        """
        Fill table with corrected hypsometric data including proper calculation of A, h, and h_tot.
        """
        # Calcola i valori medi degli intervalli
        statistical_means = self.calculate_statistical_means(valid_data, intervals)

        # Inizializza la somma progressiva di h
        h_sum = 0
        h_tot = sum(intervals[1:] - intervals[:-1])  # Valore totale di h

        self.dlg.tableWidget_tabella.setRowCount(len(intervals) - 1)

        for i in range(len(intervals) - 1):
            # Altezza dell'intervallo
            h = intervals[i + 1] - intervals[i]
            h_sum += h  # Aggiorna la somma cumulativa di h

            # Altezza cumulativa normalizzata
            h_h_tot = h_sum / h_tot if h_tot != 0 else 0

            # Area singola dell'intervallo
            a_area = partial_areas[i]

            # Area cumulativa corrente
            a_cum = cumulative_areas[i]

            # Calcola il rapporto a_cum/A (area cumulativa / area totale)
            a_cum_norm = a_cum / total_area

            # Area dell'intervallo (dA)
            if i < len(cumulative_areas) - 1:
                d_a = a_cum - cumulative_areas[i + 1]
            else:
                d_a = a_cum  # Per l'ultimo intervallo
            
            # Valore medio statistico per l'intervallo
            hmed = statistical_means[i]

            # Popolamento delle celle della tabella
            self.dlg.tableWidget_tabella.setItem(i, 0, QTableWidgetItem(f"{intervals[i]:.2f}-{intervals[i + 1]:.2f}"))
            self.dlg.tableWidget_tabella.setItem(i, 1, QTableWidgetItem(f"{a_area:.2f}"))       # da
            self.dlg.tableWidget_tabella.setItem(i, 2, QTableWidgetItem(f"{a_cum:.2f}"))        # a
            self.dlg.tableWidget_tabella.setItem(i, 3, QTableWidgetItem(f"{a_cum_norm:.4f}"))   # a/A
            self.dlg.tableWidget_tabella.setItem(i, 4, QTableWidgetItem(f"{h_sum:.2f}"))        # h cumulativo
            self.dlg.tableWidget_tabella.setItem(i, 5, QTableWidgetItem(f"{hmed:.2f}"))         # hmed
            self.dlg.tableWidget_tabella.setItem(i, 6, QTableWidgetItem(f"{h_h_tot:.4f}"))      # h/H

        # Allineamento delle colonne dopo aver scritto i dati nella tabella
        self.align_columns()

        # Ridimensionamento delle colonne
        self.resize_columns()

    def plot_graph(self, hi): 
        """Plot hypsometric curve."""
        # Estrai i valori di a_norm e h_norm dalle colonne 3 e 6 della tabella
        a_norm = []
        h_norm = []
        row_count = self.dlg.tableWidget_tabella.rowCount()

        for i in range(row_count):
            # Ottieni i valori dalla colonna 3 (a_norm) e 6 (h_norm) della tabella
            a_norm_value = float(self.dlg.tableWidget_tabella.item(i, 3).text())
            h_norm_value = float(self.dlg.tableWidget_tabella.item(i, 6).text())
            
            a_norm.append(a_norm_value)
            h_norm.append(h_norm_value)

        # Crea una nuova figura con la dimensione specificata (5.21 x 3.51 pollici corrispondente a 521x351 pixel)
        fig, ax = plt.subplots(figsize=(5.21, 3.51))

        # Traccia la curva ipsometrica usando a_norm e h_norm
        ax.plot(a_norm, h_norm, color=self.selected_color.name(), label=self.tr("Curva ipsometrica"))
        ax.set_xlabel("a/A", labelpad=5, fontweight="bold")
        ax.set_ylabel("h/H", labelpad=5, fontweight="bold")
        ax.set_title(self.tr("Curva ipsometrica"), pad=10, fontweight="bold")
        ax.legend()
        ax.grid(True)  # Mostra la griglia
        
        # Use tight_layout to automatically adjust the subplots to fit the figure area
        plt.tight_layout(pad=0.5)  # Ottimizza automaticamente i margini per evitare che vengano tagliati

        # Imposta i limiti degli assi da 0 a 1
        ax.set_xlim(0, 1.1)
        ax.set_ylim(0, 1.1)              

        # Inizializza hi_projection come None per verificare se viene calcolato successivamente
        hi_projection = None

        # Mostra il punto HI sulla curva se la checkbox e' selezionata
        if self.dlg.checkBox_HI.isChecked():
            # Trova il punto piu' vicino sulla curva ipsometrica
            for i in range(len(h_norm) - 1):
                if h_norm[i] <= hi <= h_norm[i + 1]:
                    # Interpolazione lineare per trovare il valore esatto di a/A
                    slope = (a_norm[i + 1] - a_norm[i]) / (h_norm[i + 1] - h_norm[i])
                    hi_projection = a_norm[i] + slope * (hi - h_norm[i])  # a/A
                    break  # Esci dal ciclo una volta che la proiezione e' trovata
            
            # Verifica se la proiezione di HI e' stata calcolata correttamente
            if hi_projection is not None:
                # Aggiungi linee tratteggiate proiettate sugli assi
                ax.axhline(hi, color='green', linestyle='--', linewidth=0.8)

                # Disegna un piccolo cerchio sul punto HI sulla curva
                ax.plot([hi_projection], [hi], 'o', color='red', label="HI")

                # Aggiungi il valore di HI come etichetta
                ax.text(
                    hi_projection + 0.02,  # Posiziona il testo leggermente a destra del punto
                    hi + 0.02,  # Posiziona il testo leggermente sopra il punto
                    f"HI = {hi:.3f}",  # Testo con il valore di HI
                    color="black",
                    fontsize=9,
                    fontweight='bold'  # Testo in grassetto
                )
            else:
                # Gestisci il caso in cui la proiezione di HI non sia stata trovata
                print(self.tr("Proiezione di HI non trovata."))
        
        # Crea il canvas per visualizzare il grafico nel QGraphicsView
        canvas = FigureCanvas(fig)
        canvas.setFixedSize(521, 351)  # Assicurati che il canvas si adatti alle dimensioni della vista grafica

        # Pulisce la scena precedente e aggiunge il nuovo grafico alla QGraphicsView
        self.dlg.graphicsView_grafico.setScene(QGraphicsScene())
        self.dlg.graphicsView_grafico.scene().addWidget(canvas)

        # Disegna il canvas
        canvas.draw()

    def initialize_graph(self):
        """Initialize the graph with axes from 0 to 1 and show the grid."""
        # Create a figure and axis for the plot
        fig, ax = plt.subplots(figsize=(5.21, 3.51)) # Size in inches, corresponding to 521x351 pixels at 100 dpi
        
        # Set the x and y axis limits from 0 to 1
        ax.set_xlim(0, 1.1)
        ax.set_ylim(0, 1.1)
        
        # Set labels for the axes
        ax.set_xlabel("a/A", labelpad=5, fontweight="bold", fontsize=10)
        ax.set_ylabel("h/H", labelpad=5, fontweight="bold", fontsize=10)
        ax.set_title(self.tr("Curva ipsometrica"), pad=10, fontweight="bold", fontsize=12)
        
        # Add a grid
        ax.grid(True)
        
        # Adjust the margins of the plot to make sure labels and title are not cut off
        # plt.subplots_adjust(bottom=0.15, left=0.1, right=0.9, top=0.85)  # Modifica i margini (aumenta bottom, top, left, right)
    
        # Use tight_layout to automatically adjust the subplots to fit the figure area
        plt.tight_layout(pad=0.5)  # Ottimizza automaticamente i margini per evitare che vengano tagliati
        
        # Create a canvas for displaying the figure in the graphics view
        canvas = FigureCanvas(fig)

        # Set the canvas size to match the graphics view
        canvas.setFixedSize(521, 351)
        
        # Clear the previous content and add the new plot to the QGraphicsView
        self.dlg.graphicsView_grafico.setScene(QGraphicsScene())
        self.dlg.graphicsView_grafico.scene().addWidget(canvas)
        
        # Draw the canvas
        canvas.draw()

    def update_graph_color(self):
        """Update the chart color without recalculating the values."""
        # Retrieve data from UI widgets
        hi_value = float(self.dlg.lineEdit_HI.text())   # HI

        # Retrieve data from table
        a_norm = []
        h_norm = []
        
        for row in range(self.dlg.tableWidget_tabella.rowCount()):
            a_norm.append(float(self.dlg.tableWidget_tabella.item(row, 3).text()))   # a/A
            h_norm.append(float(self.dlg.tableWidget_tabella.item(row, 6).text()))   # h/H

        # Create a new figure with the specified dimensions (5.21 x 3.51 inches)
        fig, ax = plt.subplots(figsize=(5.21, 3.51))
        
        # Draw the hypsometric curve with the selected color
        ax.plot(a_norm, h_norm, color=self.selected_color.name(), label=self.tr("Curva ipsometrica"))
        ax.set_xlabel("a/A", labelpad=5, fontweight="bold", fontsize=10)
        ax.set_ylabel("h/H", labelpad=5, fontweight="bold", fontsize=10)
        ax.set_title(self.tr("Curva ipsometrica"), pad=10, fontweight="bold", fontsize=12)
        ax.legend()
        ax.grid(True)  # Show grid

         # Adjust the layout to ensure titles and labels fit properly
        plt.tight_layout(pad=0.5)

        # Set the x and y axis limits from 0 to 1.1
        ax.set_xlim(0, 1.1)
        ax.set_ylim(0, 1.1)

        # Show HAI point on curve if checkbox is selected
        if self.dlg.checkBox_HI.isChecked():
            
            # Find the closest point to the hypsometric curve
            for i in range(len(h_norm) - 1):
                if h_norm[i] <= hi_value <= h_norm[i + 1]:
                    # Linear interpolation to find the exact value of a/A
                    slope = (a_norm[i + 1] - a_norm[i]) / (h_norm[i + 1] - h_norm[i])
                    hi_projection = a_norm[i] + slope * (hi_value - h_norm[i])  # a/A
                    break
            
            # Add dashed lines projected onto the axes
            ax.axhline(hi_value, color='green', linestyle='--', linewidth=0.8)
            
            # Draw a small circle on point HI on the curve
            ax.plot([hi_projection], [hi_value], 'o', color='red', label="HI")

            # Add the HI value as a label
            ax.text(
                hi_projection + 0.02,  # Place x slightly to the right of the dot
                hi_value + 0.02,  # Place y slightly above the point
                f"HI = {hi_value:.3f}",  # Text with HI value
                color="black",
                fontsize=9,
                fontweight='bold'       # Adds bold to text
            )

        # Create the FigureCanvas to display the graph in the graphics view
        canvas = FigureCanvas(fig)
        canvas.setFixedSize(521, 351)  # Set the canvas to a fixed size

        # Clears the previous content and adds the new chart to the graph view
        self.dlg.graphicsView_grafico.setScene(QGraphicsScene())
        self.dlg.graphicsView_grafico.scene().addWidget(canvas)

        # Draw the canvas
        canvas.draw()

    def reset_fields(self):
        """Reset all input and output fields."""
        # Controlla se self.dlg e' correttamente inizializzato e contiene tableWidget_tabella
        if self.dlg and hasattr(self.dlg, 'tableWidget_tabella') and self.dlg.tableWidget_tabella:

            # Check if there is data in the table before proceeding
            if self.dlg.tableWidget_tabella.rowCount() > 0:

                # Create the confirmation message
                reply = QMessageBox.question(self.dlg, self.tr('Conferma'),
                                            self.tr("Sei sicuro di voler resettare tutti i campi e i dati?"),
                                            QMessageBox.Yes | QMessageBox.No, QMessageBox.No)
                
                if reply == QMessageBox.Yes:
                    # If "Yes"
                    # Clear your memory
                    self.clear_memory()

                    # Reset the graph to the initial state with axes from 0 to 1 and grid
                    self.initialize_graph()
                else:
                    # If "No
                    pass
    
    def clear_memory(self):
        """Clear internal memory of stored data."""

        self.cumulative_areas = []
        self.total_area = 0.0
        self.h_min = 0.0
        self.h_max = 0.0
        self.hypsometric_mean = 0.0

        # reset
        self.dlg.lineEdit_hmin.setText("0.00")
        self.dlg.lineEdit_hmax.setText("0.00")
        self.dlg.lineEdit_A.setText("0.00")
        self.dlg.lineEdit_hmed.setText("0.00")
        self.dlg.lineEdit_HI.setText("0.00")

        # Reset the tabel
        self.dlg.tableWidget_tabella.clearContents()  # Svuota il contenuto della tabella
        self.dlg.tableWidget_tabella.setRowCount(0)   # Elimina tutte le righe     

    def save_table(self):
        """Save table data to a CSV file."""
        table = self.dlg.tableWidget_tabella

        # Check if the table contains any data
        if table.rowCount() == 0:
            QtWidgets.QMessageBox.warning(self.dlg, self.tr("Attenzione"), self.tr("La tabella e' vuota. Esegui il calcolo prima di salvare."))
            return

        filename, _ = QtWidgets.QFileDialog.getSaveFileName(self.dlg, self.tr("Salva la Tabella"), "", self.tr("CSV Files (*.csv);;Text Files (*.txt)"))
        
        # Stop if user pressed "Cancel"
        if not filename:
            return
       
        # Retrieve decimal separator from combobox (0 for period, 1 for comma)
        decimal_separator = '.' if self.dlg.cmb_decimal.currentIndex() == 0 else ','
        
        if filename:
            try:
                with open(filename, 'w', newline='', encoding='utf-8') as f:
                    writer = csv.writer(f, delimiter=';')
                    writer.writerow([self.tr("Intervalli"), "da", "a", "a/A", "h", "hmed", "h/H"])
                    
                    for row in range(table.rowCount()):
                        row_data = []
                        for col in range(table.columnCount()):
                            
                            cell_text = table.item(row, col).text()
                            
                            try:
                                # If the value is numeric, convert it to the correct format
                                numeric_value = float(cell_text)
                                formatted_value = f"{numeric_value:.6f}".replace('.', decimal_separator)
                                row_data.append(formatted_value)
                            except ValueError:
                                # If it is not a number, add the text as is
                                row_data.append(cell_text)
                        
                        writer.writerow(row_data)               

                # Show a confirmation message
                QtWidgets.QMessageBox.information(self.dlg, self.tr("Salvataggio completato"), self.tr("I dati sono stati salvati correttamente!"))
            
            except Exception as e:
                # If there is an error while saving, it displays an error message
                QtWidgets.QMessageBox.critical(self.dlg, self.tr("Errore"), self.tr(f"Si e' verificato un errore durante il salvataggio: {str(e)}"))

    def save_graph(self):
        """Save graph to an image file."""
        table = self.dlg.tableWidget_tabella

        # Check if the table contains any data
        if table.rowCount() == 0:
            QtWidgets.QMessageBox.warning(self.dlg, self.tr("Attenzione"), self.tr("Nessun dato presente sul grafico da salvare!"))
            return

        path, _ = QFileDialog.getSaveFileName(None, self.tr("Salva grafico"), "", self.tr("Images (*.png *.jpg)"))

        # Stop if user pressed "Cancel"
        if not path:
            return
        
        if path:
            try:
                plt.savefig(path)
                
                # Show a confirmation message
                QMessageBox.information(None, self.tr("Salvataggio completato"), self.tr("Il grafico e' stato salvato correttamente!"))
            
            except Exception as e:
                # Show an error message if there is a problem while saving
                QMessageBox.critical(None, self.tr("Errore"), self.tr(f"Si e' verificato un errore durante il salvataggio del grafico: {str(e)}"))
    
    def align_columns(self):
        """Align table columns as required."""
        table = self.dlg.tableWidget_tabella
        
        # Iterate over all rows and columns
        for row in range(table.rowCount()):
            for col in range(table.columnCount()):
                item = table.item(row, col)
                if item:
                    # Align the first column to the left
                    if col == 0:
                        item.setTextAlignment(Qt.AlignLeft)
                    else:
                        # Align all other columns to the right
                        item.setTextAlignment(Qt.AlignRight)
    
    def resize_columns(self):
        """Resize columns."""
        table = self.dlg.tableWidget_tabella
        table.setColumnWidth(0, 110)    # Intervalli
        table.setColumnWidth(1, 80)     # da
        table.setColumnWidth(2, 100)    # a
        table.setColumnWidth(3, 60)     # a/A
        table.setColumnWidth(4, 60)     # h
        table.setColumnWidth(5, 60)     # hmed
        table.setColumnWidth(6, 60)     # h/H
    
    def handle_close(self):
        """Clears the UI and closes the window."""
        if not hasattr(self, 'dlg') or self.dlg is None:
            return  # Nessuna finestra da chiudere

        try:
            # Check if the table contains non-empty data
            table_widget = self.dlg.tableWidget_tabella
            if table_widget is not None:
                has_non_empty_rows = any(
                    table_widget.item(row, col) is not None and table_widget.item(row, col).text().strip() != ""
                    for row in range(table_widget.rowCount())
                    for col in range(table_widget.columnCount())
                )
            else:
                has_non_empty_rows = False

            # If the table is empty, close the window directly
            if not has_non_empty_rows:
                if self.dlg:
                    self.dlg.close()
                return

            # Show a confirmation message
            reply = QMessageBox.question(
                self.dlg,
                self.tr('Conferma'),
                self.tr("Sei sicuro di voler cancellare i dati?"),
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No
            )

            # If the user chooses 'No', exit the function without doing anything
            if reply == QMessageBox.No:
                return

            # Clear the tableWidget_value
            if table_widget is not None:
                table_widget.clearContents()
                table_widget.setRowCount(0)

            # Reset the graph to its initial state
            self.initialize_graph()

            # Reset progress bar
            self.dlg.progressBar.setValue(0)

            # Reset lineEdit fields
            if self.dlg.lineEdit_hmin:
                self.dlg.lineEdit_hmin.clear()
            if self.dlg.lineEdit_hmax:
                self.dlg.lineEdit_hmax.clear()
            if self.dlg.lineEdit_A:
                self.dlg.lineEdit_A.clear()
            if self.dlg.lineEdit_hmed:
                self.dlg.lineEdit_hmed.clear()
            if self.dlg.lineEdit_HI:
                self.dlg.lineEdit_HI.clear()
            
            # Clear your memory
            if hasattr(self, 'clear_memory'):
                self.clear_memory()

            #close
            self.dlg.close()

        except Exception as e:
            QMessageBox.critical(
                self.dlg if hasattr(self, 'dlg') and self.dlg else None,
                self.tr("Errore"),
                self.tr(f"Errore durante la chiusura: {str(e)}")
            )
