# -*- coding: utf-8 -*-
"""
/***************************************************************************
 HTMLTableExporter
                                 A QGIS plugin
 Exports a table as an HTML table with photos which can later be converted into a Word document.  Perfect for organising site visit photos.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-02-09
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Lutra Consulting
        email                : info@lutraconsulting.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QDateTime
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog
from qgis.core import QgsProject, QgsSettings

from PIL import Image

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .html_table_exporter_dialog import HTMLTableExporterDialog
import os.path


class HTMLTableExporter:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'HTMLTableExporter_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&HTML Table Exporter')

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('HTMLTableExporter', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/html_table_exporter/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Export table as HTML'),
            callback=self.run,
            parent=self.iface.mainWindow())

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&HTML Table Exporter'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        dlg = HTMLTableExporterDialog(self.iface)

        # show the dialog
        dlg.show()
        # Run the dialog event loop
        result = dlg.exec_()
        # See if OK was pressed
        if result:

            html_string = '<html><head><style>img { width: 1237px; }</style></head><body><table border="1">\n'
            project_path = QgsProject.instance().absolutePath()

            photo_extensions = ['jpg', 'png', 'jpeg']

            # Find the layer they selected
            layer_name = dlg.tableComboBox.currentText()
            scale = dlg.scaleSpinBox.value()
            export_layer = None
            for layer in self.iface.mapCanvas().layers():
                if layer.name() == layer_name:
                    export_layer = layer
                    break

            html_string += '<tr>'
            for field in export_layer.fields():
                html_string += '<th>%s</th>' % field.name()
            html_string += '</tr>'

            features = layer.getFeatures()
            for feature in features:
                html_string += '<tr>'
                for att in feature.attributes():

                    if type(att) is str and \
                        att.lower().split('.')[-1] in photo_extensions and \
                        os.path.isfile(os.path.join(project_path, att)):
                        # Looks like a valid photo

                        photo_path = os.path.join(project_path, att)

                        im = Image.open(photo_path)
                        width, height = im.size
                        width *= (scale / 100.0)
                        height *= (scale / 100.0)

                        html_string += '<td><img width="%d" height="%d" src="%s"/></td>' % (width,height,photo_path.replace('/', '\\'))
                    elif type(att) is float:
                        html_string += '<td>%.2f</td>' % att
                    elif type(att) is QDateTime:
                        html_string += '<td>%s</td>' % att.toString()
                    else:
                        html_string += '<td>%s</td>' % att
                html_string += '</tr>\n'

            html_string += '\n</table></body></html>'

            s = QgsSettings()
            out_file_folder = s.value('html_table_exporter/out_location', None)

            out_file_name = QFileDialog.getSaveFileName(dlg, 'Select output file',
                                                        out_file_folder,
                                                        'HTML Files (*.html)')[0]

            out_file_folder = os.path.dirname(out_file_name)
            if os.path.isdir(out_file_folder):
                s = QgsSettings()
                s.setValue('html_table_exporter/out_location', out_file_folder)
            else:
                return
            f = open(out_file_name, 'w')
            f.write(html_string)
            f.close()