/**
 * Road Traffic noise prediction using the traffic information
 *
 * This Groovy script calculates road traffic noise levels at receivers.
 * It integrates with the NoiseModelling framework to process geometry files
 * and generate noise levels based on traffic data.
 *
 * Dependencies:
 * - H2GIS database for spatial data processing
 * - NoiseModelling framework for acoustic calculations
 * - SLF4J and Log4j for logging
 * 
 * @author Junta Tagusari
 * @version 1.0
 */
 
import org.h2gis.api.ProgressVisitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.sql.Connection;
import java.nio.file.Path;
import java.nio.file.Paths;
import org.apache.log4j.PropertyConfigurator;
import java.io.FileWriter;
import java.io.PrintWriter;

title = 'Calculation of road traffic noise'
description = 'Sound levels at receivers are calculated using information on road (including geometry and traffic conditions), buildings, etc.'

inputs = [
  roadGeomPath:[
    name : "Path of the road file",
    title : "Path of the road file",
    description : "Path of the road file",
    type : String.class
  ],
  roadTrafficDbPath:[
    name : "Path of the road file",
    title : "Path of the road file",
    description : "Path of the road file",
    min        : 0, max: 1,
    type : String.class
  ],
  sourceDirectivityDbPath:[
    name : "Path of the source directivity file",
    title : "Path of the source directivity file",
    description : "Path of the source directivity file",
    min        : 0, max: 1,
    type : String.class
  ],
  periodAtmosphericSettingsDbPath:[
    name : "Path of the atmospheric settings file",
    title : "Path of the atmospheric settings file",
    description : "Path of the atmospheric settings file",
    min        : 0, max: 1,
    type : String.class
  ],
  buildingGeomPath:[
    name : "Path of the building file",
    title : "Path of the building file",
    description : "Path of the building file",
    type : String.class
  ],
  receiverGeomPath:[
    name : "Path of the receiver file",
    title : "Path of the receiver file",
    description : "Path of the receiver file",
    type : String.class
  ],
  demGeomPath:[
    name : "Path of the dem file",
    title : "Path of the dem file",
    description : "Path of the dem file",
    min        : 0, max: 1,
    type : String.class
  ],
  groundAbsGeomPath:[
    name : "Path of the ground absorption file",
    title : "Path of the ground absorption file",
    description : "Path of the ground absorption file",
    min        : 0, max: 1,
    type : String.class
  ],
  inputSRID: [
    name: 'Projection identifier',
    title: 'Projection identifier',
    description: 'Original projection identifier (also called SRID) of your table. It should be an EPSG code, a integer with 4 or 5 digits (ex: 3857 is Web Mercator projection). ' +
            '</br>  All coordinates will be projected from the specified EPSG to WGS84 coordinates. ' +
            '</br> This entry is optional because many formats already include the projection and you can also import files without geometry attributes.</br> ' +
            '</br> <b> Default value : 4326 </b> ',
    type: Integer.class,
    min: 0, max: 1
  ],
  paramWallAlpha          : [
    name       : 'wallAlpha',
    title      : 'Wall absorption coefficient',
    description: 'Wall absorption coefficient (FLOAT between 0 : fully absorbent and strictly less than 1 : fully reflective)' +
            '</br> </br> <b> Default value : 0.1 </b> ',
    min        : 0, max: 1,
    type       : String.class
  ],
  confReflOrder           : [
    name       : 'Order of reflexion',
    title      : 'Order of reflexion',
    description: 'Maximum number of reflections to be taken into account (INTEGER).' +
            '</br> </br> <b> Default value : 1 </b>',
    min        : 0, max: 1,
    type       : String.class
  ],
  confMaxSrcDist          : [
    name       : 'Maximum source-receiver distance',
    title      : 'Maximum source-receiver distance',
    description: 'Maximum distance between source and receiver (FLOAT, in meters).' +
            '</br> </br> <b> Default value : 150 </b>',
    min        : 0, max: 1,
    type       : String.class
  ],
  confMaxReflDist         : [
    name       : 'Maximum source-reflexion distance',
    title      : 'Maximum source-reflexion distance',
    description: 'Maximum reflection distance from the source (FLOAT, in meters).' +
            '</br> </br> <b> Default value : 50 </b>',
    min        : 0, max: 1,
    type       : String.class
  ],
  confThreadNumber        : [
    name       : 'Thread number',
    title      : 'Thread number',
    description: 'Number of thread to use on the computer (INTEGER).' +
            '</br> To set this value, look at the number of cores you have.' +
            '</br> If it is set to 0, use the maximum number of cores available.' +
            '</br> </br> <b> Default value : 0 </b>',
    min        : 0, max: 1,
    type       : String.class
  ],
  confDiffVertical        : [
    name       : 'Diffraction on vertical edges',
    title      : 'Diffraction on vertical edges',
    description: 'Compute or not the diffraction on vertical edges.Following Directive 2015/996, enable this option for rail and industrial sources only.' +
            '</br> </br> <b> Default value : false </b>',
    min        : 0, max: 1,
    type       : Integer.class
  ],
  confDiffHorizontal      : [
    name       : 'Diffraction on horizontal edges',
    title      : 'Diffraction on horizontal edges',
    description: 'Compute or not the diffraction on horizontal edges.' +
            '</br> </br> <b> Default value : false </b>',
    min        : 0, max: 1,
    type       : Integer.class
  ],
  confExportSourceId      : [
    name       : 'keep source id',
    title      : 'Separate receiver level by source identifier',
    description: 'Keep source identifier in output in order to get noise contribution of each noise source.' +
            '</br> </br> <b> Default value : false </b>',
    min        : 0, max: 1, type: Integer.class
  ],
  confHumidity            : [
    name       : 'Relative humidity',
    title      : 'Relative humidity',
    description: 'Humidity for noise propagation, default value is <b>70</b>',
    min        : 0, max: 1, type: Double.class
  ],
  confTemperature         : [
    name       : 'Temperature',
    title      : 'Air temperature',
    description: 'Air temperature in degree celsius, default value is <b>15</b>',
    min        : 0, max: 1, type: Double.class
  ],
  confFavourableOccurrencesDefault: [
    name       : 'Probability of occurrences',
    title      : 'Probability of occurrences',
    description: 'comma-delimited string containing the probability of occurrences of favourable propagation conditions.' +
            'The north slice is the last array index not the first one<br/>' +
            'Slice width are 22.5&#176;: (16 slices)<br/><ul>' +
            '<li>The first column 22.5&#176; contain occurrences between 11.25 to 33.75 &#176;</li>' +
            '<li>The last column 360&#176; contains occurrences between 348.75&#176; to 360&#176; and 0 to 11.25&#176;</li></ul>Default value <b>0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5</b>',
    min        : 0, max: 1,
    type       : String.class
  ],  
  confFavorableOccurrencesDay: [
    name       : 'Probability of occurrences (Day)',
    title      : 'Probability of occurrences (Day)',
    description: 'comma-delimited string containing the probability of occurrences of favourable propagation conditions.' +
            'The north slice is the last array index not the first one<br/>' +
            'Slice width are 22.5&#176;: (16 slices)<br/><ul>' +
            '<li>The first column 22.5&#176; contain occurrences between 11.25 to 33.75 &#176;</li>' +
            '<li>The last column 360&#176; contains occurrences between 348.75&#176; to 360&#176; and 0 to 11.25&#176;</li></ul>Default value <b>0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5</b>',
    min        : 0, max: 1,
    type       : String.class
  ],
  confFavorableOccurrencesEvening: [
    name       : 'Probability of occurrences (Evening)',
    title      : 'Probability of occurrences (Evening)',
    description: 'comma-delimited string containing the probability of occurrences of favourable propagation conditions.' +
            'The north slice is the last array index not the first one<br/>' +
            'Slice width are 22.5&#176;: (16 slices)<br/><ul>' +
            '<li>The first column 22.5&#176; contain occurrences between 11.25 to 33.75 &#176;</li>' +
            '<li>The last column 360&#176; contains occurrences between 348.75&#176; to 360&#176; and 0 to 11.25&#176;</li></ul>Default value <b>0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5</b>',
    min        : 0, max: 1,
    type       : String.class
  ],
  confFavorableOccurrencesNight: [
    name       : 'Probability of occurrences (Night)',
    title      : 'Probability of occurrences (Night)',
    description: 'comma-delimited string containing the probability of occurrences of favourable propagation conditions.' +
            'The north slice is the last array index not the first one<br/>' +
            'Slice width are 22.5&#176;: (16 slices)<br/><ul>' +
            '<li>The first column 22.5&#176; contain occurrences between 11.25 to 33.75 &#176;</li>' +
            '<li>The last column 360&#176; contains occurrences between 348.75&#176; to 360&#176; and 0 to 11.25&#176;</li></ul>Default value <b>0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5</b>',
    min        : 0, max: 1,
    type       : String.class
  ],
  confRaysName     : [
    name       : 'Export propagation ray',
    title      : 'Export propagation ray',
    description: 'Export or not the propagation rays.',
    min        : 0, max: 1, type: String.class
  ],
  confMaxError     : [
    name       : 'Max Error (dB)',
    title      : 'Max Error (dB)',
    description: 'Threshold for excluding negligible sound sources in calculations. Default value: 0.1',
    min        : 0, max: 1, type: Double.class
  ],
  frequencyFieldPrepend            : [
    name       : 'Frequency field name',
    title      : 'Frequency field name',
    description: 'Frequency field name prepend. Ex. for 1000 Hz frequency the default column name is HZ1000. Default value: HZ',
    min        : 0, max: 1, type: String.class
  ],
  noiseModellingHome : [
    name: "Path of NOISEMODELLING_HOME",
    title: "Path of NOISEMODELLING_HOME",
    description: "Path of NOISEMODELLING_HOME",
    type : String.class
  ],
  exportDir : [
    name: "Path of export directory",
    title: "Path of export directory",
    description: "Path of export directory",
    min        : 0, max: 1,
    type : String.class
  ]
]

outputs = [
  result: [
    name: 'Result output string', 
    title: 'Result output string', 
    description: 'This type of result does not allow the blocks to be linked together.', 
    type: String.class
  ]
]

/**
 * Main execution function for road traffic noise calculation
 * This function orchestrates the road traffic noise modeling workflow:
 * 1. Clear existing database tables
 * 2. Import road geometry and traffic data
 * 3. Calculate noise levels at receivers from road traffic
 * 4. Export results to files
 * 
 * @param connection H2GIS database connection
 * @param input Map containing input parameters (roadGeomPath, receiverGeomPath, etc.)
 * @return List of result objects containing table names and export paths
 */
def exec(Connection connection, input) {
  Logger logger = LoggerFactory.getLogger("org.hrisk")
  
  try {
    logger.info("Starting noise calculation process")

    // Initialize the NoiseModelling utility class
    def noiseModelling = new NoiseModelling(connection)
    
    // Step 1: Clear any existing data in the database
    noiseModelling.clearDatabase()

    // set input tables
    String tableRoads = noiseModelling.importTable(input["roadGeomPath"], input["inputSRID"])
    String tableBuilding = noiseModelling.importTable(input["buildingGeomPath"], input["inputSRID"])
    String tableReceivers = noiseModelling.importTable(input["receiverGeomPath"], input["inputSRID"])
    String tableDEM = noiseModelling.importTable(input["demGeomPath"], input["inputSRID"])
    String tableGroundAbs = noiseModelling.importTable(input["groundAbsGeomPath"], input["inputSRID"])
    String tableRoadsTraffic = noiseModelling.importTable(input["roadTrafficDbPath"], null)
    String tableSourceDirectivity = noiseModelling.importTable(input["sourceDirectivityDbPath"], null)
    String tableAtmosphericSettings = noiseModelling.importTable(input["periodAtmosphericSettingsDbPath"], null)

    Boolean confDiffVertical = noiseModelling.parseBoolean(input["confDiffVertical"])
    Boolean confDiffHorizontal = noiseModelling.parseBoolean(input["confDiffHorizontal"])
    Boolean confExportSourceId = noiseModelling.parseBoolean(input["confExportSourceId"])

    String confRaysName = "RAYS"
    if (input["confRaysName"] != null) {
      confRaysName = input["confRaysName"]
    }

    // Get list of tables before running the noise calculation
    def tables_before_alg = noiseModelling.tableNames()
    logger.info("Current tables: " + tables_before_alg)

    // run calculation
    Map args = [
        "tableBuilding": tableBuilding, 
        "tableRoads": tableRoads, 
        "tableRoadsTraffic": tableRoadsTraffic,
        "tableSourceDirectivity": tableSourceDirectivity,
        "tableAtmosphericSettings": tableAtmosphericSettings,
        "tableReceivers": tableReceivers,
        "tableDEM": tableDEM, 
        "tableGroundAbs": tableGroundAbs,
        "paramWallAlpha": input["paramWallAlpha"],
        "confReflOrder": input["confReflOrder"],
        "confMaxSrcDist": input["confMaxSrcDist"],
        "confMaxReflDist": input["confMaxReflDist"],
        "confThreadNumber": input["confThreadNumber"],
        "confDiffVertical": confDiffVertical,
        "confDiffHorizontal": confDiffHorizontal,
        "confExportSourceId": confExportSourceId,
        "confHumidity": input["confHumidity"],
        "confTemperature": input["confTemperature"],
        "frequencyFieldPrepend": input["frequencyFieldPrepend"] != null ? input["frequencyFieldPrepend"] : "HZ",
        "confFavourableOccurrencesDefault": input["confFavourableOccurrencesDefault"],
        "confFavorableOccurrencesDay": input["confFavorableOccurrencesDay"],
        "confFavorableOccurrencesEvening": input["confFavorableOccurrencesEvening"],
        "confFavorableOccurrencesNight": input["confFavorableOccurrencesNight"],
        "confRaysName": confRaysName,
        "confMaxError": input["confMaxError"] != null ? input["confMaxError"] : 0.1
      ].findAll{ it.value!=null }

    noiseModelling.runScript(
      "NoiseModelling/Noise_level_from_traffic",
      args
    )
    
    // Get list of tables after running the noise calculation
    def tables_after_alg = noiseModelling.tableNames()
    logger.info("Current tables: " + tables_after_alg)

    // Identify the new tables created by the calculation
    def result_tables = tables_after_alg - tables_before_alg
    logger.info("Result tables created by noise calculation: " + result_tables)

    // Step 4: Export all result tables to GeoJSON files
    results = []
    for (tbl in result_tables){
      geom_export = noiseModelling.exportTables(tbl, input["exportDir"])
      results.add(
        [
          "tableName": tbl,
          "exportPath": geom_export
        ]
      )
    }

    logger.info("Noise calculation process completed successfully")
    
    return results
  
  } finally {
  }
}

