from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication, QVariant)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterNumber,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterExtent,
  QgsProcessingParameterFolderDestination,
  QgsVectorLayer,
  QgsProcessingParameterString
  )


from ..algutil.hriskutil import HrUtil
from ..algutil.hrisknoisemodelling import NoiseModelling
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors, HriskHome
import os

class receiverfacade(QgsProcessingAlgorithm):
  PARAMETERS = { 
    "BUILDING": {
      "crs_reference": True, # this parameter is used as CRS reference
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("receiverfacade","Building layer"),
        "types": [QgsProcessing.TypeVectorPolygon]
      },
      "nm_key": "buildingGeomPath"
    },    
    "SOURCE": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("receiverfacade","Source layer"),
        "types": [QgsProcessing.TypeVectorPoint,QgsProcessing.TypeVectorLine,QgsProcessing.TypeVectorPolygon],
        "optional": True,
      },
      "nm_key": "sourceGeomPath"
    },    
    "FENCE_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("receiverfacade","Calculation extent"),
        "defaultValue": None,
        "optional": True,
      }
    },
    "DELTA": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverfacade","Distance between receivers (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 1.0, "defaultValue": 3.0, "maxValue": 100.0
      },
      "nm_key": "delta"
    },
    "HEIGHT": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverfacade","Height of receivers (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.01, "defaultValue": 4.0, "maxValue": 100.0
      },
      "nm_key": "height"
    },
    "DISTANCE": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverfacade","Distance from the wall (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.01, "defaultValue": 0.1, "maxValue": 5.0
      },
      "nm_key": "distance"
    },
    "RUN_CALCULATION": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverfacade","Run the calculation (or just prepare the files)?"),
        "defaultValue": True
      }
    },
    "OVERWRITE": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverfacade","Overwrite results?"),
        "defaultValue": False
      }
    },
    "WORKING_DIRECTORY": {
      "advanced": True,
      "ui_func": QgsProcessingParameterFolderDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverfacade","Working directory"),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT
      }
    },
    "SCRIPT_RUNNER": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverfacade","Script runner name"),
        "defaultValue": "set_receiver_facade",
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("receiverfacade","Receivers at facade" ),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT
      }     
    }
  }
  
  NM_OUTPUT = {
    "RECEIVER": ["RECEIVERS"]
  }
  
  GROOVY_SCRIPT = os.path.join(HriskHome, "groovy", "receiverfacade.groovy")
  
  def __init__(self) -> None:
    super().__init__()
    self.UTIL = HrUtil(self)

  def initAlgorithm(self, config):
    self.UTIL.initParameters()

  def processAlgorithm(self, parameters, context, feedback):  
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess(with_nm=True)
    target_crs = self.UTIL.parseCrs()
    nm_version = self.UTIL.getNoiseModellingVersion()

    self.NOISEMODELLING = NoiseModelling(
      crs = target_crs,
      work_dir= self.parameterAsString(parameters, "WORKING_DIRECTORY", context),
      feedback = feedback,
      overwrite= self.parameterAsBool(parameters, "OVERWRITE", context)
    )
    
    self.NOISEMODELLING.initArgs(groovy_script = self.GROOVY_SCRIPT)
    
    if parameters.get("FENCE_EXTENT") is not None:
      rect = self.UTIL.asQgsReferencedRectangle(parameters["FENCE_EXTENT"], target_crs)
      self.NOISEMODELLING.setFeatureRequest(rect = rect)
      fence_layer = self.NOISEMODELLING.extentAsLayer(context, rect, target_crs)
      self.NOISEMODELLING.vectorLayerToArg(fence_layer, "FENCE_EXTENT", {}, "fenceGeomPath")
    
    for ui_key, ui_settings in self.PARAMETERS.items():
      if parameters.get(ui_key) is not None:
        value = parameters[ui_key]
      else:
        value = ui_settings["ui_args"].get("defaultValue", None)
      if value is not None:
        self.NOISEMODELLING.uiParametersToArg(ui_key, ui_settings, value)
    
    if nm_version[0] == "4":
      nm_runner = "org.noisemodelling.runner.Main"
    elif nm_version[0] == "5":
      nm_runner = "org.noise_planet.noisemodelling.runner.Main"
      
    self.NOISEMODELLING.setJavaCommand(
      nm_runner = nm_runner,
      script_name = self.parameterAsString(parameters, "SCRIPT_RUNNER", context),
    )
    
    if not self.parameterAsBool(parameters, "RUN_CALCULATION", context):
      return {"OUTPUT": None}      
  
    results = self.NOISEMODELLING.run()
    feedback.setProgress(100)

    receiver_result = QgsVectorLayer(results.get(self.NM_OUTPUT["RECEIVER"][0]))
    
    fields_with_values = {
      "HISTORY": {
        "type": QVariant.String,
        "value": self.CURRENT_PROCESS,
        "append": False
      }
    }
    
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= receiver_result,
      param_sink = "OUTPUT",
      fields_with_values = fields_with_values
    )
    
    PostProcessors[dest_id] = HrPostProcessor(history = [self.CURRENT_PROCESS])
    
    return {"OUTPUT": dest_id}


  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Building facade")

  def group(self):
    return self.tr('Set receivers')

  def groupId(self):
    return 'receiver'

  def createInstance(self):
    return receiverfacade()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
