from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication)
from qgis.core import (
  QgsCoordinateReferenceSystem,
  QgsProcessingParameterExtent,
  QgsProcessingParameterDistance,
  QgsProcessingParameterCrs, 
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterRasterDestination,
  QgsProperty,
  QgsProcessingParameterString,
  QgsProcessingParameterNumber
  )
from qgis import processing

from .fetchabstract import fetchabstract
from ..algutil.hriskvar import PostProcessors
from ..hriskapi import HrPostProcessor

class fetchdemvectorja(fetchabstract):
  
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchdemvectorja","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemvectorja","Target CRS (Cartesian coordinates)")
      }
    },
    "BUFFER": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemvectorja","Buffer of the fetch area (using Target CRS)"),
        "defaultValue": 0.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "TILEMAP_URL": {
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "optional": True,
        "description": QT_TRANSLATE_NOOP("fetchdemvectorja","Base-URL of the vector-tile map"),
        "defaultValue": "https://cyberjapandata.gsi.go.jp/xyz/experimental_dem10b/{z}/{x}/{y}.geojson"
      }
    },
    "TILEMAP_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "optional": True,
        "description": QT_TRANSLATE_NOOP("fetchdemvectorja","CRS of the vector-tile map"),        
        "defaultValue": "EPSG:6668" # must be specified as string, because optional parameter cannot be set as QgsCoordinateReferenceSystem
      }
    },
    "TILEMAP_ZOOM": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "optional": True,
        "description": QT_TRANSLATE_NOOP("fetchdemvectorja","Zoom level of the vector-tile map"),
        "type": QgsProcessingParameterNumber.Integer,
        "defaultValue": 18
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemvectorja","Elevation points")
      }
    },
    "OUTPUT_RASTER": {
      "ui_func": QgsProcessingParameterRasterDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchdemvectorja","Elevation raster" )
      }
    }
  }  
  
  def initAlgorithm(self, config):    
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("FETCH_EXTENT", extent)
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()

  def processAlgorithm(self, parameters, context, feedback):
    
    self.setFetchArea(parameters,context,feedback,QgsCoordinateReferenceSystem("EPSG:6668"))
    self.setTileMapArgs(parameters, context, feedback, "Point")
    
    self.fetchFeaturesFromTile(parameters, context, feedback)
    dem_raw = self.FETCH_FEATURE
    
    
    # CRS transform    
    dem_transformed = self.transformToTargetCrs(parameters,context,feedback,dem_raw)
    
    # set z value
    dem_z = processing.run(
      "native:setzvalue",
      {
        "INPUT": dem_transformed,
        "Z_VALUE": QgsProperty.fromExpression('"alti"'),
        "OUTPUT": "memory:dem"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    # substitute self constant with the fetched vector layer          
    dem_final = processing.run(
      "hrisk:initelevationpoint",{
        "INPUT": dem_z,
        "TARGET_CRS": self.parameterAsCrs(parameters, "TARGET_CRS", context),
        "OVERWRITE": True,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    dem_final_fts = context.temporaryLayerStore().mapLayers()[dem_final]
    (sink, dest_id) = self.parameterAsSink(
      parameters, "OUTPUT", context,
      dem_final_fts.fields(), dem_final_fts.wkbType(), dem_final_fts.sourceCrs()
    )
    sink.addFeatures(dem_final_fts.getFeatures())
    
    # register history and postprocessor for the vector output
    PostProcessors[dest_id] = PostProcessors[dem_final].clone()
    
    dem_raster = processing.run(
      "gdal:rasterize",
      {
        "INPUT": dem_raw,
        "FIELD": "alti",
        "UNITS": 1,
        "WIDTH": 0.4 / 3600.0,
        "HEIGHT": 0.4 / 3600.0,
        "DATA_TYPE": 6,
        "INIT": 0,
        "OUTPUT": self.parameterAsOutputLayer(parameters, "OUTPUT_RASTER", context)
      },
      context = context,
      is_child_algorithm = True      
    )["OUTPUT"]
    
    # register history and postprocessor for the raster output
    self.setCurrentProcess(use_caller=False)
    PostProcessors[dem_raster] = HrPostProcessor(
      history=[self.CURRENT_PROCESS],  
      color_args = {"coloring": "single_band_pseudo_color", "theme": "Greens", "opacity": 0.8},
      set_min_to_zero = True
    )
        
    return {"OUTPUT": dest_id, "OUTPUT_RASTER": dem_raster}   
    
  
  def displayName(self):
    return self.tr("Elevation points (Ja vector)")

  def group(self):
    return self.tr('Fetch geometries (Ja)')

  def groupId(self):
    return 'fetchgeomja'

  def createInstance(self):
    return fetchdemvectorja()
