from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication)
from qgis.core import (
  QgsRectangle,
  QgsProcessingAlgorithm,
  QgsReferencedRectangle,
  QgsCoordinateReferenceSystem,
  QgsProcessingParameterExtent,
  QgsProcessingParameterDistance,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterCrs, 
  QgsVectorLayer,
  QgsProcessingParameterNumber
  )
from qgis import processing

from ..algutil.hriskutil import HrUtil
from ..algutil.hrisktile import WebMercatorTile

class fetchbuildingja(QgsProcessingAlgorithm):
  
  # UIs
  PARAMETERS = {  
    "FETCH_EXTENT": {
      "ui_func": QgsProcessingParameterExtent,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("fetchbuildingja","Extent for fetching data")
      }
    },
    "TARGET_CRS": {
      "ui_func": QgsProcessingParameterCrs,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchbuildingja","Target CRS (Cartesian coordinates)")
      }
    },
    "BUFFER": {
      "ui_func": QgsProcessingParameterDistance,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchbuildingja","Buffer of the fetch area (using Target CRS)"),
        "defaultValue": 0.0,
        "parentParameterName": "TARGET_CRS"
      }
    },
    "MAX_DOWNLOAD": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchbuildingja","Maximum number of download"),
        "type": QgsProcessingParameterNumber.Integer,
        "defaultValue": 100
      }
    },
    "HEIGHT": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args":{
        "optional": True,
        "description" : QT_TRANSLATE_NOOP("fetchbuildingja","Default height of buildings"),
        "type": QgsProcessingParameterNumber.Double,
        "defaultValue": 8.0
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("fetchbuildingja","Building")
      }
    }
  }  
  
  FETCH_BASE_URL = "https://cyberjapandata.gsi.go.jp/xyz/experimental_bvmap/{z}/{x}/{y}.pbf"
  FETCH_RESULT_FILTER = "|layername=building|geometrytype=Polygon"
  FETCH_CRS = QgsCoordinateReferenceSystem("EPSG:3857")
  FETCH_ZOOM = 16
  FETCH_GEOM_TYPE = "Polygon"
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
    
  # initialization of the algorithm
  def initAlgorithm(self, config):    
    (extent, target_crs) = self.UTIL.getExtentAndCrsUsingCanvas()
    self.UTIL.setDefaultValue("FETCH_EXTENT", extent)
    self.UTIL.setDefaultValue("TARGET_CRS", target_crs.authid())
    self.UTIL.initParameters()
  
  # execution of the algorithm
  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback)
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess()
    # get target x-y CRS, to apply the buffer and determine the fetch area
    target_crs = self.parameterAsCrs(parameters, "TARGET_CRS", context)
    
    # check whether the target CRS is x-y coordinates
    self.UTIL.checkCrsAsCartesian(target_crs)
    
    # get the extent, using the target CRS
    fetch_extent = self.parameterAsExtent(
      parameters, "FETCH_EXTENT", context, 
      self.parameterAsCrs(parameters, "TARGET_CRS", context)
      )
    
    # get the buffer
    buffer = self.parameterAsDouble(parameters, "BUFFER",context)
    
    # get the fetch area, using the extent and buffer
    fetch_area = QgsReferencedRectangle(
      QgsRectangle(
        fetch_extent.xMinimum() - buffer,
        fetch_extent.yMinimum() - buffer,
        fetch_extent.xMaximum() + buffer,
        fetch_extent.yMaximum() + buffer
      ),
      target_crs
    )
    
    
    tile = WebMercatorTile(zoom = self.FETCH_ZOOM)
    tiles_list = tile.cellXyIdx(fetch_area)
    
    fetch_args = {}
    for i, (tx, ty) in enumerate(tiles_list):
      url_parsed = self.FETCH_BASE_URL.format(
        z = self.FETCH_ZOOM, x = tx, y = ty
      )
      
      fetch_args[f"{i+1}/{len(tiles_list)}"] = {
        "url": url_parsed,
        "zoom": self.FETCH_ZOOM,
        "tx": tx,
        "ty": ty,
        "geom_type": self.FETCH_GEOM_TYPE
      }
    
    if len(fetch_args) > self.parameterAsInt(parameters, "MAX_DOWNLOAD", context):
      feedback.reportError(self.tr("Too many downloads are required: ") + str(len(fetch_args)))
      raise Exception(self.tr("Too many downloads are required: ") + str(len(fetch_args)))

    fetch_results = self.UTIL.downloadFilesConcurrently(args = fetch_args)
    fetch_results_filtered = {}
    for key, path in fetch_results.items():
      try:
        fetch_results_filtered[key] = QgsVectorLayer(
          path + self.FETCH_RESULT_FILTER
          )
      except:
        feedback.pushInfo(
          self.tr("The layer is not loaded: ") + 
          fetch_args[key]["url"] + self.FETCH_RESULT_FILTER
        )
        
    feedback.pushInfo(self.tr("Coordinate transformation (local -> world)"))
    modified_result = {}
    for key, vec_layer in fetch_results_filtered.items():
      # constants
      tile_length = tile.unitLength()[0]
      pixel_length = tile_length / 4096 # 4096 is the number of pixels in a tile
      x_orig, y_orig = tile.origin()
      
      if vec_layer.featureCount() > 0:
        # affine transformation to obtain x and y for a given CRS
        transformed_result = processing.run(
          "native:affinetransform",
          {
            "INPUT": vec_layer,
            "DELTA_X":    fetch_args[key]["tx"]    * tile_length + x_orig,
            "DELTA_Y": - (fetch_args[key]["ty"]+1) * tile_length + y_orig,
            "SCALE_X": pixel_length,
            "SCALE_Y": pixel_length,
            "OUTPUT": "TEMPORARY_OUTPUT"
          },
          context = context,
          is_child_algorithm = True
        )["OUTPUT"]

        modified_result[key] = processing.run(
          "native:assignprojection",
          {
            "INPUT": transformed_result,
            "CRS": self.FETCH_CRS,
            "OUTPUT": "TEMPORARY_OUTPUT"
          },
          context = context,
          is_child_algorithm = True
        )["OUTPUT"]
    
    bldg_raw = processing.run(
      "native:mergevectorlayers",
      {
        "LAYERS": list(modified_result.values()),
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
      # transform
    bldg_transformed = processing.run(
    "native:reprojectlayer", 
    {
      "INPUT": bldg_raw,
      "TARGET_CRS": target_crs,
      "OUTPUT": "TEMPORARY_OUTPUT"
    },
    context = context,
    is_child_algorithm = True
    )["OUTPUT"]      
    
    
    feedback.pushInfo(self.tr("Snapping and dissolving are applied."))

    # snap geometry
    bldg_snap = processing.run(
      "native:snapgeometries", 
      {
        "INPUT": bldg_transformed,
        "REFERENCE_LAYER": context.getMapLayer(bldg_transformed),
        "TOLERANCE": 0.1,
        "BEHAVIOR": 0,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
      
    # dissolve
    bldg_dissolve = self.UTIL.dissolve(bldg_snap)
      
    # extract
    bldg_extract = processing.run(
      "native:extractbyextent",
      {
        "INPUT": bldg_dissolve,
        "EXTENT": fetch_area,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
            
    bldg_final = processing.run(
      "hrisk:initbuilding",{
        "INPUT": bldg_extract,
        "TARGET_CRS": target_crs,
        "HEIGHT": self.parameterAsDouble(parameters, "HEIGHT", context),
        "OVERWRITE": True,
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
      
      
    bldg_final_fts = context.getMapLayer(bldg_final)
    new_fields = self.UTIL.newFieldsWithHistory(bldg_final_fts.fields())
    (sink, dest_id) = self.parameterAsSink(
      parameters, "OUTPUT", context,
      new_fields, bldg_final_fts.wkbType(), target_crs
    )
    self.UTIL.addFeaturesWithHistoryToSink(
      sink, bldg_final_fts, new_fields,
      current_process=self.CURRENT_PROCESS
    )
    
    return {"OUTPUT": dest_id}
    
  def name(self):
    return self.__class__.__name__

  def displayName(self):
    return self.tr("Buildings (Ja)")
  
  def group(self):
    return self.tr('Fetch geometries (Ja)')

  def groupId(self):
    return 'fetchgeomja'

  def createInstance(self):
    return fetchbuildingja()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
