
from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QCoreApplication, QVariant)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterString,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterNumber,
  QgsProcessingParameterFeatureSink,
  QgsFeatureRequest,
  QgsReferencedRectangle,
  QgsVectorLayer,
  QgsFields,
  QgsField,
  QgsWkbTypes,
  QgsFeature,
  QgsGeometry,
  QgsProcessingParameterRasterLayer,
  QgsProcessingParameterFolderDestination
  )

import datetime
import os
from qgis import processing

from ..algutil.hriskvar import PostProcessors, HriskHome
from ..algutil.hriskutil import HrUtil
from ..algutil.hrisknoisemodelling import NoiseModelling
from ..algutil.hriskpostprocessor import HrPostProcessor

class noisefromtraffic(QgsProcessingAlgorithm):
  PARAMETERS = {
    "ROAD": {
      "crs_reference": True, # this parameter is used as CRS reference
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Road layer"),
        "types": [QgsProcessing.TypeVectorLine]
      },
      "nm_key":"roadGeomPath"
    },
    "BUILDING": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Building layer"),
        "types": [QgsProcessing.TypeVectorPolygon]
      },
      "nm_key": "buildingGeomPath"
    },
    "RECEIVER": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Receiver layer"),
        "types": [QgsProcessing.TypeVectorPoint]
      },
      "nm_key": "receiverGeomPath"
    },
    "DEM": {
      "ui_func": QgsProcessingParameterRasterLayer,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Elevation layer (raster)"),
        "optional": True
      },
      "nm_key": "demGeomPath"
    },
    "DEM_VECTOR": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Elevation layer (point vector)"),
        "types": [QgsProcessing.TypeVectorPoint],
        "optional": True
      },
      "nm_key": "demGeomPath"
    },
    "GROUND_ABS": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Ground absorption layer"),
        "types": [QgsProcessing.TypeVectorPolygon],
        "optional": True
      },
      "nm_key":"groundAbsGeomPath"
    },
    "ROAD_TRAFFIC": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "optional": True,
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Road traffic database (NM 5.0)")
      },
      "nm_key":"roadTrafficDbPath",
      "ignore_crs": True
    },
    "SOURCE DIRECTIVITY": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "optional": True,
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Source directivity database (NM 5.0)")
      },
      "nm_key":"sourceDirectivityDbPath",
      "ignore_crs": True
    },
    "ATMOSPHERIC SETTINGS": {
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Atmospheric settings database (NM 5.0)"),
        "optional": True
      },
      "nm_key":"periodAtmosphericSettingsDbPath",
      "ignore_crs": True   
    },
  
    "MAX_SRC_DIST": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Max distance between source and receiver (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 100.0, "defaultValue": 500.0, "maxValue": 1000.0
      },
      "nm_key": "confMaxSrcDist"
    },
    "MAX_REFL_DIST": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Max distance between source and reflection wall (m)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.0, "defaultValue": 350.0, "maxValue": 1000.0
      },
      "nm_key": "confMaxReflDist"
    },
    "REFL_ORDER": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Max number of reflections (times)"),
        "type": QgsProcessingParameterNumber.Integer,
        "minValue": 0, "defaultValue": 1, "maxValue": 5
      },
      "nm_key": "confReflOrder"
    },
    "HUMIDITY": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Relative humidity (%)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.0, "defaultValue": 70.0, "maxValue": 100.0
      },
      "nm_key": "confHumidity"
    },
    "TEMPERATURE": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Temperature (°C)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": -40.0, "defaultValue": 15.0, "maxValue": 50.0
      },
      "nm_key": "confTemperature"
    },
    "WALL_ALPHA": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Absorption coefficient at wall (0: fully reflective - 1: fully absorbent)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.0, "defaultValue": 0.1, "maxValue": 1.0
      },
      "nm_key": "paramWallAlpha"
    }, 
    "DIFF_VERTICAL": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Diffraction at vertical edge"),
        "defaultValue": False
      },
      "nm_key": "confDiffVertical"
    },
    "DIFF_HORIZONTAL": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Diffraction at horizontal edge"),
        "defaultValue": True
      },
      "nm_key": "confDiffHorizontal"
    },
    "FAV_OCURRENCE_DEFAULT": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Probability of occurrence of favorable condition (NM 5.0)"),
        "defaultValue": "0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5"
      },
      "nm_key": "confFavourableOccurrencesDefault"
    },
    "FAV_OCURRENCE_DAY": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Probability of occurrence of favorable condition during daytime (NM 4.0)"),
        "defaultValue": "0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5"
      },
      "nm_key": "confFavorableOccurrencesDay"
    },
    "FAV_OCURRENCE_EVENING": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Probability of occurrence of favorable condition during evening (NM 4.0)"),
        "defaultValue": "0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5"
      },
      "nm_key": "confFavorableOccurrencesEvening"
    },
    "FAV_OCURRENCE_NIGHT": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Probability of occurrence of favorable condition during night (NM 4.0)"),
        "defaultValue": "0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5"
      },
      "nm_key": "confFavorableOccurrencesNight"
    },
    "THREAD_NUMBER": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Number of threads to calculate (0: all)"),
        "type": QgsProcessingParameterNumber.Integer,
        "minValue": 0, "defaultValue": 0, "maxValue": 20
      },
      "nm_key": "confThreadNumber"
    },
    "MAX_ERROR": {
      "advanced": True,
      "ui_func": QgsProcessingParameterNumber,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Threshold for excluding negligible sound sources in calculations (dB). (NM 5.0)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.0, "defaultValue": 0.1, "maxValue": 3.0
      },
      "nm_key": "confMaxError"
    },
    "FREQUENCY_FIELD_PREPEND": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Frequency field name prepend. Ex. for 1000 Hz frequency the default column name is HZ1000. (NM 5.0)"),
        "defaultValue": "Hz"
      },
      "nm_key": "frequencyFieldPrepend"
    },
    "SOURCE_ID": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Export source id"),
        "defaultValue": False
      },
      "nm_key": "confExportSourceId"
    },
    "UNIQUE_FIELDS": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Field(s) to identify the geometry"),
        "defaultValue": "IDRECEIVER,IDSOURCE"
      }
    },
    "COLOR_FIELDS": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Field to be used for coloring the results"),
        "defaultValue": "LDEN_LAEQ,LAEQ"
      }
    },
    "GROUP_FIELDS": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Field(s) to group the results. Set none if not needed."),
        "defaultValue": "PERIOD"
      }
    },
    "RUN_CALCULATION": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Run the calculation (or just prepare the files)?"),
        "defaultValue": True
      }
    },
    "OVERWRITE": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Overwrite results?"),
        "defaultValue": False
      }
    },
    "WORKING_DIRECTORY": {
      "advanced": True,
      "ui_func": QgsProcessingParameterFolderDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Working directory"),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT
      }
    },
    "SCRIPT_RUNNER": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Script runner name"),
        "defaultValue": "noise_from_traffic",
      }
    },
    "CALC_ARGS": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Arguments used for the calculation" ),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT
      },
      "visibleByDefault": False
    },    
    "RECEIVER_WITH_LEVEL": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Receiver with level" ),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT
      },
      "visibleByDefault": True
    },
    "SOUND_RAY": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("noisefromtraffic","Sound ray" ),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT
      },
      "visibleByDefault": False
    }
  }

  NM_OUTPUT = {
    "nm_4.0": ["LDAY_GEOM", "LEVENING_GEOM", "LNIGHT_GEOM", "LDEN_GEOM"],
    "nm_5.0": ["RECEIVERS_LEVEL"],
    "RAYS": ["RAYS"]
  }
  
  GROOVY_SCRIPT = os.path.join(HriskHome, "groovy", "noisefromtraffic.groovy")
  
  
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
  
  def initAlgorithm(self, config):
    self.UTIL.initParameters()

  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback) 
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess(with_nm=True)
    target_crs = self.UTIL.parseCrs()
    
    nm_version = self.UTIL.getNoiseModellingVersion()
    
    group_name = "results_" + datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    
    # set noisemodelling output paths
    self.NOISEMODELLING = NoiseModelling(
      crs = target_crs,
      work_dir= self.parameterAsString(parameters, "WORKING_DIRECTORY", context),
      feedback = feedback,
      overwrite = self.parameterAsBool(parameters, "OVERWRITE", context)
    )
    
    self.NOISEMODELLING.initArgs(groovy_script = self.GROOVY_SCRIPT)
    
    assert parameters.get("DEM") is None or parameters.get("DEM_VECTOR") is None, \
      "Only one of DEM or DEM_VECTOR can be used."
    
    for ui_key, ui_settings in self.PARAMETERS.items():
      if parameters.get(ui_key) is not None:
        value = parameters[ui_key]
      else:
        value = ui_settings["ui_args"].get("defaultValue", None)
      if value is not None:
        self.NOISEMODELLING.uiParametersToArg(ui_key, ui_settings, value)

    if nm_version[0] == "4":
      nm_runner = "org.noisemodelling.runner.Main"
    elif nm_version[0] == "5":
      nm_runner = "org.noise_planet.noisemodelling.runner.Main"
      
    self.NOISEMODELLING.setJavaCommand(
      nm_runner = nm_runner,
      script_name = self.parameterAsString(parameters, "SCRIPT_RUNNER", context),
    )
    
    # set calculation parameters and output as a layer
    feedback.pushInfo(self.tr("Export the calculation settings."))
    
    CALC_ARGS = {
      "JAVA_CMD": self.NOISEMODELLING.JAVA_CMD,
      "HISTORY": self.UTIL.parseCurrentProcess(with_nm=True)
    } | self.NOISEMODELLING.ARGS
    
    receiver_layer = self.parameterAsSource(parameters, "RECEIVER", context).materialize(QgsFeatureRequest(), feedback)
    calc_extent = QgsReferencedRectangle(receiver_layer.extent(), receiver_layer.sourceCrs())
    
    args_flds = QgsFields()
    for key, value in CALC_ARGS.items():
      if type(value) in [int, bool]:
        args_flds.append(QgsField(key, QVariant.Int))
      elif type(value) in [float]:
        args_flds.append(QgsField(key, QVariant.Double))
      elif type(value) in [str]:
        args_flds.append(QgsField(key, QVariant.String))
    
    
    (calc_args_sink, calc_args_dest_id) = self.parameterAsSink(
      parameters, "CALC_ARGS", context,
      args_flds, QgsWkbTypes.Polygon, calc_extent.crs()
    )
    
    ft = QgsFeature(args_flds)
    ft.setGeometry(QgsGeometry.fromRect(calc_extent))
    for k, v in CALC_ARGS.items():
      ft[k] = v
    calc_args_sink.addFeature(ft)
    
    PostProcessors[calc_args_dest_id] = HrPostProcessor(
      history = [self.CURRENT_PROCESS], coloring = "simple_bbox",
      group = group_name,
      visibility=self.PARAMETERS["CALC_ARGS"]["visibleByDefault"]
    )
    
    if not self.parameterAsBool(parameters, "RUN_CALCULATION", context):
      return {"OUTPUT": None}      
  
    results = self.NOISEMODELLING.run()
    feedback.setProgress(100)
    
    args_for_formatting_results = {
      "UNIQUE_FIELDS": self.parameterAsString(parameters, "UNIQUE_FIELDS", context),
      "GROUP_FIELDS": self.parameterAsString(parameters, "GROUP_FIELDS", context),
      "COLOR_FIELDS": self.parameterAsString(parameters, "COLOR_FIELDS", context),
      "OUTPUT": "TEMPORARY_OUTPUT"
    }
    
    # for NM 5.0
    if nm_version[0] == "5":
      for output_key in self.NM_OUTPUT["nm_5.0"]:
        args_for_formatting_results |= {output_key: results.get(output_key)}
    elif nm_version[0] == "4":
      for output_key in self.NM_OUTPUT["nm_4.0"]:
        args_for_formatting_results |= {output_key: results.get(output_key)}
        
    args_for_formatting_results = {k: v for k, v in args_for_formatting_results.items() if v is not None}
    
    rcv_level_formatted_id = processing.run(
      "hrisk:formatnmresults",
      args_for_formatting_results,
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    rcv_level_formatted = context.getMapLayer(rcv_level_formatted_id)
    
    
    fields_with_values = {
      "HISTORY": {
        "type": QVariant.String,
        "value": self.CURRENT_PROCESS,
        "append": False
      }
    }
    
    rcv_level_dest_id = self.UTIL.outputVectorLayer(
      vector_layer = rcv_level_formatted,
      param_sink = "RECEIVER_WITH_LEVEL",
      fields_with_values = fields_with_values
    )
    
    color_keys = self.parameterAsString(parameters, "COLOR_FIELDS", context).split(",")
    color_keys_exist = [ck for ck in color_keys if ck in rcv_level_formatted.fields().names()]
    
    if len(color_keys_exist) > 0:
      feedback.pushInfo(self.tr("Coloring the results by field: ") + color_keys_exist[0])
      color_args = {
        "coloring": "graduated_symbol",
        "attribute": color_keys_exist[0],
        "theme": "Default"
      }
    else:
      feedback.pushWarning(self.tr("The fields for coloring the results are not available: ") + ", ".join(color_keys))
      color_args = {"coloring": "none"}
      
    PostProcessors[rcv_level_dest_id] = HrPostProcessor(
      history = [self.CURRENT_PROCESS],
      group = group_name,
      visibility=self.PARAMETERS["RECEIVER_WITH_LEVEL"]["visibleByDefault"],
      **color_args
    )
    
    
    try:
      # Sound-ray results as a layer
      sound_ray = QgsVectorLayer(results[self.NM_OUTPUT["RAYS"][0]])
      sound_ray_dest_id = self.UTIL.outputVectorLayer(
        vector_layer = sound_ray,
        param_sink = "SOUND_RAY",
        fields_with_values = fields_with_values
      )
      PostProcessors[sound_ray_dest_id] = HrPostProcessor(
        history = [self.CURRENT_PROCESS],
        group = group_name, coloring = "none",
        visibility=self.PARAMETERS["SOUND_RAY"]["visibleByDefault"]
      )
    except:
      feedback.pushWarning(self.tr("Sound ray results are not available."))
      sound_ray_dest_id = None
        
    
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)
    return {
      "CALC_ARGS": calc_args_dest_id,
      "RECEIVER_WITH_LEVEL": rcv_level_dest_id,
      "SOUND_RAY": sound_ray_dest_id
      }
    
  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Prediction from traffic")

  def group(self):
    return self.tr("Predict sound level")

  def groupId(self):
    return "soundlevel"

  def createInstance(self):
    return noisefromtraffic()


  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
