from qgis.PyQt.QtCore import (QT_TRANSLATE_NOOP, QVariant, QCoreApplication)
from qgis.core import (
  QgsProcessing,
  QgsProcessingAlgorithm,
  QgsProcessingParameterFeatureSource,
  QgsProcessingParameterNumber,
  QgsProcessingParameterString,
  QgsProcessingParameterFeatureSink,
  QgsProcessingParameterField,
  QgsFeatureRequest,
  QgsProcessingParameterBoolean,
  QgsProcessingParameterFolderDestination,
  )

from qgis import processing
from ..algutil.hriskutil import HrUtil
from ..algutil.hrisknoisemodelling import NoiseModelling
from ..algutil.hriskpostprocessor import HrPostProcessor
from ..algutil.hriskvar import PostProcessors, HriskHome

import os

class isosurface(QgsProcessingAlgorithm):
  PARAMETERS = { 
    "INPUT_LAYER": {
      "crs_reference": True, # this parameter is used as CRS reference
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("isosurface","Sound level layer"),
        "types": [QgsProcessing.TypeVectorPoint]
      }
    },
    "INPUT_PK_FIELD": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("isosurface","Receiver ID Field of the Sound level layer"),
        "parentLayerParameterName": "INPUT_LAYER",
        "defaultValue": "IDRECEIVER"
      }
    },
    "INPUT_RESULT_FIELD": {
      "ui_func": QgsProcessingParameterField,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("isosurface","Sound-level Field of the Sound level layer"),
        "parentLayerParameterName": "INPUT_LAYER",
        "defaultValue": "LAEQ"
      },
      "nm_key": "resultTableField"
    },
    "TRIANGLES": { # Note: the name "TRIANGLES" should not be changed
      "ui_func": QgsProcessingParameterFeatureSource,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("isosurface","Triangle layer"),
        "types": [QgsProcessing.TypeVectorPolygon]
      },
      "nm_key": "triangleGeomPath",
      "save_layer_get_path": True
    },
    "ISO_CLASS": {
      "ui_func": QgsProcessingParameterString,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("isosurface","Separation of sound levels for isosurfaces (e.g. 35.0,40.0)"),
        "defaultValue": "35.0,40.0,45.0,50.0,55.0,60.0,65.0,70.0,75.0,80.0,200.0",
        "multiLine": False
      },
      "nm_key": "isoClass"
    },    
    "SMOOTH_COEF": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args":{
        "description": QT_TRANSLATE_NOOP("isosurface","Smoothing parameter (Bezier curve coefficient)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.0, "defaultValue": 1.0, "maxValue": 2.0
      },
      "nm_key": "smoothCoefficient"
    },
    "OPACITY": {
      "ui_func": QgsProcessingParameterNumber,
      "ui_args":{
        "optional": True,
        "description": QT_TRANSLATE_NOOP("isosurface","Opacity of the fill (0: transparent - 1: opaque)"),
        "type": QgsProcessingParameterNumber.Double,
        "minValue": 0.0, "defaultValue": None, "maxValue": 1.0,
        "defaultValue": 1.0
      }
    },
    "RUN_CALCULATION": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("isosurface","Run the calculation (or just prepare the files)?"),
        "defaultValue": True
      }
    },
    "OVERWRITE": {
      "advanced": True,
      "ui_func": QgsProcessingParameterBoolean,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("isosurface","Overwrite results?"),
        "defaultValue": False
      }
    },
    "WORKING_DIRECTORY": {
      "advanced": True,
      "ui_func": QgsProcessingParameterFolderDestination,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("isosurface","Working directory"),
        "defaultValue": QgsProcessing.TEMPORARY_OUTPUT
      }
    },
    "SCRIPT_RUNNER": {
      "advanced": True,
      "ui_func": QgsProcessingParameterString,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("isosurface","Script runner name"),
        "defaultValue": "noise_from_emission",
      }
    },
    "OUTPUT": {
      "ui_func": QgsProcessingParameterFeatureSink,
      "ui_args": {
        "description": QT_TRANSLATE_NOOP("isosurface","Isosurface")
      }
    }
  }
  
  NM_OUTPUT = {"CONTOUR": ["CONTOURING_NOISE_MAP"]}
  GROOVY_SCRIPT = os.path.join(HriskHome, "groovy", "isosurface.groovy")
        
  def __init__(self):
    super().__init__()
    self.UTIL = HrUtil(self)
  
  def initAlgorithm(self, config):
    self.UTIL.initParameters()

  def processAlgorithm(self, parameters, context, feedback):
    
    self.UTIL.registerProcessingParameters(parameters, context, feedback) 
    self.CURRENT_PROCESS = self.UTIL.parseCurrentProcess(with_nm=True)
    target_crs = self.UTIL.parseCrs()
    nm_version = self.UTIL.getNoiseModellingVersion()
    
    triangles = self.parameterAsSource(parameters, "TRIANGLES", context).materialize(QgsFeatureRequest(), feedback)
    assert "PK_1" in triangles.fields().names(), "PK_1 not found in the triangle layer"    
    assert "PK_2" in triangles.fields().names(), "PK_2 not found in the triangle layer"    
    assert "PK_3" in triangles.fields().names(), "PK_3 not found in the triangle layer"    
    assert "CELL_ID" in triangles.fields().names(), "CELL_ID not found in the triangle layer"    
    
    
    self.NOISEMODELLING = NoiseModelling(
      crs = target_crs,
      work_dir= self.parameterAsString(parameters, "WORKING_DIRECTORY", context),
      feedback = feedback,
      overwrite= self.parameterAsBool(parameters, "OVERWRITE", context)
    )
    
    self.NOISEMODELLING.initArgs(groovy_script = self.GROOVY_SCRIPT)
    
    
    input_lyr = self.parameterAsSource(parameters, "INPUT_LAYER", context).materialize(QgsFeatureRequest(), feedback)
    pk_field = self.parameterAsString(parameters, "INPUT_PK_FIELD", context)
    level_field = self.parameterAsString(parameters, "INPUT_RESULT_FIELD", context)
    
    input_uri = processing.run(
      "native:retainfields",
      {
        "INPUT": input_lyr,
        "FIELDS": [pk_field, level_field],
        "OUTPUT": os.path.join(self.NOISEMODELLING.WORK_DIR, "geo","tmp.geojson")
      },
      context = context,
      is_child_algorithm = True
    )["OUTPUT"]
    
    if nm_version[0] == "4" and level_field != "LAEQ":
      input_uri = processing.run(
        "native:renametablefield",
        {
          "INPUT": input_uri,
          "FIELD": level_field,
          "NEW_NAME": "LAEQ",
          "OUTPUT": os.path.join(self.NOISEMODELLING.WORK_DIR, "geo", "tmp_2.geojson")
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]
    
    valid_pk_field = "PK" if nm_version[0] == "4" else "IDRECEIVER"
    if pk_field != valid_pk_field:
      input_uri = processing.run(
        "native:renametablefield",
        {
          "INPUT": input_uri,
          "FIELD": pk_field,
          "NEW_NAME": valid_pk_field,
          "OUTPUT": os.path.join(self.NOISEMODELLING.WORK_DIR, "geo","tmp_3.geojson")
        },
        context = context,
        is_child_algorithm = True
      )["OUTPUT"]
    
    
    for ui_key, ui_settings in self.PARAMETERS.items():
      if parameters.get(ui_key) is not None:
        value = parameters[ui_key]
      else:
        value = ui_settings["ui_args"].get("defaultValue", None)
      if value is not None:
        self.NOISEMODELLING.uiParametersToArg(ui_key, ui_settings, value)
    
    if nm_version[0] == "4":
      nm_runner = "org.noisemodelling.runner.Main"
    elif nm_version[0] == "5":
      nm_runner = "org.noise_planet.noisemodelling.runner.Main"
    
    self.NOISEMODELLING.vectorLayerToArg(input_uri, "INPUT", {}, "resultGeomPath")
    
    self.NOISEMODELLING.setJavaCommand(
      nm_runner = nm_runner,
      script_name = self.parameterAsString(parameters, "SCRIPT_RUNNER", context),
    )
    
    if not self.parameterAsBool(parameters, "RUN_CALCULATION", context):
      return {"OUTPUT": None}      
  
    results = self.NOISEMODELLING.run()
    feedback.setProgress(100)
    
    # add level field
    iso_breaks = [-999.0] + [float(x) for x in self.parameterAsString(parameters, "ISO_CLASS", context).split(",")]
    iso_mean = []
    for i in range(1, len(iso_breaks)):
      iso_mean.append((iso_breaks[i] + iso_breaks[i - 1]) / 2)
    
    isosurface_layer = processing.run(
      "native:fieldcalculator",
      {
        "INPUT": results.get(self.NM_OUTPUT["CONTOUR"][0]),
        "FIELD_NAME": "ISOLVL_MEAN",
        "FIELD_TYPE": 0,
        "FIELD_LENGTH": 10,
        "FIELD_PRECISION": 3,
        "FORMULA": "array(" + ",".join([str(x) for x in iso_mean]) + ")[\"ISOLVL\"]",
        "OUTPUT": "TEMPORARY_OUTPUT"
      },
      is_child_algorithm = True,
      context = context
    )["OUTPUT"]
    isosurface_layer = context.getMapLayer(isosurface_layer)
    
    dest_id = self.UTIL.outputVectorLayer(
      vector_layer= isosurface_layer,
      param_sink = "OUTPUT",
      fields_with_values= {
        "HISTORY":{
          "type": QVariant.String,
          "value": self.CURRENT_PROCESS,
          "append": False
        }
      }
    )
    
    PostProcessors[dest_id] = HrPostProcessor(
      history = [self.CURRENT_PROCESS],
      coloring = "graduated_symbol", 
      attribute = "ISOLVL_MEAN", 
      opacity = parameters.get("OPACITY"),
      theme = "Default"
      )
    
    self.UTIL.registerPostProcessAlgorithm(context, PostProcessors)
    
    return {"OUTPUT": dest_id}          

  def name(self):
    return self.__class__.__name__
  
  def displayName(self):
    return self.tr("Isosurface")

  def group(self):
    return self.tr('Predict sound level')

  def groupId(self):
    return 'soundlevel'

  def createInstance(self):
    return isosurface()

  # placing here is necessary, when employing pylupdate
  def tr(self, string):
    return QCoreApplication.translate(self.__class__.__name__, string)
