/**
 * Receiver points at Delaunay grid point Script
 * 
 * This Groovy script Receiver at Delaunay grid point.
 * It integrates with the NoiseModelling framework to process building and source files
 * and generate receivers at Delaunay grid points for acoustic modeling purposes.
 * 
 * Dependencies:
 * - H2GIS database for spatial data processing
 * - NoiseModelling framework for acoustic calculations
 * - SLF4J and Log4j for logging
 * 
 * @author Junta Tagusari
 * @version 1.0
 */
 
import org.h2gis.api.ProgressVisitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.sql.Connection;
import java.nio.file.Path;
import java.nio.file.Paths;
import org.apache.log4j.PropertyConfigurator;
import java.io.FileWriter;
import java.io.PrintWriter;

title = 'Receiver at Delaunay grid point'
description = 'Geometry of receivers at Delaunay grid points is obtained'

inputs = [
  buildingGeomPath:[
    name : "Path of the building file",
    title : "Path of the building file",
    description : "Path of the building file",
    type : String.class
  ],
  sourceGeomPath:[
    name : "Path of the source file",
    title : "Path of the source file",
    description : "Path of the source file",
    min        : 0, max: 1,
    type : String.class
  ],
  fence:[
    name : "Polygon that specifies the fence",
    title : "Polygon that specifies the fence",
    description : "Polygon that specifies the fence",
    min        : 0, max: 1,
    type : String.class
  ],
  inputSRID: [
    name: 'Projection identifier',
    title: 'Projection identifier',
    description: 'Original projection identifier (also called SRID) of your table. It should be an EPSG code, a integer with 4 or 5 digits (ex: 3857 is Web Mercator projection). ' +
            '</br>  All coordinates will be projected from the specified EPSG to WGS84 coordinates. ' +
            '</br> This entry is optional because many formats already include the projection and you can also import files without geometry attributes.</br> ' +
            '</br> <b> Default value : 4326 </b> ',
    type: Integer.class,
    min: 0, max: 1
  ],
  maxPropDist : [
    name: 'Maximum Propagation Distance',
    title: 'Maximum Propagation Distance',
    description: 'Set Maximum propagation distance in meters. Avoid loading to much geometries when doing Delaunay triangulation. (FLOAT)' +
            '</br> </br> <b> Default value : 500 </b>',
    min: 100, max: 2000,
    type: Double.class
  ],
  maxCellDist : [
    name: 'Maximum Cell Size',
    title: 'Maximum Cell Size',
    description: 'Maximum distance used to split the domain into sub-domains. (FLOAT)' +
            '</br> </br> <b> Default value : 600 </b>',
    min: 100, max: 2000,
    type: Double.class
  ],
  roadWidth          : [
    name: 'Source Width',
    title: 'Source Width',
    description: 'Set Road Width in meters. No receivers closer than road width distance.(FLOAT) ' +
            '</br> </br> <b> Default value : 2 </b>',
    min: 1, max: 20,
    type: Double.class
  ],
  maxArea            : [
    name: 'Maximum Area',
    title: 'Maximum Area',
    description: 'Set Maximum Area in m2. No triangles larger than provided area. Smaller area will create more receivers. (FLOAT)' +
            '</br> </br> <b> Default value : 2500 </b> ',
    min: 100, max: 10000,
    type: Double.class
  ],
  height: [
    name       : 'height',
    title      : 'height',
    description: 'Height of receivers in meters (FLOAT)' +
            '</br> </br> <b> Default value : 4 </b> ',
    min        : 0, max: 10,
    type       : Double.class
  ],
  isoSurfaceInBuildings: [
    name: 'Create IsoSurfaces over buildings',
    title: 'Create IsoSurfaces over buildings',
    description: 'If enabled isosurfaces will be visible at the location of buildings',
    min: 0, max: 1,
    type: Boolean.class,
  ],
  noiseModellingHome : [
    name: "Path of NOISEMODELLING_HOME",
    title: "Path of NOISEMODELLING_HOME",
    description: "Path of NOISEMODELLING_HOME",
    type : String.class
  ],
  exportDir : [
    name: "Path of export directory",
    title: "Path of export directory",
    description: "Path of export directory",
    min        : 0, max: 1,
    type : String.class
  ]
]

outputs = [
  result: [
    name: 'Result output string', 
    title: 'Result output string', 
    description: 'This type of result does not allow the blocks to be linked together.', 
    type: String.class
  ]
]

/**
 * Main execution function for Delaunay grid receiver generation
 * This function orchestrates the Delaunay grid receiver creation workflow:
 * 1. Clear existing database tables
 * 2. Import building, source, and fence geometry data
 * 3. Generate receiver points at Delaunay grid positions
 * 4. Export results to files
 * 
 * @param connection H2GIS database connection
 * @param input Map containing input parameters (buildingGeomPath, sourceGeomPath, etc.)
 * @return List of result objects containing table names and export paths
 */
def exec(Connection connection, input) {

  Logger logger = LoggerFactory.getLogger("org.hrisk")
  
  try {
    logger.info("Starting noise calculation process")
    
    // Initialize the NoiseModelling utility class
    def noiseModelling = new NoiseModelling(connection)
    
    // Step 1: Clear any existing data in the database
    noiseModelling.clearDatabase()

    // Step 2: Import the geometry data into the database
    String tableBuilding = noiseModelling.importTable(input["buildingGeomPath"], input["inputSRID"])
    String sourcesTableName = noiseModelling.importTable(input["sourceGeomPath"], input["inputSRID"])
    String fence = null
    if (input["fence"]) {
      fence = input["fence"]
    } 

    // Get list of tables before running the noise calculation
    def tables_before_alg = noiseModelling.tableNames()
    logger.info("Current tables: " + tables_before_alg)

    // Step 3: Execute the road emission calculation
    Map args = [
        "tableBuilding": tableBuilding, 
        "sourcesTableName": sourcesTableName, 
        "fence": fence, 
        "maxPropDist": input["maxPropDist"],
        "maxCellDist": input["maxCellDist"],
        "roadWidth": input["roadWidth"],
        "maxArea": input["maxArea"],
        "height": input["height"],
        "isoSurfaceInBuildings": input["isoSurfaceInBuildings"] == 1 ? true : null
      ].findAll{ it.value!=null }

    noiseModelling.runScript(
      "Receivers/Delaunay_Grid",
      args
    )

    // Get list of tables after running the noise calculation
    def tables_after_alg = noiseModelling.tableNames()
    logger.info("Current tables: " + tables_after_alg)

    // Identify the new tables created by the calculation
    def result_tables = tables_after_alg - tables_before_alg
    logger.info("Result tables created by noise calculation: " + result_tables)


    // Step 4: Export all result tables to GeoJSON files
    results = []
    for (tbl in result_tables){
      geom_export = noiseModelling.exportTables(tbl, input["exportDir"])
      results.add(
        [
          "tableName": tbl,
          "exportPath": geom_export
        ]
      )
    }

    logger.info("Noise calculation process completed successfully")
    
    return results
  
  } finally {
    // Cleanup completed - no additional resources to close
  }
}

